/* ====BELTRAMIFLOW.H =============================================================================
 *
 * Author: Patrice Koehl, June 2018
 * Department of Computer Science
 * University of California, Davis
 *
 * This file implements methods for computing a quasi-conformal map on the surface of a sphere
 * to reduce conformal distortions
 *
 * These methods are based on the paper:
 *	T. W. Wong and H. Zhao, "Computation of quasiconformal surface maps using discrete Beltrami
 *	flow", SIAM J. Imaging Sci., 7(4), 2675–2699 (2014).
 *
 =============================================================================================== */

#pragma once

  /* ===== INCLUDES AND TYPEDEFS =================================================================
   *
   =============================================================================================== */

  #define _USE_MATH_DEFINES // for M_PI

  #include <vector>
  #include <cmath>

  typedef Eigen::Triplet<double> Triplet;


  /* ===== Definition of the class ===============================================================
   *
   =============================================================================================== */

   class Beltrami {

  public:

	void beltramiFlow(Mesh& mesh);

  private:
	// Define a local frame for a triangle
	int trigLocalFrame(Vector& p1, Vector& p2, Vector& p3, 
	Vector& w1, Vector& w2, Vector& w3, double *x1, double *x2, double *x3);
	
	// Save coordinates in local frame for each face
  	void saveTrigFrame(Mesh& mesh, std::vector<Vector >& F_info);

	// Compute basis for the tangent plane at a given point on Sphere
  	void tangentPlane(Vector& p1, Vector& a, Vector& b); 

	// Set three fixed points
	void findFixedPoints(Mesh& mesh, int *IdxFixed, int *nfixed);

	// Beltrami coefficients for each triangle
	int beltramiCoef(Mesh& mesh, std::vector<Vector >& F_info, 
	double *mu_target, double *mu, double *nu);

	// Surface area of all triangles in the mesh
	void trigArea(Mesh& mesh, double *Area);

	// Beltrami energy over the whole mesh
	double beltramiEnergy(Mesh& mesh, double *Area, double *coef);

	// Initialize Hessian for minimizing Beltrami energy
	void initHessian(Mesh& mesh, int nfixed);

	// Re-Initialize Hessian for minimizing Beltrami energy
	void resetHessian(Mesh& mesh, int nfixed);

	// Compute the outer product between two vectors
	void outerProduct(double *A, double *B, double *AB);

	// Contribution of one triangle to the gradient and Hessian of the energy of the flow
	void trigContribHessian(int nvertex, int idx1, Vector& p1, 
	int idx2, Vector& p2, int idx3, Vector& p3, double Area, double alpha, double beta);

	// Solve flow
	void solveFlow(Mesh& mesh, int nfixed, double *nu, double *Area, double *Disp);

	// Apply displacement
	void applyDisp(Mesh& mesh, double step, double *Disp);

	// Reset Mesh to saved coordinates
	void resetMesh(Mesh& mesh, double *Xsave);

	// Save current Mesh geometry
	void saveMesh(Mesh& mesh, double *Xsave);

	// Error max on Beltrami coefficients
	double errorMax(int n_faces, double *mu_target, double *mu);

   protected:

	Eigen::SparseMatrix<double> H;
	Eigen::VectorXd B, Sol;

	bool H_is_set;

  };
  /* ===============================================================================================
   TrigLocalFrame: for a given triangle (p1,p2,p3), map three vertices in a local frame (w1, w2, w3)
               (where w3 is orthogonal to the triangle), such that in this new frame,
		p1 -> (0,0,0)
		p2 -> (x1,0,0)
		p3 -> (x2,x3,0)
   Output:
		w1, w2, w3
		x1, x2, x3
		nin: if normal of (p1, p2, p3) points inward, nin = 1, and 0 otherwise
   =============================================================================================== */

  int Beltrami::trigLocalFrame(Vector& p1, Vector& p2, Vector& p3, Vector& w1, Vector& w2, Vector& w3, 
	double *x1, double *x2, double *x3)
  {

	double a1, a2, a3;
	double tmp;
	int nrev = 0;

	Vector center;

	center = (p1+p2+p3);

	w1 = p2-p1;
	a1 = w1.norm();
	w1 = w1/a1;

	w2 = p3-p1;
	a2 = dot(w2 , w1);

	w3 = cross(w1 , w2);
	a3 = w3.norm();
	w3 = w3/a3;
	tmp = dot(w3 , center);
	if (tmp < 0) {
		nrev = 1;
	}

	w2 = cross(w3 , w1);
	a3 = dot((p3-p1) , w2);

	*x1 = a1; *x2 = a2; *x3 = a3;

	return nrev;
  }

  /* ===============================================================================================
   saveTrigFrame: for all faces in the original mesh, save local frame
   =============================================================================================== */

  void Beltrami::saveTrigFrame(Mesh& mesh, std::vector<Vector >& F_info)
  {

	HalfEdgeIter hAB, hBC, hCA;
	VertexIter vA, vB, vC;

	Vector p1, p2, p3, v1, v2, v3;

	double a1, a2, a3;
	int nrev, idx;

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{

		idx = f_iter->index;

  		/* ===============================================================================
		Coordinates of vertices of triangle in original mesh
   		================================================================================== */

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		p1 = hAB->vertex->position;
                p2 = hBC->vertex->position;
                p3 = hCA->vertex->position;

  		/* ===============================================================================
		Apply map: 
			p1 -> (0,0), p2-> (a1, 0) p3 -> (a2, a3)
   		================================================================================== */

		nrev = trigLocalFrame(p1, p2, p3, v1, v2, v3, &a1, &a2, &a3);

  		/* ===============================================================================
		Save a1, a2, a3 as a property of the face
   		================================================================================== */

		p1[0] = a1; p1[1] = a2; p1[2] = a3;
		F_info[idx] = p1;

	}
  }

  /* =============================================================================================
  Compute (current) basis for tangent plane at a given vertex
   =============================================================================================== */

  void Beltrami::tangentPlane(Vector& point, Vector& a, Vector& b)
  {

	double x, y, z;
	double tmp;

  	/* ===============================================================================
	Coordinates of vertex iv
   	================================================================================== */

	x = point[0]; 
	y = point[1]; 
	z = point[2]; 

  	/* ===============================================================================
	Define vector a
   	================================================================================== */

	if(std::abs(x) >= std::abs(y) && std::abs(x) >= std::abs(z)) 
	{
		tmp = 1.0/std::sqrt(x*x+y*y);
		a[0] = y*tmp; a[1] = -x*tmp; a[2] = 0;
	} 
	else if(std::abs(y) >= std::abs(x) && std::abs(y) >= std::abs(z))
	{
		tmp = 1.0/std::sqrt(y*y+z*z);
		a[0] = 0; a[1] = z*tmp; a[2] = -y*tmp;
	} 
	else
	{
		tmp = 1.0/std::sqrt(x*x+z*z);
		a[0] = -z*tmp; a[1] = 0; a[2] = x*tmp;
	}

  	/* ===============================================================================
	Define vector b = point % a (cross product)
   	================================================================================== */

	b = cross(point , a);

  }

  /* =============================================================================================
  Find fixed point
   3 points are kept fixed during the flow: we set those three points as the three vertices that are
   the closest to the North pole (0,0,1), the point (0,1,0), and the South pole (0,0,-1)
   =============================================================================================== */

  void Beltrami::findFixedPoints(Mesh& mesh, int *IdxFixed, int *nfixed) 
  {

	Vector point;

	double x, y, z;
	int idx;

	VertexIter first = mesh.vertices.begin();
	int idxN = first->index;
	int idxY = first->index;
	int idxS = first->index;
	double cN = first->position2[2];
	double cY = first->position2[1];
	double cS = first->position2[2];

	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
                point = v_it->position2;
		idx = v_it->index;
		x = point[0]; y = point[1]; z = point[2];
		if(y > cY) {
			cY = y; idxY = idx;
		}
		if(z > cN) {
			cN = z; idxN = idx;
		}
		if(z < cS) {
			cS = z; idxS = idx;
		}
	}

	IdxFixed[0] = idxN;
	IdxFixed[1] = idxY;
	IdxFixed[2] = idxS;

	idx = 0;
	int id;
	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		id = v_it->index;
		if(id==IdxFixed[0] || id==IdxFixed[1] || id==IdxFixed[2]) {
			v_it->indexN = -1;
		} else {
			v_it->indexN = idx;
			idx++;
		}
	}
	*nfixed = 3;
	
  }

  /* =============================================================================================
  Compute (current) Beltrami coefficients of triangles
   =============================================================================================== */

  int Beltrami::beltramiCoef(Mesh& mesh, std::vector<Vector >& F_info, 
	double *mu_target, double *mu, double *nu)
  {

	VertexIter vA, vB, vC;
	HalfEdgeIter hAB, hBC, hCA;

	Vector q1, q2, q3, v1, v2, v3;

	double u_x, u_y, v_x, v_y;
	double a1, a2, a3, b1, b2, b3;
	double a, b, c, d, den, mu_r, mu_i, f_r, f_i, nu_r, nu_i;

	int idx;
	int nrev;
	int nrevtot = 0;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		idx = f_iter->index;
		q1 = F_info[idx];
		a1 = q1[0];
		a2 = q1[1];
		a3 = q1[2];

  		/* ===============================================================================
		Coordinates of vertices of triangle if in original mesh, and in current spherical mesh
   		================================================================================== */

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		q1 = hAB->vertex->position2;
                q2 = hBC->vertex->position2;
                q3 = hCA->vertex->position2;

  		/* ===============================================================================
		Apply map: 
			p1 -> (0,0), p2-> (a1, 0) p3 -> (a2, a3)
			q1 -> (0,0), q2-> (b1, 0) q3 -> (b2, b3)
   		================================================================================== */

		nrev = trigLocalFrame(q1, q2, q3, v1, v2, v3, &b1, &b2, &b3);
		nrevtot += nrev;

  		/* ===============================================================================
		Compute map:
			f(x+i y) = u(x,y) + i v(x,y):
			(u_x  v_x) (a1  a2) = (b1 b2)
                        (u_y  v_y) (0   a3) = (0  b3)
   		================================================================================== */

		u_x = a3*b1; u_y = a1*b2 - b1*a2;
		v_x = 0; v_y = a1*b3;

  		/* ===============================================================================
		Compute Beltrami coefficient:
				u_x - v_y + i (v_x + u_y)
			mu =    -------------------------
				u_x + v_y + i (v_x - u_y)
   		================================================================================== */

		a = u_x - v_y; b = v_x + u_y;
		c = u_x + v_y; d = v_x - u_y;
		den = 1.0/(c*c+d*d);
		mu_r = (a*c+b*d)*den;
		mu_i = (b*c-a*d)*den;

		mu[2*idx]   = mu_r;
		mu[2*idx+1] = mu_i;

  		/* ===============================================================================
		Compute nu:
						f_z*f_z        1
			nu =  (mu_target-mu) *  -------- * ----------
						| f_z |^2  1 - |mu|^2
		where
			f_z = u_x + v_y + i (v_x - u_y);

		we define f as the ratio f_z*f_z/(|f_z|^2
   		================================================================================== */

		f_r = (c*c-d*d)*den;
		f_i = 2*c*d*den;

		a = mu_target[2*idx] - mu_r;
		b = mu_target[2*idx+1] - mu_i;

		den = 1.0/(1 - mu_r*mu_r - mu_i*mu_i);
		nu_r = (a*f_r - b*f_i)*den;
		nu_i = (a*f_i + b*f_r)*den;

		nu[2*idx]   = nu_r;
		nu[2*idx+1] = nu_i;

	}

	return nrevtot;

  }

  /* ===============================================================================================
   trigArea: Computes surface area of all faces in current mesh
   =============================================================================================== */

  void Beltrami::trigArea(Mesh& mesh, double *Area)
  {
	Vector pointA, pointB, pointC;
	VertexIter vA, vB, vC;
	HalfEdgeIter hAB, hBC, hCA;


	int idx;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{

		idx = f_iter->index;

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		pointA = hAB->vertex->position2;
                pointB = hBC->vertex->position2;
                pointC = hCA->vertex->position2;

		Area[idx] = 0.5 * cross(pointA-pointB , pointA-pointC).norm() ;

	}


  }

  /* =============================================================================================
  Compute energy for Beltrami flow
   =============================================================================================== */

  double Beltrami::beltramiEnergy(Mesh& mesh, double *Area, double *Coef)
  {

	double energy = 0;
	double tarea;
	double mu_r, mu_i;
	int idx;

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{

		idx = f_iter->index;
		tarea = Area[idx];

  		/* ===============================================================================
		Energy for that triangle associated with its current Beltrami coefficient
   		================================================================================== */

		mu_r = Coef[2*idx];
		mu_i = Coef[2*idx+1];

		energy += tarea*(mu_r*mu_r+mu_i*mu_i);
	}

	return energy;
  }


  /* =============================================================================================
   Init Hessian matrix to solve for flow
   =============================================================================================== */

  void Beltrami::initHessian(Mesh& mesh, int nfixed)
  {

  /* ==================================================================================================
	The number of non zero elements in the matrix is:
		4*nvertex + 8*nedge
	where nvertex is the number of "mobile" vertices (tot_vertex-3), and nedge is the number
	of edges that do not link to a fixed vertex
   ==================================================================================================*/

	int nvertex = mesh.vertices.size() - nfixed;

  /* ==================================================================================================
	Initialize all values for regular constraints to 0
   ==================================================================================================*/

	int idx, jdx;
	std::vector<Triplet> Mat_coefficients;
	double zero = 0;

	for (VertexIter v_it = mesh.vertices.begin(); v_it!= mesh.vertices.end(); v_it++)
	{
		idx = v_it->indexN; 
		if( idx >=0 ) {
			Mat_coefficients.push_back(Triplet(idx, idx, zero));
			Mat_coefficients.push_back(Triplet(idx, idx+nvertex, zero));
			Mat_coefficients.push_back(Triplet(idx+nvertex, idx, zero));
			Mat_coefficients.push_back(Triplet(idx+nvertex, idx+nvertex, zero));
		}
	}

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;

	for (EdgeIter e_it = mesh.edges.begin(); e_it!= mesh.edges.end(); e_it++)
	{
		he = e_it->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		idx = v_i->indexN;
		jdx = v_j->indexN;

		if( idx >=0 && jdx >=0 ) {
			Mat_coefficients.push_back(Triplet(idx, jdx, zero));
			Mat_coefficients.push_back(Triplet(idx, jdx+nvertex, zero));
			Mat_coefficients.push_back(Triplet(idx+nvertex, jdx, zero));
			Mat_coefficients.push_back(Triplet(idx+nvertex, jdx+nvertex, zero));

			Mat_coefficients.push_back(Triplet(jdx, idx, zero));
			Mat_coefficients.push_back(Triplet(jdx, idx+nvertex, zero));
			Mat_coefficients.push_back(Triplet(jdx+nvertex, idx, zero));
			Mat_coefficients.push_back(Triplet(jdx+nvertex, idx+nvertex, zero));
		}
	}

	H.setFromTriplets(Mat_coefficients.begin(), Mat_coefficients.end());

  }

  /* =============================================================================================
   Reset Hessian matrix to solve for flow
   =============================================================================================== */

  void Beltrami::resetHessian(Mesh& mesh, int nfixed)
  {

  /* ==================================================================================================
	Reset all values for regular constraints to 0
   ==================================================================================================*/

	int idx, jdx;

	int nvertex = mesh.vertices.size()-nfixed;

	for (VertexIter v_it = mesh.vertices.begin(); v_it!= mesh.vertices.end(); v_it++)
	{
		idx = v_it->indexN; 
		if( idx >=0 ) {
			H.coeffRef(idx, idx)     = 0;
			H.coeffRef(idx, idx+nvertex)   = 0;
			H.coeffRef(idx+nvertex, idx)   = 0;
			H.coeffRef(idx+nvertex, idx+nvertex)   = 0;
		}
	}

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;

	for (EdgeIter e_it = mesh.edges.begin(); e_it!= mesh.edges.end(); e_it++)
	{
		he = e_it->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		idx = v_i->indexN;
		jdx = v_j->indexN;

		if( idx >=0 && jdx >=0 ) {
			H.coeffRef(idx, jdx)     = 0;
			H.coeffRef(idx, jdx+nvertex)   = 0;
			H.coeffRef(idx+nvertex, jdx)   = 0;
			H.coeffRef(idx+nvertex, jdx+nvertex) = 0;

			H.coeffRef(jdx, idx)     = 0;
			H.coeffRef(jdx, idx+nvertex)   = 0;
			H.coeffRef(jdx+nvertex, idx)   = 0;
			H.coeffRef(jdx+nvertex, idx+nvertex) = 0;
		}
	}

  }

  /* =============================================================================================
     Compute the outer product between two vectors
   =============================================================================================== */

   void Beltrami::outerProduct(double *A, double *B, double *AB)
   {
	AB[0] = A[0]*B[0]; AB[1] = A[0]*B[1]; AB[2]= A[0]*B[2]; 
	AB[3] = A[1]*B[0]; AB[4] = A[1]*B[1]; AB[5]= A[1]*B[2]; 
	AB[6] = A[2]*B[0]; AB[7] = A[2]*B[1]; AB[8]= A[2]*B[2]; 
   }

  /* =============================================================================================
    Contribution of one triangle to the gradient and Hessian of the energy of the flow

    The energy for that triangle is:
	E = E1 + E2 = Area * (A u + B v -2*alpha)^2 + Area* (Cu + Dv -2*beta)^2
    We compute both gradient and Hessian wrt u and v
   =============================================================================================== */

  void Beltrami::trigContribHessian(int nvertex,
		int idx1, Vector& p1, int idx2, Vector& p2, int idx3, Vector& p3,
		double Area, double alpha, double beta)
  {

	int idx1b, idx2b, idx3b;
	idx1b = idx1 + nvertex;
	idx2b = idx2 + nvertex;
	idx3b = idx3 + nvertex;

  /* ==================================================================================================
	Compute local frame for triangle, and local frames for tangent planes to all three points
   ==================================================================================================*/

	Vector w1, w2, w3;
	double x1, x2, x3;

	int nrev;
	nrev = trigLocalFrame(p1, p2, p3, w1, w2, w3, &x1, &x2, &x3);

	Vector a1, a2, a3, b1, b2, b3;
	tangentPlane(p1, a1, b1);
	tangentPlane(p2, a2, b2);
	tangentPlane(p3, a3, b3);

  /* ==================================================================================================
	Compute Different dot products between w vectors and a and b vectors
   ==================================================================================================*/

	double a1w1, a2w1, a3w1, a1w2, a2w2, a3w2;
	double b1w1, b2w1, b3w1, b1w2, b2w2, b3w2;

	a1w1 = dot(a1 , w1); a2w1 = dot(a2 , w1); a3w1 = dot(a3 , w1);
	a1w2 = dot(a1 , w2); a2w2 = dot(a2 , w2); a3w2 = dot(a3 , w2);

	b1w1 = dot(b1 , w1); b2w1 = dot(b2 , w1); b3w1 = dot(b3 , w1);
	b1w2 = dot(b1 , w2); b2w2 = dot(b2 , w2); b3w2 = dot(b3 , w2);

  /* ==================================================================================================
	Write E1 = Area*( Ga u + Gb v -2 alpha)^2
	where 
		u= (u1, u2, u3)
	and
		v = (v1, v2, v3)
	the displacements for vertices p1, p2, p3 along their X and Y directions (in tangent planes)
   ==================================================================================================*/

	double x1inv = 1.0/x1;
	double x1x3inv = 1.0/(x1*x3);

	double Ba[3], Bb[3], Ca[3], Cb[3], Ga[3], Gb[3];

	Ba[0] = -a1w1*x1inv; Ba[1] = a2w1*x1inv; Ba[2] = 0;
	Bb[0] = -b1w1*x1inv; Bb[1] = b2w1*x1inv; Bb[2] = 0;

	Ca[0] = (x2-x1)*a1w2*x1x3inv; Ca[1] = -x2*a2w2*x1x3inv; Ca[2] = x1*a3w2*x1x3inv;
	Cb[0] = (x2-x1)*b1w2*x1x3inv; Cb[1] = -x2*b2w2*x1x3inv; Cb[2] = x1*b3w2*x1x3inv;

	for(int i = 0; i < 3; i++) {
		Ga[i] = Ba[i] - Ca[i]; Gb[i] = Bb[i] - Cb[i];
	}

	double e1 = -2*alpha;

  /* ==================================================================================================
	Write E2 = Area*( Ha u + Hb v -2 beta)^2
   ==================================================================================================*/

	double Da[3], Db[3], Ea[3], Eb[3], Ha[3], Hb[3];

	Da[0] = -a1w2*x1inv; Da[1] = a2w2*x1inv; Da[2] = 0;
	Db[0] = -b1w2*x1inv; Db[1] = b2w2*x1inv; Db[2] = 0;

	Ea[0] = (x2-x1)*a1w1*x1x3inv; Ea[1] = -x2*a2w1*x1x3inv; Ea[2] = x1*a3w1*x1x3inv;
	Eb[0] = (x2-x1)*b1w1*x1x3inv; Eb[1] = -x2*b2w1*x1x3inv; Eb[2] = x1*b3w1*x1x3inv;

	for(int i = 0; i < 3; i++) {
		Ha[i] = Da[i] + Ea[i]; Hb[i] = Db[i] + Eb[i];
	}

	double e2 = -2*beta;

  /* ==================================================================================================
	Gradients along u and v
   ==================================================================================================*/

	double gradU[3], gradV[3];
	for(int i = 0; i < 3; i++) {
		gradU[i] = -2*Area*(e1*Ga[i] + e2*Ha[i]);
		gradV[i] = -2*Area*(e1*Gb[i] + e2*Hb[i]);
	}

	if(idx1 >=0 ) {B[idx1] += gradU[0]; B[idx1b] += gradV[0];}; 
	if(idx2 >=0 ) {B[idx2] += gradU[1]; B[idx2b] += gradV[1];}; 
	if(idx3 >=0 ) {B[idx3] += gradU[2]; B[idx3b] += gradV[2];}; 

  /* ==================================================================================================
	Hessians
   ==================================================================================================*/

	double GaGa[9], GbGb[9], GaGb[9], GbGa[9];

	outerProduct(Ga, Ga, GaGa);
	outerProduct(Gb, Gb, GbGb);
	outerProduct(Ga, Gb, GaGb);
	outerProduct(Gb, Ga, GbGa);

	double HaHa[9], HbHb[9], HaHb[9], HbHa[9];

	outerProduct(Ha, Ha, HaHa);
	outerProduct(Hb, Hb, HbHb);
	outerProduct(Ha, Hb, HaHb);
	outerProduct(Hb, Ha, HbHa);

	if(idx1 >=0) {

		H.coeffRef(idx1, idx1) += 2*Area*(GaGa[0] + HaHa[0]);
		H.coeffRef(idx1, idx1b) += 2*Area*(GaGb[0] + HaHb[0]);
		H.coeffRef(idx1b, idx1b) += 2*Area*(GbGb[0] + HbHb[0]);
		H.coeffRef(idx1b, idx1) += 2*Area*(GbGa[0] + HbHa[0]);

		if(idx2 >= 0) {
			H.coeffRef(idx1, idx2) += 2*Area*(GaGa[1] + HaHa[1]);
			H.coeffRef(idx2, idx1) += 2*Area*(GaGa[3] + HaHa[3]);
			H.coeffRef(idx1, idx2b) += 2*Area*(GaGb[1] + HaHb[1]);
			H.coeffRef(idx2, idx1b) += 2*Area*(GaGb[3] + HaHb[3]);
			H.coeffRef(idx1b, idx2) += 2*Area*(GbGa[1] + HbHa[1]);
			H.coeffRef(idx2b, idx1) += 2*Area*(GbGa[3] + HbHa[3]);
			H.coeffRef(idx1b, idx2b) += 2*Area*(GbGb[1] + HbHb[1]);
			H.coeffRef(idx2b, idx1b) += 2*Area*(GbGb[3] + HbHb[3]);
		}

		if(idx3 >= 0) {
			H.coeffRef(idx1, idx3) += 2*Area*(GaGa[2] + HaHa[2]);
			H.coeffRef(idx3, idx1) += 2*Area*(GaGa[6] + HaHa[6]);
			H.coeffRef(idx1, idx3b) += 2*Area*(GaGb[2] + HaHb[2]);
			H.coeffRef(idx3, idx1b) += 2*Area*(GaGb[6] + HaHb[6]);
			H.coeffRef(idx1b, idx3) += 2*Area*(GbGa[2] + HbHa[2]);
			H.coeffRef(idx3b, idx1) += 2*Area*(GbGa[6] + HbHa[6]);
			H.coeffRef(idx1b, idx3b) += 2*Area*(GbGb[2] + HbHb[2]);
			H.coeffRef(idx3b, idx1b) += 2*Area*(GbGb[6] + HbHb[6]);
		}
	}

	if(idx2 >=0) {

		H.coeffRef(idx2, idx2) += 2*Area*(GaGa[4] + HaHa[4]);
		H.coeffRef(idx2, idx2b) += 2*Area*(GaGb[4] + HaHb[4]);
		H.coeffRef(idx2b, idx2) += 2*Area*(GbGa[4] + HbHa[4]);
		H.coeffRef(idx2b, idx2b) += 2*Area*(GbGb[4] + HbHb[4]);

		if(idx3 >= 0) {
			H.coeffRef(idx2, idx3) += 2*Area*(GaGa[5] + HaHa[5]);
			H.coeffRef(idx3, idx2) += 2*Area*(GaGa[7] + HaHa[7]);
			H.coeffRef(idx2, idx3b) += 2*Area*(GaGb[5] + HaHb[5]);
			H.coeffRef(idx3, idx2b) += 2*Area*(GaGb[7] + HaHb[7]);
			H.coeffRef(idx2b, idx3) += 2*Area*(GbGa[5] + HbHa[5]);
			H.coeffRef(idx3b, idx2) += 2*Area*(GbGa[7] + HbHa[7]);
			H.coeffRef(idx2b, idx3b) += 2*Area*(GbGb[5] + HbHb[5]);
			H.coeffRef(idx3b, idx2b) += 2*Area*(GbGb[7] + HbHb[7]);
		}

	}

	if(idx3 >= 0) {
		H.coeffRef(idx3, idx3) += 2*Area*(GaGa[8] + HaHa[8]);
		H.coeffRef(idx3, idx3b) += 2*Area*(GaGb[8] + HaHb[8]);
		H.coeffRef(idx3b, idx3) += 2*Area*(GbGa[8] + HbHa[8]);
		H.coeffRef(idx3b, idx3b) += 2*Area*(GbGb[8] + HbHb[8]);
	}

  }

  /* =============================================================================================
  Solve Flow: follows the method described in:

 	T. W. Wong and H. Zhao, "Computation of quasiconformal surface maps using discrete Beltrami
 	flow", SIAM J. Imaging Sci., 7(4), 2675–2699 (2014).

   =============================================================================================== */

  void Beltrami::solveFlow(Mesh& mesh, int nfixed, double *nu, double *Area, double *Disp)
  {

  /* ==================================================================================================
	Reset Hessian and gradient to 0 (note that additional constraints remain the same)
   ==================================================================================================*/

	resetHessian(mesh, nfixed);
	int nvertex = mesh.vertices.size()-nfixed;
	for(int i = 0; i < 2*nvertex; i++) B[i] = 0;

  /* ==================================================================================================
	Compute Hessian and gradient by summing over all triangles
   ==================================================================================================*/

	HalfEdgeIter hAB, hBC, hCA;
	Vector p1, p2, p3;

	double alpha, beta, area;
	int iface, idx1, idx2, idx3;

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{

		iface = f_iter->index;

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		p1 = hAB->vertex->position2;
                p2 = hBC->vertex->position2;
                p3 = hCA->vertex->position2;

		idx1 = hAB->vertex->indexN;
                idx2 = hBC->vertex->indexN;
                idx3 = hCA->vertex->indexN;

		if(idx1 >= 0 || idx2 >=0 || idx3 >=0) {
			alpha = nu[2*iface];
			beta  = nu[2*iface+1];
			area  = Area[iface];
			trigContribHessian(nvertex, idx1, p1, idx2, p2, idx3, p3, area, alpha, beta);
		}
	}

  /* ==================================================================================================
	Solve system
   ==================================================================================================*/

//	Eigen::SimplicialCholesky<Eigen::SparseMatrix<double>> chol(H);  // performs a Cholesky factorization of A
	Eigen::CholmodDecomposition<Eigen::SparseMatrix<double>> chol(H);

	if (chol.info() != Eigen::Success){
		std::cout << "Failed Cholesky factorization for Beltrami flow linear system" << std::endl;
		exit(1);
	}

	Sol = chol.solve(B); 

	if (chol.info() != Eigen::Success){
		std::cout << "Failed solving system for Beltrami flow" << std::endl;
		exit(1);
	}

//	double err1 = (H*Sol - B).norm();
//	std::cout << "Error in linear system: " << err1 << std::endl;
//	std::cout << " " << std::endl;

  /* ==================================================================================================
	Get displacements from solutions
   ==================================================================================================*/

	int idx, id1;
	Vector a, b;
	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
                p1 = v_it->position2;
		idx = v_it->index;
		id1 = v_it->indexN;
		if( id1 < 0) {
			Disp[3*idx]   = 0;
			Disp[3*idx+1] = 0;
			Disp[3*idx+2] = 0;
		} else {
			tangentPlane(p1, a, b);
			Disp[3*idx]   = Sol[id1]*a[0] + Sol[id1+nvertex]*b[0];
			Disp[3*idx+1] = Sol[id1]*a[1] + Sol[id1+nvertex]*b[1];
			Disp[3*idx+2] = Sol[id1]*a[2] + Sol[id1+nvertex]*b[2];
		}
	}

  }

  /* =============================================================================================
  Apply (fraction of) displacement
   =============================================================================================== */

  void Beltrami::applyDisp(Mesh& mesh, double step, double *Disp)
  {
	Vector p1;
	double size;
	int idx;

	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
                p1 = v_it->position2;
		idx = v_it ->index;
		p1[0] += step*Disp[3*idx];
		p1[1] += step*Disp[3*idx+1];
		p1[2] += step*Disp[3*idx+2];
		size = p1.norm();
		p1 = p1/size;
		v_it->position2[0] = p1[0];
		v_it->position2[1] = p1[1];
		v_it->position2[2] = p1[2];
	}
  }

  /* =============================================================================================
  Reset Mesh to saved coordinates
   =============================================================================================== */

  void Beltrami::resetMesh(Mesh& mesh, double *Xsave)
  {
	int idx;
	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		idx = v_it ->index;
		v_it->position2[0] = Xsave[3*idx];
		v_it->position2[1] = Xsave[3*idx+1];
		v_it->position2[2] = Xsave[3*idx+2];
	}
  }

  /* =============================================================================================
  Save Mesh coordinates
   =============================================================================================== */

  void Beltrami::saveMesh(Mesh& mesh, double *Xsave)
  {

	int idx;
	for (VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		idx = v_it ->index;
		Xsave[3*idx]    = v_it->position2[0];
		Xsave[3*idx+1]  = v_it->position2[1];
		Xsave[3*idx+2]  = v_it->position2[2];
	}
  }

  /* ===== ErrorMax       ============================================================================
   *
   * Maximum error between expected Beltrami coefficients, and current coefficients
   *
   ==================================================================================================*/

  double Beltrami::errorMax(int n_faces, double *mu_target, double *mu)
  {
	double mut_r, mut_i, mu_r, mu_i;
	double err;

	double err_max = 0;
	for(int i = 0; i < n_faces; i++)
	{
		mut_r = mu_target[2*i];
		mut_i = mu_target[2*i+1];
		mu_r  = mu[2*i];
		mu_i  = mu[2*i+1];

		err = std::sqrt( (mut_r-mu_r)*(mut_r-mu_r) + (mut_i-mu_i)*(mut_i-mu_i) );
		err_max = std::max(err_max, err);
	}

	return err_max;
  }

  /* ===== Beltrami Flow   ============================================================================
   * Refine current spherical embedding to minimize conformal distorsion using Beltrami flow, as
   * described in:
   *	T. W. Wong and H. Zhao, "Computation of quasiconformal surface maps using discrete Beltrami
   *	flow", SIAM J. Imaging Sci., 7(4), 2675–2699 (2014).
   ==================================================================================================*/

  void Beltrami::beltramiFlow(Mesh& mesh) 
  {
	int niter_max = 1000;
	double TOL    = 1.e-8;
	int wtype     = 1;

	int n_vertices = mesh.vertices.size();
	int n_faces    = mesh.faces.size();

  /* ==================================================================================================
	Set weight per triangle: if wtype=0, weights = 1, otherwise use area
   ==================================================================================================*/

	double *Area = new double[n_faces];

	if(wtype==0) {
		for(int i = 0; i < n_faces; i++) {
			Area[i] = 1.0/n_faces;
		}
	} else {
		trigArea(mesh, Area);
	}

  /* ==================================================================================================
	Find three points that will stay fixed
   ==================================================================================================*/

	int nfixed;
	int IdxFixed[3];
	findFixedPoints(mesh, IdxFixed, &nfixed); 

  /* ==================================================================================================
	Create Eigen matrices and arrays needed to solve flow
   ==================================================================================================*/

	int ntot = 2*n_vertices - 2*nfixed;
	H.resize(ntot, ntot);
	B.resize(ntot);
	Sol.resize(ntot);
	double *Disp  = new double[3*n_vertices];
	double *Xsave = new double[3*n_vertices];
	std::vector<Vector > F_info;
	F_info.reserve(n_faces);

	initHessian(mesh, nfixed);
	H_is_set = 1;
	for(int i = 0; i <ntot; i++) B[i] = 0; 

  /* ==================================================================================================
	Compute current Beltrami Coefficients
   ==================================================================================================*/

	saveTrigFrame(mesh, F_info);

	double *mu        = new double[2*n_faces];
	double *mu_target = new double[2*n_faces];
	double *nu        = new double[2*n_faces];

	memset(mu_target, 0, 2*n_faces*sizeof(double));

	int nrev = beltramiCoef(mesh, F_info, mu_target, mu, nu);

  /* ==================================================================================================
	Initial energy that will be refined
   ==================================================================================================*/

	double ene0, ene1;
	ene0 = beltramiEnergy(mesh, Area, mu);

	double step = 0;
	double sup_mu;

  /* ==================================================================================================
	Now perform refinement
   ==================================================================================================*/

	sup_mu = errorMax(n_faces, mu_target, mu);
	int zero = 0;

	std::cout << "        " << "===========================================================================" << std::endl;
	std::cout << "        " << "       Iter       Step size          Energy        Error max     # of flips" << std::endl;
        std::cout << "        " << "===========================================================================" << std::endl;
        std::cout << "        " << "   " << std::setw(8)<< zero << "    " << std::setw(12) << step;
	std::cout << "    " << std::setw(12) << ene0 << "     " << std::setw(12) << sup_mu;
	std::cout << "      " << std::setw(8) << zero << std::endl;


	for(int niter = 0; niter < niter_max; niter++)
	{

  		/* ====================================================================================
		Save a copy of the current mesh geometry
   		======================================================================================*/

		saveMesh(mesh, Xsave);

  		/* ====================================================================================
		Solve flow
   		======================================================================================*/

		solveFlow(mesh, nfixed, nu, Area, Disp);

  		/* ====================================================================================
		Find "optimal" step size:
			- try first step = 1; if corresponding energy ene1 < ene0, apply step
			- if ene1 > ene0, or some flips have occurred, then:
				set step = 0.5*step
   		======================================================================================*/

		step = 1.0;
		applyDisp(mesh, step, Disp);
		nrev = beltramiCoef(mesh, F_info, mu_target, mu, nu);
		ene1 = beltramiEnergy(mesh, Area, mu);

		while ((ene1 > ene0 || nrev > 0) && (step > 0.0001)) {
			step = 0.5*step;
			resetMesh(mesh, Xsave);
			applyDisp(mesh, step, Disp);
			nrev = beltramiCoef(mesh, F_info, mu_target, mu, nu);
			ene1 = beltramiEnergy(mesh, Area, mu);
		}
		sup_mu = errorMax(n_faces, mu_target, mu);

        	std::cout << "        " << "   " << std::setw(8)<< niter+1 << "    " << std::setw(12) << step;
		std::cout << "    " << std::setw(12) << ene1 << "     " << std::setw(12) << sup_mu;
		std::cout << "      " << std::setw(8) << nrev << std::endl;

		if(std::abs(ene1 - ene0) < TOL*ene1) break;
		ene0 = ene1;
	}
        std::cout << "        " << "===========================================================================" << std::endl;
	std::cout << " " << std::endl;

	delete [] Area; delete [] Xsave; delete [] mu; delete [] mu_target; delete [] nu;
	F_info.clear();

  }
