/* ====WillmoreFlow.h ===========================================================================
 *
 * Author: Patrice Koehl, July 2018
 * Department of Computer Science
 * University of California, Davis
 *
 * This file applies a conformalized Willmore Flow to parameterize a surface onto the sphere
 *
 * It is based on the paper:
 *      K. Crane, U. Pinkall, and P. Schroeder, "Robust fairing via conformal curvature flow"
 *       ACM Transactions on Graphics, Vol. 32, No. 4 (2013).
 * and on the code SpinXform_OpenGL from Keenan Crane (available at:
 *	https://www.cs.cmu.edu/~kmcrane/index.html#code
 =============================================================================================== */

#pragma once

  /* ===== INCLUDES AND TYPEDEFS =================================================================
   *
   =============================================================================================== */

#include <vector>
#include <cmath>
#include "Quaternion.h"
#include "ConformalError.h"

/*================================================================================================
  Prototypes for BLAS and LAPACK
================================================================================================== */

extern "C" {

        void dscal_(int * n, double * alpha, double * X, int *incx);
        double ddot_(int * n, double * u, int * incu, double * v, int *incv);

        void dgemv_(char * trans, int * m, int * n, double * alpha, double *A,
                int *lda, double * X, int * incx, double *beta, double * Y, int * incy);

}

 Eigen::CholmodDecomposition<Eigen::SparseMatrix<double>> solverE;
 Eigen::CholmodDecomposition<Eigen::SparseMatrix<double>> solverL;

 int nE = 0;
 int nL = 0;

  /* ===== The Willmore class    =================================================================
   *
   =============================================================================================== */

class Willmore
 {
  public:
	// initialize Willmore flow
        void initFlow(Mesh& mesh, double dt);

	// Perform one step of Willmore flow
        Vector solveOneStep(Mesh& mesh, int niter, double dt);

	// terminate Willmore flow to sphere
        void stopFlow(Mesh& mesh);

   protected:

	// init matrices
	void initMatrix(Mesh& mesh, Eigen::SparseMatrix<double> &Mat);
	void initMatrixL(Mesh& mesh, Eigen::SparseMatrix<double> &Mat);

	// reset matrices
	void resetMatrix(Mesh& mesh, Eigen::SparseMatrix<double> &Mat);
	void resetMatrixL(Mesh& mesh, Eigen::SparseMatrix<double> &Mat);

	// curvature (one value per vertex)
	std::vector<double> curv;

	// area (one value per vertex)
	std::vector<double> area;

	// normal (one value per vertex)
	std::vector<Vector> Normals;

	// controls change in curvature (one value per vertex)
	std::vector<double> rho;

	// local similarity transformation (one value per vertex)
	std::vector<Quaternion> lambda;

	// divergence of target edge vectors
	Eigen::VectorXd omega;

	// temp vectors
	Eigen::VectorXd temp1, temp2;

	// Constraint matrix
	double *Constraints;

	double Area0;

	Eigen::SparseMatrix<double> L; //Laplace matrix
	Eigen::SparseMatrix<double> E; //matrix for eigenvalue problem

	// update real sparse matrix with a quaternion value
	void updateMatrix(Eigen::SparseMatrix<double>& Mat, int i0, int j0, double Q[4][4]);
	void updateMatrixL(Eigen::SparseMatrix<double>& Mat, int i0, int j0, double Q[4][4]);

	// update real vector with a quaternion value
	void updateVector(Eigen::VectorXd& Vect, int i0, Quaternion q);

	// removeMean for a vector of quaternions (stored as a regular vector of doubles)
	void removeMeanQ(Eigen::VectorXd& Vect);

	// compute current mean curvature at each vertex
	void meanCurvature(Mesh& mesh);

	void applyConstraints();

	// flow in curvature: compute new mean curvature array rho
	void buildNewRho(Mesh& mesh, double dt);

	// build Eigenvalue problem to compute functions lambda
	void buildEigenvalueProblem(Mesh& mesh);

	// solve for functions lambda by solving an Eigenvalue problem
	void solveForLambda();

	// Build Poisson problem to convert tangents to coordinates
	void buildPoissonProblem(Mesh& mesh);

	// Build current Laplacian over the mesh
	void buildLaplacian(Mesh& mesh);

	// Build tangents
	void buildOmega(Mesh& mesh);

	// solve for new coordinates
	void solveForPositions(Mesh& mesh);

	// Recenter and scale mesh
	void normalizeSolution(Mesh& mesh);

	// Gram Schmidt: one orthogonalization
	void gramSchmidt(int N, int M, double *V, double *coef);

	// Gram Schmidt: multiple orthogonalization
	void orthoNormalize(int N, int M, int Madd, double *V);

	// Compute Mesh info: area, vol, sphericity
	void computeAVS(Mesh& mesh, double *Area, double *Vol, double *S);

	// Fit Sphere
	void fitSphere(Mesh& mesh);
};

  /* =============================================================================================
   Init matrix
   =============================================================================================== */

  void Willmore::initMatrix(Mesh& mesh, Eigen::SparseMatrix<double> &Mat)
  {

	// Initialize all values to 0

	int idx, jdx;
	std::vector<Triplet> Mat_coefficients;
	double zero = 0;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idx = v->index;
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat_coefficients.push_back(Triplet(4*idx+u, 4*idx+v, zero));
			}
		}
	}

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;
	
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		idx = v_i->index;
		jdx = v_j->index;
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat_coefficients.push_back(Triplet(4*idx+u, 4*jdx+v, zero));
			}
		}
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat_coefficients.push_back(Triplet(4*jdx+u, 4*idx+v, zero));
			}
		}
	}

	Mat.setFromTriplets(Mat_coefficients.begin(), Mat_coefficients.end());

  }
  /* =============================================================================================
   Init matrix L
   =============================================================================================== */

  void Willmore::initMatrixL(Mesh& mesh, Eigen::SparseMatrix<double> &Mat)
  {

	// Initialize all values to 0

	int idx, jdx;
	std::vector<Triplet> Mat_coefficients;
	double zero = 0;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end() - 1; v++)
	{
		idx = v->index;
		for(int u = 0; u < 4; u++) {
			Mat_coefficients.push_back(Triplet(4*idx+u, 4*idx+u, zero));
		}
	}

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;
	int n_vertices = mesh.vertices.size();
	
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		idx = v_i->index;
		jdx = v_j->index;

		if(idx != n_vertices-1 && jdx != n_vertices -1) {
			for(int u = 0; u < 4; u++) {
				Mat_coefficients.push_back(Triplet(4*idx+u, 4*jdx+u, zero));
			}
			for(int u = 0; u < 4; u++) {
				Mat_coefficients.push_back(Triplet(4*jdx+u, 4*idx+u, zero));
			}
		}
	}

	Mat.setFromTriplets(Mat_coefficients.begin(), Mat_coefficients.end());

  }


  /* =============================================================================================
   Reset matrix
   =============================================================================================== */

  void Willmore::resetMatrix(Mesh& mesh, Eigen::SparseMatrix<double> &Mat)
  {

	int idx, jdx;
	double zero = 0;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idx = v->index;
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat.coeffRef(4*idx+u, 4*idx+v) = zero;
			}
		}
	}

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;
	
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		idx = v_i->index;
		jdx = v_j->index;
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat.coeffRef(4*idx+u, 4*jdx+v) = zero;
			}
		}
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat.coeffRef(4*jdx+u, 4*idx+v) = zero;
			}
		}
	}

  }

  /* =============================================================================================
   Reset matrix L
   =============================================================================================== */

  void Willmore::resetMatrixL(Mesh& mesh, Eigen::SparseMatrix<double> &Mat)
  {

	int idx, jdx;
	double zero = 0;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end()-1; v++)
	{
		idx = v->index;
		for(int u = 0; u < 4; u++) {
			Mat.coeffRef(4*idx+u, 4*idx+u) = zero;
		}
	}

	int n_vertices = mesh.vertices.size();

	HalfEdgeIter he, he2;
	VertexIter v_i, v_j;
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		v_i = he->vertex;
		v_j = he2->vertex;

		idx = v_i->index;
		jdx = v_j->index;

		if(idx != n_vertices-1 && jdx != n_vertices-1) {
			for(int u = 0; u < 4; u++) {
				Mat.coeffRef(4*idx+u, 4*jdx+u) = zero;
			}
			for(int u = 0; u < 4; u++) {
				Mat.coeffRef(4*jdx+u, 4*idx+u) = zero;
			}
		}
	}

  }

  /* ================================================================================================
   * Init flow
   ==================================================================================================*/

  void Willmore::initFlow(Mesh& mesh, double dt)
  {
	int n_vertices = mesh.vertices.size();
	int n4         = 4*n_vertices;
	int n4m1       = 4*(n_vertices-1);

	// Create Eigen matrices and arrays needed to minimize curvatures

	E.resize(n4, n4);
	L.resize(n4m1, n4m1);
	curv.resize(n_vertices);
	area.resize(n_vertices);
	Normals.resize(n_vertices);
	rho.resize(n_vertices);
	lambda.resize(n_vertices);
	omega.resize(n4m1);
	temp1.resize(n4);
	temp2.resize(n4);

	Constraints = new double[4*n_vertices];

	initMatrix(mesh, E);
	initMatrixL(mesh, L);

	temp1.setZero();
	for(int i = 0; i < temp1.size()/4; i++) temp1.coeffRef(4*i) = 1.0;
	temp1.normalize();

	// Initialize positions
	for(VertexIter v_iter = mesh.vertices.begin(); v_iter != mesh.vertices.end(); v_iter++)
	{
		v_iter->position2[0] = v_iter->position[0];
		v_iter->position2[1] = v_iter->position[1];
		v_iter->position2[2] = v_iter->position[2];
	}

	int zero = 0;
	double Area, Vol, S;
	computeAVS(mesh, &Area, &Vol, &S);
	Area0 = Area;
	Vector r = ConformalError::quasiConformalError(mesh);
	double quasi = r[2];

	std::cout << "        " << "===================================================================================" << std::endl;
	std::cout << "        " << "       Iter       Step size        Area     QuasiConf. ratio   Sphericity          " << std::endl;
        std::cout << "        " << "===================================================================================" << std::endl;
        std::cout << "        " << "   " << std::setw(8)<< zero << "    " << std::setw(12) << dt;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << Area;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << quasi;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << S << std::endl;

  }

  /* ================================================================================================
   * End flow
   ==================================================================================================*/

  void Willmore::stopFlow(Mesh& mesh)
  {
        std::cout << "        " << "===================================================================================" << std::endl;
	std::cout << " " << std::endl;

	fitSphere(mesh);
  }

  /* =================================================================================================
  * Mesh area, Vol, sphericity
   ==================================================================================================*/

  void Willmore::computeAVS(Mesh& mesh, double *Area, double *Vol, double *S)
  {
	double A, V;

	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;
	Vector p1, p2, p3;

	A = 0; V = 0;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		p1 = v1->position2;
		p2 = v2->position2;
		p3 = v3->position2;

		A += cross(p1-p2,p1-p3).norm();

		V += dot(p1 , cross(p2,p3) );
	}

	A = 0.5*A;
	V = std::abs(V)/6.0;
	*Area = A;
	*Vol  = V;
	*S    = std::pow(M_PI,1.0/3.0) * std::pow(6.0*V,2.0/3.0) / A;

  }

  /* =============================================================================================
   *	Solve one step
   =============================================================================================== */

Vector Willmore::solveOneStep(Mesh& mesh, int niter, double dt)
{

	// Compute update curvature
	buildNewRho(mesh, dt);

	// solve eigenvalue problem for local similarity transformation lambda
	buildEigenvalueProblem(mesh);
	solveForLambda();

	// solve Poisson problem for new vertex positions
	buildPoissonProblem(mesh);
	solveForPositions(mesh);

	normalizeSolution(mesh);

	double Area, Vol, S;
	computeAVS(mesh, &Area, &Vol, &S);
	Vector r = ConformalError::quasiConformalError(mesh);
	double quasi = r[2];

	std::cout << "        " << "   " << std::setw(8)<< niter+1 << "    " << std::setw(12) << dt;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << Area ;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << quasi;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << S << std::endl;

	return Vector(S, quasi, 0);
}

  /* =============================================================================================
   *	Build new rho
   =============================================================================================== */

void Willmore::buildNewRho(Mesh & mesh, double dt)
{
	// Compute current curvature
	meanCurvature(mesh);

	// Initialize rho
	for(int i = 0; i < rho.size(); i++) {
		rho[i] = -curv[i];
	}

	// Build constrains and correct rho
	applyConstraints();

	// Set rho
	for(int i = 0; i < rho.size(); i++) rho[i] = dt*rho[i];
}

  /* =============================================================================================
   *	Apply constraints on flow
   =============================================================================================== */

  void Willmore::applyConstraints()
  {

	int nvertices = rho.size();

	for(int i = 0; i < nvertices; i++)
	{
		Constraints[i] = 1.0/nvertices;
		Constraints[i+nvertices] = Normals[i][0];
		Constraints[i+2*nvertices] = Normals[i][1];
		Constraints[i+3*nvertices] = Normals[i][2];
	}

	int m = 0; int madd = 4;
	orthoNormalize(nvertices, m, madd, Constraints);

	int nconstraints = 4;
//	int nconstraints = 1;
	double *coef = new double[nconstraints];
	for(int i = 0; i < nconstraints; i++) {
		coef[i] = 0.0;
		for(int j = 0; j < nvertices; j++) coef[i] += rho[j]*Constraints[i*nvertices+j];
	}
	for(int i = 0; i < nconstraints; i++) {
		for(int j = 0; j < nvertices; j++) rho[j] -= coef[i]*Constraints[i*nvertices+j];
	}

	double sum = 0, sumx = 0, sumy=0, sumz=0;
	for(int i = 0; i < nvertices; i++) {
		sum += rho[i];
		sumx += rho[i]*Normals[i][0];
		sumy += rho[i]*Normals[i][1];
		sumz += rho[i]*Normals[i][2];
	}
	std::cout << "sum_rho: " << sum << std::endl;
	std::cout << "rho_Nx: " << sumx << std::endl;
	std::cout << "rho_Ny: " << sumy << std::endl;
	std::cout << "rho_Nz: " << sumz << std::endl;

  }
	
  /* =============================================================================================
   * Remove mean from a vector of quaternions (stored as a vector of reals)
   =============================================================================================== */

  void Willmore::removeMeanQ(Eigen::VectorXd& Vect)
  {
	Quaternion q, mean;

	int n = Vect.size();
	int nq = n/4;

	mean = 0;
	for(int i = 0; i < nq; i++) {
		q = Quaternion( Vect[4*i], Vect[4*i+1], Vect[4*i+2], Vect[4*i+3]);
		mean += q;
	}

	mean /= (double) nq;

	double v0, v1, v2, v3;
	v0 = mean.re();
	v1 = mean.im().x;
	v2 = mean.im().y;
	v3 = mean.im().z;

	for(int i = 0; i < nq; i++) {
		Vect[4*i+0] -= v0;
		Vect[4*i+1] -= v1;
		Vect[4*i+2] -= v2;
		Vect[4*i+3] -= v3;
	}
  }

  /* =============================================================================================
   * update real vector with a quaternion value
   =============================================================================================== */

  void Willmore::updateVector(Eigen::VectorXd& Vect, int i0, Quaternion q)
  {
	Vect[4*i0+0] += q.re();
	Vect[4*i0+1] += q.im().x;
	Vect[4*i0+2] += q.im().y;
	Vect[4*i0+3] += q.im().z;
  }

  /* =============================================================================================
   * update real sparse matrix with a quaternion value
   =============================================================================================== */

  void Willmore::updateMatrix(Eigen::SparseMatrix<double>& Mat, int i0, int j0, double Q[4][4])
  {
	int idx, jdx;
	for(int u = 0; u < 4; u++) {
		for(int v = 0; v < 4; v++) {
			idx = 4*i0 + u;
			jdx = 4*j0 + v;
			Mat.coeffRef(idx, jdx) += Q[u][v];
		}
	}
   }

  /* =============================================================================================
   * update real sparse matrix with a quaternion value
   =============================================================================================== */

  void Willmore::updateMatrixL(Eigen::SparseMatrix<double>& Mat, int i0, int j0, double Q[4][4])
  {
	int idx, jdx;
	for(int u = 0; u < 4; u++) {
		idx = 4*i0 + u;
		jdx = 4*j0 + u;
		Mat.coeffRef(idx, jdx) += Q[u][u];
	}
   }

  /* =============================================================================================
   *	Compute current mean curvature at each vertex
   =============================================================================================== */

  void Willmore::meanCurvature(Mesh& mesh)
  {

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		curv[v->index] = 0;
		area[v->index] = 0;
		Normals[v->index] = v->normal2();
	}

	VertexIter v1, v2, v3;
	int idxA, idxB, idxC;
	double A;

	// First, compute area around each vertex 
	for( FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++ )
	{
		A = f_iter->area2();

		v1 = f_iter->he->vertex;
		v2 = f_iter->he->next->vertex;
		v3 = f_iter->he->next->next->vertex;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		area[idxA] += A;
		area[idxB] += A;
		area[idxC] += A;
	}

	double weight, cot_beta_1, cot_beta_2;
	HalfEdgeIter he, he2;
	Vector u, v, p1, p2, p3;
	int idx, jdx;

	// Now loop over all edges
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		v1 = e->he->vertex;
		v2 = e->he->flip->vertex;
		idx = v1->index;
		jdx = v2->index;
		p1 = v1->position2;
		p2 = v2->position2;

		p3 = e->he->prev->vertex->position2;
		u = p1-p3; v = p2-p3;
		cot_beta_1 = dot(u, v) / cross(u, v).norm();

		p3 = e->he->flip->prev->vertex->position2;
		u = p1-p3; v = p2-p3;
		cot_beta_2 = dot(u, v) / cross(u, v).norm();

		weight = cot_beta_1 + cot_beta_2;
		curv[idx] += weight*dot(p2-p1, Normals[idx]);
		curv[jdx] += weight*dot(p1-p2, Normals[jdx]);
	}

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		curv[v->index] *= -0.25/area[v->index];
	}
  }

  /* =============================================================================================
   *	Build eigenvalue problem
   =============================================================================================== */

void Willmore::buildEigenvalueProblem(Mesh& mesh)
{

	// Reset Eigenvalue matrix to 0
	resetMatrix(mesh, E);

	// Pointers to mesh structure
	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;
	Vector p1, p2, p3, p;
	int idxA, idxB, idxC;
	double A, a, b, c;
	Quaternion e[3];
	Quaternion q;
	double Q[4][4];

	// visit each face
	for( FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++ )
	{

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		p1 = v1->position2;
		p2 = v2->position2;
		p3 = v3->position2;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		A = 0.5 * cross(p1-p2,p1-p3).norm();
		a = -1. / (4.*A);
		b = 1./ 6.;
		c = A / 9.;

		int I[3] = {idxA, idxB, idxC};

		// compute edges across from each vertex
		p = p3-p2;
		e[0] = Quaternion(0., p[0], p[1], p[2]);
		p = p1-p3;
		e[1] = Quaternion(0., p[0], p[1], p[2]);
		p = p2-p1;
		e[2] = Quaternion(0., p[0], p[1], p[2]);

		// increment matrix entry for each ordered pair of vertices

		for( int i = 0; i < 3; i++ ) {
			for( int j = 0; j < 3; j++ ) {
				q = a*e[i]*e[j] + b*(rho[I[i]]*e[j]-rho[I[j]]*e[i]) + c*rho[I[i]]*rho[I[j]];
				q.toMatrix(Q);
				updateMatrix(E, I[i], I[j], Q);
			}
		}
	}
  }

  /* =============================================================================================
   *	Build Poisson problem
   =============================================================================================== */

void Willmore::buildPoissonProblem(Mesh& mesh)
{
	buildLaplacian(mesh);
	buildOmega(mesh);
}

  /* =============================================================================================
   *	Build Laplacian matrix (for Poisson problem)
   =============================================================================================== */

void Willmore::buildLaplacian(Mesh& mesh)
{
	// Reset L matrix ot zero
	resetMatrixL(mesh, L);

	// Pointers to mesh structure
	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;
	Vector p1, p2, p3, p;
	int idxA, idxB, idxC;
	int n_vertices = mesh.vertices.size();

	double cotan_A, cotan_B, cotan_C;
	Quaternion qA, qB, qC, q;
	double QA[4][4], QB[4][4], QC[4][4], Q[4][4];

	// visit each face
	for( FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++ )
	{

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		p1 = v1->position2;
		p2 = v2->position2;
		p3 = v3->position2;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		cotan_A = dot(p2-p1, p3-p1) / ( cross(p2-p1, p3-p1).norm());
		cotan_B = dot(p1-p2, p3-p2) / ( cross(p1-p2, p3-p2).norm());
		cotan_C = dot(p1-p3, p2-p3) / ( cross(p1-p3, p2-p3).norm());

		qA = -0.5*cotan_A;
		qB = -0.5*cotan_B;
		qC = -0.5*cotan_C;
		qA.toMatrix(QA);
		qB.toMatrix(QB);
		qC.toMatrix(QC);

		q = -qB - qC; q.toMatrix(Q);
		if(idxA != n_vertices-1) updateMatrixL(L, idxA, idxA, Q);
		q = -qA - qC; q.toMatrix(Q);
		if(idxB != n_vertices-1) updateMatrixL(L, idxB, idxB, Q);
		q = -qA - qB; q.toMatrix(Q);
		if(idxC != n_vertices-1) updateMatrixL(L, idxC, idxC, Q);

		if(idxA != n_vertices-1 && idxB != n_vertices-1) {
			updateMatrixL(L, idxA, idxB, QC);
			updateMatrixL(L, idxB, idxA, QC);
		}
		if(idxA != n_vertices-1 && idxC != n_vertices-1) {
			updateMatrixL(L, idxA, idxC, QB);
			updateMatrixL(L, idxC, idxA, QB);
		}
		if(idxB != n_vertices-1 && idxC != n_vertices-1) {
			updateMatrixL(L, idxB, idxC, QA);
			updateMatrixL(L, idxC, idxB, QA);
		}
	}

}

  /* =============================================================================================
   *	Build Omega vector (new tangent vector), for Poisson problem
   =============================================================================================== */

 void Willmore::buildOmega(Mesh& mesh)
 {

	// clear current omega vector
	for( size_t i = 0; i < omega.size(); i++ ) omega[i] = 0.;

	// Pointers to mesh structure
	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;
	Vector p1, p2, p3, p;
	int idxA, idxB, idxC;
	int idA, idB, idC;
	int n_vertices = mesh.vertices.size();

	double cotan_A, cotan_B, cotan_C;
	Quaternion qA, qB, qC, q, qm;
	Quaternion eTildeAB, eTildeBC, eTildeCA;

	// visit each face
	for( FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++ )
	{

		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		idA = v1->index;
		idB = v2->index;
		idC = v3->index;

		idxA = std::min(idA, std::min(idB, idC));
		idxC = std::max(idA, std::max(idB, idC));
		idxB = idA + idB + idC - idxA - idxC;

		p1 = mesh.vertices[idxA].position2;
		p2 = mesh.vertices[idxB].position2;
		p3 = mesh.vertices[idxC].position2;

		cotan_A = dot(p2-p1, p3-p1) / ( cross(p2-p1, p3-p1).norm());
		cotan_B = dot(p1-p2, p3-p2) / ( cross(p1-p2, p3-p2).norm());
		cotan_C = dot(p1-p3, p2-p3) / ( cross(p1-p3, p2-p3).norm());

		qA = lambda[idxA];
		qB = lambda[idxB];
		qC = lambda[idxC];

		Quaternion eAB = Quaternion(0., p2[0] -p1[0], p2[1]-p1[1], p2[2]-p1[2]);
		Quaternion eBC = Quaternion(0., p3[0] -p2[0], p3[1]-p2[1], p3[2]-p2[2]);
		Quaternion eCA = Quaternion(0., p1[0] -p3[0], p1[1]-p3[1], p1[2]-p3[2]);

		eTildeAB = (1./3.) * (~qA) * eAB * qA +
		           (1./6.) * (~qA) * eAB * qB +
		           (1./6.) * (~qB) * eAB * qA +
		           (1./3.) * (~qB) * eAB * qB ;
		eTildeBC = (1./3.) * (~qB) * eBC * qB +
		           (1./6.) * (~qB) * eBC * qC +
		           (1./6.) * (~qC) * eBC * qB +
		           (1./3.) * (~qC) * eBC * qC ;
		eTildeCA = (1./3.) * (~qC) * eCA * qC +
		           (1./6.) * (~qC) * eCA * qA +
		           (1./6.) * (~qA) * eCA * qC +
		           (1./3.) * (~qA) * eCA * qA ;

		q = 0.5*cotan_C * eTildeAB; qm = -q;
		if(idxA != n_vertices-1) updateVector(omega, idxA, qm);
		if(idxB != n_vertices-1) updateVector(omega, idxB, q);
		q = 0.5*cotan_A * eTildeBC; qm = -q;
		if(idxB != n_vertices-1) updateVector(omega, idxB, qm);
		if(idxC != n_vertices-1) updateVector(omega, idxC, q);
		q = 0.5*cotan_B * eTildeCA; qm = -q;
		if(idxC != n_vertices-1) updateVector(omega, idxC, qm);
		if(idxA != n_vertices-1) updateVector(omega, idxA, q);

	}

}

/*================================================================================================
 Solve for new coordinates
================================================================================================== */

void Willmore::solveForPositions(Mesh& mesh)
{

	// Cholesky factorization of L
	if(nL==0) solverL.analyzePattern(L);
        nL++;
	solverL.factorize(L);

	temp2 = solverL.solve(omega);

	for(VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end()-1 ; v++) {
		v->position2[0] = temp2[4*v->index + 1];
		v->position2[1] = temp2[4*v->index + 2];
		v->position2[2] = temp2[4*v->index + 3];
	}

	int nv = mesh.vertices.size();
	mesh.vertices[nv-1].position2[0] = 0;
	mesh.vertices[nv-1].position2[1] = 0;
	mesh.vertices[nv-1].position2[2] = 0;

}

  /* =============================================================================================
   *	Normalize mesh: re-center, and scale so that points are at most at distance 1 from center
   =============================================================================================== */

  void Willmore::normalizeSolution(Mesh& mesh)
  {

	double Area, Vol, S;
	computeAVS(mesh, &Area, &Vol, &S);
	double Scale = std::sqrt(Area0/Area);

	Vector center;
	center[0] = 0; center[1] = 0; center[2]=0;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->position2 *= Scale;
		center[0] += v->position2[0];
		center[1] += v->position2[1];
		center[2] += v->position2[2];
	}
	int n_vertices = mesh.vertices.size();
	center /= n_vertices;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->position2 -= center;
	}

  }

/*================================================================================================
 gramSchmidt: applies the Gram Schmidt ortho-normalization technique to ortho-normalize V(:, M)
                 against V(:,0:M-1).
                 On output, V(:,0:M) should be orthonormal.
		It is important that the input matrix V(:, 0:M) is orthonormal
	Input:
		N: number of rows in matrix
		M: current # of columns that are orthonormal
		V : the matrix
		coef: work array of size (M+1)
	Output:
		V, the fully orthonormal matrix

================================================================================================== */

void Willmore::gramSchmidt(int N, int M, double *V, double *coef)
{

	double reorth = 0.5;

	double normX, scal, new_norm;
	int inc=1;
	char Trans = 'T';
	char NoTrans = 'N';
	double alpha = 1;
	double alpham = -1;
	double beta0 = 0;
	double beta1 = 1;
	int repeat;

	int current = M;

	double normC;

	//Make sure vector is not 0 and of norm 1

	normX = ddot_(&N, &V[current*N], &inc, &V[current*N], &inc);
	normX = std::sqrt(normX);

	scal = 1.0/normX;
	dscal_(&N, &scal, &V[current*N], &inc);
	normX = 1.0;

	// Allow for repeats if orthogonalization fails twice (which "should" not happen)

	repeat = 0;
	while (repeat < 5 ) {

		// Project new vector onto all preceeding vectors in matrix

		dgemv_(&Trans, &N, &current, &alpha, V, &N, &V[current*N], &inc, &beta0, 
		coef, &inc);
		normC = ddot_(&current, coef, &inc, coef, &inc);
		normC = std::sqrt(normC);

		// Remove projection

		dgemv_(&NoTrans, &N, &current, &alpham, V, &N, coef, &inc, &beta1, 
		&V[current*N], &inc);

		// Check new vector

		new_norm = ddot_(&N, &V[current*N], &inc, &V[current*N], &inc);
		new_norm = std::sqrt(new_norm);

		scal = 1/new_norm;
		dscal_(&N, &scal, &V[current*N], &inc);

		if(new_norm > reorth*normX) {

			//If pass renormalization test, normalize and we are done for this vector
			return;

		}

		//If not, repeat process

		repeat++;

	}
}

/*================================================================================================
 orthoNormalize: applies the DGKS ortho-normalization technique to ortho-normalize V(:, M:M+Madd)
                 against V(:,0:M).
                 On output, V(:,0:M+Madd) should be orthonormal.
		It is important that the input matrix V(:, 1:M) is orthonormal

	Input:
		N: number of rows in matrix
		M: current # of columns that are orthonormal
		Madd: number of columns not yet orthogonal, and scaled
		V : the matrix
		coef: work array of size (M+Madd)
	Output:
		V, the fully orthonormal matrix

================================================================================================== */

void Willmore::orthoNormalize(int N, int M, int Madd, double *V)
{
	double *coef = new double[M+Madd];
	int current;
	for (int m = 0; m< Madd; m++)
	{
		current = M + m;
		gramSchmidt(N, current, V, coef);

	}

}

/*================================================================================================
 PowerMethod: finds the largest eigenvalue of the matrix H
================================================================================================== */

void Willmore::solveForLambda()
{

	int iter_max;
	if(nE==0) {
		iter_max = 20;
	} else {
		iter_max = 20;
	}
	double eig0, eig1;
	double tol = 1.e-10;

	// initialize vector to the identity
//	temp1.setZero();
//	for(int i = 0; i < temp1.size()/4; i++) temp1.coeffRef(4*i) = 1.0;
//	temp1.normalize();

	// Cholesky factorization of E
	if(nE==0) solverE.analyzePattern(E);
        nE++;
	solverE.factorize(E);

 	// Power iterations

	eig0 = 0.0;

	for(int iter =0; iter < iter_max; iter++)
	{
 		// inv(E) - vector multiply

		temp2 = solverE.solve(temp1);

 		//Normalize new vector

		eig1 = temp1.dot(temp2);
		temp2.normalize();

 		// Check for convergence

		if(std::abs(eig1-eig0) < tol) {
			break;
		}

 		// Prepare for next iteration

		temp1 = temp2;
		eig0 = eig1;

	}

	temp1.normalize();
	for(int i = 0; i < lambda.size(); i++) {
		lambda[i] = Quaternion( temp1[4*i], temp1[4*i+1], temp1[4*i+2], temp1[4*i+3]);
	}

}

  /* ===============================================================================================
   FitSphere: Fit a sphere into the vertices of a mesh, and then translate the mesh such that
		the center of the sphere is at 0,0,0

   Input:
	  mesh:	 the mesh data structure (pointer to the structure)
   =============================================================================================== */

  void Willmore::fitSphere(Mesh& mesh)
  {
	int n_vertices = mesh.vertices.size();

	Eigen::MatrixXd A(n_vertices, 4);
	Eigen::VectorXd B(n_vertices);
	Eigen::VectorXd C(4);

	Vector pointA, centerA;

	double xA, yA, zA;
	int idx = 0;
	for(VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		pointA = v_it->position2;
		xA = pointA[0]; yA = pointA[1]; zA = pointA[2];
		A(idx, 0) = 2*xA; A(idx, 1) = 2*yA; A(idx, 2) = 2*zA;
		A(idx, 3) = 1.0;
		B[idx]    = xA*xA + yA*yA + zA*zA;
		idx++;
	}

	C = A.bdcSvd(Eigen::ComputeThinU | Eigen::ComputeThinV).solve(B);

	centerA[0] = C[0]; centerA[1] = C[1]; centerA[2] = C[2];

	for(VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		pointA = v_it->position2;
		pointA = pointA-centerA;
		pointA = pointA / pointA.norm();
		v_it->position2[0] = pointA[0];
		v_it->position2[1] = pointA[1];
		v_it->position2[2] = pointA[2];
	}
  }
