#pragma once

#include <stack>
#include "Types.h"
#include "HalfEdge.h"
#include "Edge.h"

class IDT {
public:
	// Generate implicit Delaunay triangulation for mesh
	static void iDT(Mesh& mesh);

	// Computes all edge length for initial mesh
	static void computeLength(Mesh& mesh);

	// Flip an edge
	static void flipEdge(EdgeIter e0);

	// Check flip
	static void checkFlip(EdgeIter e0);

	// Check Delaunay
	static bool checkDelaunay(EdgeIter e0);

	// is Flippable
	static bool isFlippable(EdgeIter e0);

};

  /* ===============================================================================================
   computeLength: computes initial length of all edges
   =============================================================================================== */

  void IDT::computeLength(Mesh& mesh)
  {

	for(EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		HalfEdgeIter h = e->he;
		Vector p0 = h->vertex->position;
		Vector p1 = h->flip->vertex->position;
		double length = (p0-p1).norm();
		e->length = length;
	}
  }

  /* ===============================================================================================
  Procedure to check if an edge is flippable
   =============================================================================================== */

bool IDT::isFlippable(EdgeIter e0) {
	int degree0 = e0->he->vertex->degree();
	int degree1 = e0->he->flip->vertex->degree();
	if(degree0 == 3 || degree1 == 3) return false;

	int ie1 = e0->he->edge->index;
	int ie2 = e0->he->next->edge->index;
	int ie3 = e0->he->next->next->edge->index;
	int je1 = e0->he->flip->edge->index;
	int je2 = e0->he->flip->next->edge->index;
	int je3 = e0->he->flip->next->next->edge->index;

	int nequal = 0;
	if(ie1==je1 || ie1==je2 || ie1==je3) nequal++;
	if(ie2==je1 || ie2==je2 || ie2==je3) nequal++;
	if(ie3==je1 || ie3==je2 || ie3==je3) nequal++;

	if(nequal>1) return false;

	ie1 = e0->he->vertex->index;
	ie2 = e0->he->flip->vertex->index;
	je1 = e0->he->prev->vertex->index;
	je2 = e0->he->flip->prev->vertex->index;

	int id1, id2, id3;
	int if1, if2, if3;
	id1 = std::min(ie1, std::min(je1, je2)); id3 = std::max(ie1, std::max(je1, je2)); id2 = ie1+je1+je2-id1-id3;
	if1 = std::min(ie2, std::min(je1, je2)); if3 = std::max(ie2, std::max(je1, je2)); if2 = ie2+je1+je2-if1-if3;

	int i0, j0, k0;
	int i1, j1, k1;
	HalfEdgeIter he = e0->he->prev->vertex->he;
	HalfEdgeCIter h = he;
        do {
		FaceIter f=h->face;
                i0 = f->he->vertex->index; 
                j0 = f->he->next->vertex->index; 
                k0 = f->he->next->next->vertex->index; 
                i1 = std::min(i0, std::min(j0, k0)); k1 = std::max(i0, std::max(j0, k0)); j1 = i0+j0+k0-i1-k1;
		if(id1==i1 && id2==j1 && id3 == k1) return false;
		if(if1==i1 && if2==j1 && if3 == k1) return false;

                h = h->flip->next;
        } while (h != he);

	return true;
}

  /* ===============================================================================================
  Procedure to check if an edge is locally Delaunay
   =============================================================================================== */

bool IDT::checkDelaunay(EdgeIter e0)
{
	HalfEdgeIter h0 = e0->he;
	HalfEdgeIter h1 = h0->next;
	HalfEdgeIter h2 = h1->next;
	EdgeIter e1 = h1->edge;
	EdgeIter e2 = h2->edge;
	double a = e0->length;
	double b = e1->length;
	double c = e2->length;
	double val = ((a-b+c)*(a+b-c))/((a+b+c)*(-a+b+c));
	double ta = std::sqrt(val);
	double cota = (1-ta*ta)/(2*ta);

	h0 = h0->flip;
	h1 = h0->next;
	h2 = h1->next;
	e1 = h1->edge;
	e2 = h2->edge;
	a = e0->length;
	b = e1->length;
	c = e2->length;
	val = ((a-b+c)*(a+b-c))/((a+b+c)*(-a+b+c));
	double tb = std::sqrt(val);
	double cotb = (1-tb*tb)/(2*tb);

	if(cota+cotb < 0) {
//		std::cout << "cota + cotb: " << cota + cotb << std::endl;
		return false;
	}
	return true;
}

  /* ===============================================================================================
  Procedure to flip an edge and compute length of new edge
   =============================================================================================== */

  void IDT::flipEdge(EdgeIter e0) 
  {

	// Collect current half edges in the two triangles incident on e0

	HalfEdgeIter h0 = e0->he;
	HalfEdgeIter h1 = h0->next;
	HalfEdgeIter h2 = h1->next;
	HalfEdgeIter h3 = h0->flip;
	HalfEdgeIter h4 = h3->next;
	HalfEdgeIter h5 = h4->next;
	HalfEdgeIter h6 = h1->flip;
	HalfEdgeIter h7 = h2->flip;
	HalfEdgeIter h8 = h4->flip;
	HalfEdgeIter h9 = h5->flip;

	// Collect current edges

	EdgeIter e1 = h1->edge;
	EdgeIter e2 = h2->edge;
	EdgeIter e3 = h4->edge;
	EdgeIter e4 = h5->edge;

	// Collect current vertex

	VertexIter v0 = h0->vertex;
	VertexIter v1 = h3->vertex;
	VertexIter v2 = h2->vertex;
	VertexIter v3 = h5->vertex;

	double val, a, b, c, d, e;
	double ta2, td2;
	a = e3->length;
	b = e4->length;
	c = e1->length;
	d = e2->length;
	e = e0->length;
	val = ((a-b+e)*(a+b-e))/((a+b+e)*(-a+b+e));
	ta2 = std::sqrt(val);
	val = ((d-c+e)*(d+c-e))/((d+c+e)*(-d+c+e));
	td2 = std::sqrt(val);
	val = (ta2+td2)/ (1-ta2*td2);
	val = (1.0-val*val)/(1+val*val);
	double f = b*b + c*c -2*b*c*val;
	if(f==0) {
		std::cout << "problem with edge : " << e0->index << std::endl;
	}

	// Now reassign half edges

	h0->vertex = v2;
	h1->flip = h9;
	h1->edge = e4;
	h1->vertex = v3;
	h2->flip = h6;
	h2->edge = e1;
	h2->vertex = v1;
	h3->vertex = v3;
	h4->flip = h7;
	h4->edge = e2;
	h4->vertex = v2;
	h5->flip = h8;
	h5->edge = e3;
	h5->vertex = v0;
	h6->flip = h2;
	h7->flip = h4;
	h8->flip = h5;
	h9->flip = h1;

	// reassign half edge for vertices

	v0->he = h5;
	v1->he = h2;
	v2->he = h0;
	v3->he = h3;

	// reassign half edge for edges

	e1 ->he = h2;
	e2 ->he = h4;
	e3 ->he = h5;
	e4 ->he = h1;

	e0->length = std::sqrt(f);

  }

  /* ===============================================================================================
   iDT: Build intrinsic Delaunay triangulation for current mesh

   Uses the algorithm described in:
	M. Fisher, B. Springborn, P. Schroeder, and A.I. Bobenko, "An algorithm for the construction
	of intrinsic Delaunay triangulations. Applications to Digital Geometry Processing"

   =============================================================================================== */

   void IDT::iDT(Mesh& mesh)
  {

	std::stack<EdgeIter> edge_stack;
	int *Mark = new int[mesh.edges.size()];

	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		Mark[e->index] = 1;
		VertexIter v1 = e->he->vertex;
		VertexIter v2 = e->he->flip->vertex;
		if(v1->NorthPole || v2->NorthPole) continue;
		if(v1->inNorthPoleVicinity && v2 -> inNorthPoleVicinity) continue;
		edge_stack.push(e);
	}

	int nflip = 0;
	EdgeIter e;
	std::vector<EdgeIter> list;
	while( ! edge_stack.empty() )
	{
		e = edge_stack.top();
		edge_stack.pop();
		list.clear();
		Mark[e->index] = 0;
		if(! checkDelaunay(e) && isFlippable(e)) {
			flipEdge(e);
			nflip++;
			HalfEdgeIter h = e->he;
			list.push_back(h->next->edge);
			list.push_back(h->next->next->edge);
			HalfEdgeIter h2 = h->flip;
			list.push_back(h2->next->edge);
			list.push_back(h2->next->next->edge);
			for(int i = 0; i < 4; i++) {
				EdgeIter ei = list[i];
				if(Mark[ei->index]==0) {
					Mark[ei->index] = 1;
					edge_stack.push(ei);
				}
			}
		}
	}

	delete [] Mark;

	int nerr = 0;
	for(EdgeIter e = mesh.edges.begin(); e < mesh.edges.end(); e++) {
		VertexIter v1 = e->he->vertex;
		VertexIter v2 = e->he->flip->vertex;
		if(v1->NorthPole || v2->NorthPole) continue;
		if(v1->inNorthPoleVicinity && v2 -> inNorthPoleVicinity) continue;
		if(!checkDelaunay(e) && isFlippable(e))  nerr++;
	}
//	std::cout << "Number of bad edges remaining: "<< nerr << std::endl;
//	std::cout << " " << std::endl;
  }

  /* ===============================================================================================
   iDT: check flip algorithm
   =============================================================================================== */

  void IDT::checkFlip(EdgeIter e0) 
  {

	// Collect current half edges in the two triangles incident on e0

	HalfEdgeIter h0 = e0->he;
	HalfEdgeIter h1 = h0->next;
	HalfEdgeIter h2 = h1->next;
	HalfEdgeIter h3 = h0->flip;
	HalfEdgeIter h4 = h3->next;
	HalfEdgeIter h5 = h4->next;
	HalfEdgeIter h6 = h1->flip;
	HalfEdgeIter h7 = h2->flip;
	HalfEdgeIter h8 = h4->flip;
	HalfEdgeIter h9 = h5->flip;

	// Collect current edges

	EdgeIter e1 = h1->edge;
	EdgeIter e2 = h2->edge;
	EdgeIter e3 = h4->edge;
	EdgeIter e4 = h5->edge;

	// Collect current vertex

	VertexIter v0 = h0->vertex;
	VertexIter v1 = h3->vertex;
	VertexIter v2 = h2->vertex;
	VertexIter v3 = h5->vertex;

	// Print all information:

	std::cout << " edge considered: " << e0->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 0: next      : " << h0->next->index << std::endl;
	std::cout << "he 0: prev      : " << h0->prev->index << std::endl;
	std::cout << "he 0: flip      : " << h0->flip->index << std::endl;
	std::cout << "he 0: edge      : " << h0->edge->index << std::endl;
	std::cout << "he 0: face      : " << h0->face->index << std::endl;
	std::cout << "he 0: vertex    : " << h0->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 1: next      : " << h1->next->index << std::endl;
	std::cout << "he 1: prev      : " << h1->prev->index << std::endl;
	std::cout << "he 1: flip      : " << h1->flip->index << std::endl;
	std::cout << "he 1: edge      : " << h1->edge->index << std::endl;
	std::cout << "he 1: face      : " << h1->face->index << std::endl;
	std::cout << "he 1: vertex    : " << h1->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 2: next      : " << h2->next->index << std::endl;
	std::cout << "he 2: prev      : " << h2->prev->index << std::endl;
	std::cout << "he 2: flip      : " << h2->flip->index << std::endl;
	std::cout << "he 2: edge      : " << h2->edge->index << std::endl;
	std::cout << "he 2: face      : " << h2->face->index << std::endl;
	std::cout << "he 2: vertex    : " << h2->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 3: next      : " << h3->next->index << std::endl;
	std::cout << "he 3: prev      : " << h3->prev->index << std::endl;
	std::cout << "he 3: flip      : " << h3->flip->index << std::endl;
	std::cout << "he 3: edge      : " << h3->edge->index << std::endl;
	std::cout << "he 3: face      : " << h3->face->index << std::endl;
	std::cout << "he 3: vertex    : " << h3->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 4: next      : " << h4->next->index << std::endl;
	std::cout << "he 4: prev      : " << h4->prev->index << std::endl;
	std::cout << "he 4: flip      : " << h4->flip->index << std::endl;
	std::cout << "he 4: edge      : " << h4->edge->index << std::endl;
	std::cout << "he 4: face      : " << h4->face->index << std::endl;
	std::cout << "he 4: vertex    : " << h4->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 5: next      : " << h5->next->index << std::endl;
	std::cout << "he 5: prev      : " << h5->prev->index << std::endl;
	std::cout << "he 5: flip      : " << h5->flip->index << std::endl;
	std::cout << "he 5: edge      : " << h5->edge->index << std::endl;
	std::cout << "he 5: face      : " << h5->face->index << std::endl;
	std::cout << "he 5: vertex    : " << h5->vertex->index << std::endl;

	std::cout << " " << std::endl;

	std::cout << "Vertex v0       : " << v0->index << std::endl;
	std::cout << "Vertex v0 he    : " << v0->he->index << std::endl;
	std::cout << "Vertex v1       : " << v1->index << std::endl;
	std::cout << "Vertex v1 he    : " << v1->he->index << std::endl;
	std::cout << "Vertex v2       : " << v2->index << std::endl;
	std::cout << "Vertex v2 he    : " << v2->he->index << std::endl;
	std::cout << "Vertex v3       : " << v3->index << std::endl;
	std::cout << "Vertex v3 he    : " << v3->he->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "edge e0         : " << e0->index << std::endl;
	std::cout << "edge e0 he      : " << e0->he->index << std::endl;
	std::cout << "edge e1         : " << e1->index << std::endl;
	std::cout << "edge e1 he      : " << e1->he->index << std::endl;
	std::cout << "edge e2         : " << e2->index << std::endl;
	std::cout << "edge e2 he      : " << e2->he->index << std::endl;
	std::cout << "edge e3         : " << e3->index << std::endl;
	std::cout << "edge e3 he      : " << e3->he->index << std::endl;
	std::cout << "edge e4         : " << e4->index << std::endl;
	std::cout << "edge e4 he      : " << e4->he->index << std::endl;
	std::cout << " " << std::endl;

	std::cout <<"face f0          : " << h0->face->index << std::endl;
	std::cout <<"face f0 he       : " << h0->face->he->index << std::endl;
	std::cout <<"face f1          : " << h3->face->index << std::endl;
	std::cout <<"face f1 he       : " << h3->face->he->index << std::endl;
	std::cout << " " << std::endl;

	// Now reassign half edges

	h0->vertex = v2;
	h1->flip = h9;
	h1->edge = e4;
	h1->vertex = v3;
	h2->flip = h6;
	h2->edge = e1;
	h2->vertex = v1;
	h3->vertex = v3;
	h4->flip = h7;
	h4->edge = e2;
	h4->vertex = v2;
	h5->flip = h8;
	h5->edge = e3;
	h5->vertex = v0;
	h6->flip = h2;
	h7->flip = h4;
	h8->flip = h5;
	h9->flip = h1;

	// reassign half edge for vertices

	v0->he = h5;
	v1->he = h2;
	v2->he = h0;
	v3->he = h1;

	// reassign half edge for edges

	e1 ->he = h2;
	e2 ->he = h4;
	e3 ->he = h5;
	e4 ->he = h1;

	std::cout << "he 0: next      : " << h0->next->index << std::endl;
	std::cout << "he 0: prev      : " << h0->prev->index << std::endl;
	std::cout << "he 0: flip      : " << h0->flip->index << std::endl;
	std::cout << "he 0: edge      : " << h0->edge->index << std::endl;
	std::cout << "he 0: face      : " << h0->face->index << std::endl;
	std::cout << "he 0: vertex    : " << h0->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 1: next      : " << h1->next->index << std::endl;
	std::cout << "he 1: prev      : " << h1->prev->index << std::endl;
	std::cout << "he 1: flip      : " << h1->flip->index << std::endl;
	std::cout << "he 1: edge      : " << h1->edge->index << std::endl;
	std::cout << "he 1: face      : " << h1->face->index << std::endl;
	std::cout << "he 1: vertex    : " << h1->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 2: next      : " << h2->next->index << std::endl;
	std::cout << "he 2: prev      : " << h2->prev->index << std::endl;
	std::cout << "he 2: flip      : " << h2->flip->index << std::endl;
	std::cout << "he 2: edge      : " << h2->edge->index << std::endl;
	std::cout << "he 2: face      : " << h2->face->index << std::endl;
	std::cout << "he 2: vertex    : " << h2->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 3: next      : " << h3->next->index << std::endl;
	std::cout << "he 3: prev      : " << h3->prev->index << std::endl;
	std::cout << "he 3: flip      : " << h3->flip->index << std::endl;
	std::cout << "he 3: edge      : " << h3->edge->index << std::endl;
	std::cout << "he 3: face      : " << h3->face->index << std::endl;
	std::cout << "he 3: vertex    : " << h3->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 4: next      : " << h4->next->index << std::endl;
	std::cout << "he 4: prev      : " << h4->prev->index << std::endl;
	std::cout << "he 4: flip      : " << h4->flip->index << std::endl;
	std::cout << "he 4: edge      : " << h4->edge->index << std::endl;
	std::cout << "he 4: face      : " << h4->face->index << std::endl;
	std::cout << "he 4: vertex    : " << h4->vertex->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "he 5: next      : " << h5->next->index << std::endl;
	std::cout << "he 5: prev      : " << h5->prev->index << std::endl;
	std::cout << "he 5: flip      : " << h5->flip->index << std::endl;
	std::cout << "he 5: edge      : " << h5->edge->index << std::endl;
	std::cout << "he 5: face      : " << h5->face->index << std::endl;
	std::cout << "he 5: vertex    : " << h5->vertex->index << std::endl;

	std::cout << " " << std::endl;

	std::cout << "Vertex v0       : " << v0->index << std::endl;
	std::cout << "Vertex v0 he    : " << v0->he->index << std::endl;
	std::cout << "Vertex v1       : " << v1->index << std::endl;
	std::cout << "Vertex v1 he    : " << v1->he->index << std::endl;
	std::cout << "Vertex v2       : " << v2->index << std::endl;
	std::cout << "Vertex v2 he    : " << v2->he->index << std::endl;
	std::cout << "Vertex v3       : " << v3->index << std::endl;
	std::cout << "Vertex v3 he    : " << v3->he->index << std::endl;
	std::cout << " " << std::endl;

	std::cout << "edge e0         : " << e0->index << std::endl;
	std::cout << "edge e0 he      : " << e0->he->index << std::endl;
	std::cout << "edge e1         : " << e1->index << std::endl;
	std::cout << "edge e1 he      : " << e1->he->index << std::endl;
	std::cout << "edge e2         : " << e2->index << std::endl;
	std::cout << "edge e2 he      : " << e2->he->index << std::endl;
	std::cout << "edge e3         : " << e3->index << std::endl;
	std::cout << "edge e3 he      : " << e3->he->index << std::endl;
	std::cout << "edge e4         : " << e4->index << std::endl;
	std::cout << "edge e4 he      : " << e4->he->index << std::endl;
	std::cout << " " << std::endl;

	std::cout <<"face f0          : " << h0->face->index << std::endl;
	std::cout <<"face f0 he       : " << h0->face->he->index << std::endl;
	std::cout <<"face f1          : " << h3->face->index << std::endl;
	std::cout <<"face f1 he       : " << h3->face->he->index << std::endl;
	std::cout << " " << std::endl;

  }

