/* ===============================================================================================
   Viewer: classes and functions used for OpenGL displays of meshes and their images onto
	   the plane or onto the sphere

   Author:  Patrice Koehl Date:    5/10/2019 Version: 1
   =============================================================================================== */

#pragma once

#define GL_SILENCE_DEPRECATION

/* ===============================================================================================
   Local, and system includes
   =============================================================================================== */

#include <iomanip>
#include <limits>
#include <list>

#include "Camera.h"
#include "RenderMesh.h"
#include "nanogui/nanogui.h"
#include "MeshIO.h"
#include "ShadersSource.h"
#include "Tutte.h"
#include "Normalize.h"
#include "Distortion.h"
#include "ConformalError.h"
#include "RicciFlow.h"
#include "YamabeFlow.h"
#include "YamabeFlowIDT.h"
#include "cMCF.h"
#include "BeltramiFlow.h"
#include "WillmoreFlow.h"

/* ===============================================================================================
   Some defines for the camera (mostly colors)
   =============================================================================================== */

#define EYE 3.5

const glm::vec3 white(1.0, 1.0, 1.0);
const glm::vec3 skyblue(0.643, 0.647, 0.815);
const glm::vec3 red(1.0, 0.0, 0.0);
const glm::vec3 orange(1.0, 0.55, 0.0);
const glm::vec3 yellow(1.0, 0.9, 0.1);

/* ===============================================================================================
   The different classes
	-PlotType: 	characteristics of the mesh representation
	-BoundaryType:  information about boundaries of the mesh
	-ViewPane: 	class for 2D representation
	-VertexHandle:	to associate objects to vertices
	-ModelState:	class for the different models
	-Viewer:	main class for Viewer functions
   =============================================================================================== */

enum class PlotType {
	constant,
	shaded,
	normals,
	quasiConformalError,
	angularError,
	areaScaling
};

class ViewPane {
public:
	ViewPane() {}
	ViewPane(double l, double b, double w, double h): left(l), bottom(b), width(w), height(h) {}
	double left, bottom, width, height;
	double aspect() { return width/height; }
};

class VertexHandle {
public:
	VertexIter vertex;
	std::shared_ptr<RenderMesh> sphere3D;
	std::shared_ptr<RenderMesh> sphereUV;

	bool operator==(const VertexHandle& handle) const {
		return vertex == handle.vertex;
	}
};

class ModelState {
public:
	ModelState():
	surfaceIsClosed(false),
	mappedToSphere(false) {}

	std::vector<std::shared_ptr<RenderMesh>> renderMeshes;

	bool surfaceIsClosed;
	bool mappedToSphere;

	VertexIter selectedVertexHandle;
	std::list<VertexHandle> handles;

};

class Viewer {
public:
	// displays the viewer until the program ends
	static void init(Mesh& model_);

private:
	// initialization
	static void initGLFW();
	static void initGui();
	static void initGL();
	static void initShaders();
	static void initCameras();
	static void initTransforms();
	static void initLights();
	static void initModel();
	static void initRenderMeshes();

	// clear
	static void clearScene();
	static void clear();
	static void clearViewport(const ViewPane& pane);

	// update
	static void updatePlotLabel();
	static void updateUniforms(int index, int width, int height);
	static void updateRenderMeshes();
	static void updateViewPanes();
	static void updateCameraZoom(int idx_cam);
	static void update(bool updateZoom = true);
	static void updateSelectedMesh();

	static void updateSphere(std::shared_ptr<RenderMesh> sphere,
							 const glm::vec3& center, const glm::vec3& color);
	static void addSphere(std::shared_ptr<RenderMesh>& sphere,
						  const glm::vec3& center, const glm::vec3& color);
	static void addSphere(std::vector<std::shared_ptr<RenderMesh>>& spheres,
						  const glm::vec3& center, const glm::vec3& color);

	static void updateVertexHandle(VertexIter v);
	static void updateVertexHandlesUVs();

	// add and remove vertex handles
	static void addVertexHandle(VertexIter v, double newAngle);
	static void removeVertexHandle(VertexIter v);
	static void removeVertexHandles();
	static std::vector<VertexIter> getHandleVertices();

	// normalized device coordinates
	static Vector ndc(int x, int y, int width, int height);

	// glut callbacks
	static void cursorPosCallback(GLFWwindow *w, double x, double y);
	static void mouseButtonCallback(GLFWwindow *w, int button, int action, int modifiers);
	static void keyCallback(GLFWwindow *w, int key, int scancode, int action, int mods);
	static void scrollCallback(GLFWwindow *w, double x, double y);
	static void charCallback(GLFWwindow *w, unsigned int codepoint);
	static void framebufferSizeCallback(GLFWwindow *w, int width, int height);

	// display
	static void drawRenderMeshes(const ViewPane& pane, int index);
	static void drawScene();
	static void draw();

	// report error
	static void reportError(const std::string& error);

	// members
	static GLFWwindow *window;
	static nanogui::Screen *gui;
	static nanogui::Label *plotLabel;
	static nanogui::Label *angleControlLabel;
	static nanogui::ComboBox *paramComboBox;
	static nanogui::ComboBox *plotComboBox;
	static nanogui::TextBox *angleTextBox;
	static nanogui::Label *instructionText;
	static nanogui::CheckBox *wireframeCheckBox;
	static nanogui::CheckBox *applyNormCheckBox;
	static nanogui::CheckBox *applyQuasiCheckBox;
	static nanogui::CheckBox *applyRefMCFCheckBox;
	static nanogui::CheckBox *showConeCheckBox;
	static nanogui::CheckBox *exportNormalizedUVsCheckBox;

	static int windowWidth;
	static int windowHeight;
	static int framebufferWidth;
	static int framebufferHeight;

	static ViewPane surfacePane;
	static ViewPane uvPane;
	static ViewPane fullPane;

	static GLuint transformUbo;
	static GLuint lightUbo;
	static float patternSize;

	static std::vector<std::shared_ptr<Camera>> cameras;

	static Shader modelShader;
	static Shader flatShader;
	static Shader wireframeShader;
	static bool showWireframe;
	static bool applyNorm;
	static bool applyQuasi;
	static bool applyRefMCF;
	static bool showCone;

	static bool SUCCESS;

	static Vector origUp;
	static Vector mouse, origMouse;
	static int clickX, unclickX;
	static int clickY, unclickY;
	static bool mouseDown;

	static bool pickingEnabled;
	static bool shiftClick;
	static bool ctrlClick;
	static bool altClick;

	static std::string objPath;
	static Mesh *mesh;
	static Mesh model;
	static ModelState *state;
	static int selectedMesh;
	static bool loadedModel;

	static PlotType plotType;
	static bool colorsNeedUpdate;

	static Tutte tutte;
	static Ricci ricci;
	static Yamabe yamabe;
	static YamabeIDT yamabeIDT;
	static Beltrami beltrami;
	static cMCF mcf;
	static Willmore willmore;

};

/* ===============================================================================================
   Viewer function: 
	init:	maintain display until program ends
   =============================================================================================== */

void Viewer::init(Mesh& model_)
{
	mesh = new Mesh(model_);
	model = *mesh;
	state = new ModelState();

	initGLFW();
	initGui();
	initGL();
	initShaders();
	initCameras();
	initTransforms();
	initLights();
	initModel();
	initRenderMeshes();
	updateCameraZoom(0);

	// render and poll events
	while (!glfwWindowShouldClose(window)) {
		glfwPollEvents();
		draw();
	}

	// clear and terminate
	clear();
	glfwTerminate();
}

/* ===============================================================================================
   Viewer function: 
	errorCallback:	prints out information on error as standard error
   =============================================================================================== */

void errorCallback(int error, const char* description)
{
	std::cerr << description << std::endl;
}

/* ===============================================================================================
   Viewer function: 
	initGLFW:	as stated in name, initiatizes GLFW
   =============================================================================================== */

void Viewer::initGLFW()
{
	glfwSetErrorCallback(errorCallback);
	if (!glfwInit()) {
		std::cerr << "Unable to initalize glfw" << std::endl;
		exit(EXIT_FAILURE);
	}

	glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 4);
	glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 1);
	glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
	glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);

	glfwWindowHint(GLFW_SAMPLES, 0);
	glfwWindowHint(GLFW_RED_BITS, 8);
	glfwWindowHint(GLFW_GREEN_BITS, 8);
	glfwWindowHint(GLFW_BLUE_BITS, 8);
	glfwWindowHint(GLFW_ALPHA_BITS, 8);
	glfwWindowHint(GLFW_STENCIL_BITS, 8);
	glfwWindowHint(GLFW_DEPTH_BITS, 24);
	glfwWindowHint(GLFW_RESIZABLE, GL_TRUE);

	// initialize window
	window = glfwCreateWindow(windowWidth, windowHeight, "Conformal Mappings to the Sphere", NULL, NULL);
	if (!window) {
		std::cerr << "Unable to initalize glfw window" << std::endl;
		glfwTerminate();
		exit(EXIT_FAILURE);
	}

	glfwMakeContextCurrent(window);
#if defined(NANOGUI_GLAD)
	if (!gladLoadGLLoader((GLADloadproc) glfwGetProcAddress)) {
		std::cerr << "Unable to initalize glad" << std::endl;
		exit(EXIT_FAILURE);
	}
#endif
	glfwSwapInterval(1);

	// specify callbacks
	glfwSetCursorPosCallback(window, Viewer::cursorPosCallback);
	glfwSetMouseButtonCallback(window, Viewer::mouseButtonCallback);
	glfwSetKeyCallback(window, Viewer::keyCallback);
	glfwSetScrollCallback(window, Viewer::scrollCallback);
	glfwSetCharCallback(window, Viewer::charCallback);
	glfwSetFramebufferSizeCallback(window, Viewer::framebufferSizeCallback);
}

/* ===============================================================================================
   Viewer function: 
	initGui:	as stated in name, initiatizes GUI for application
   =============================================================================================== */

void Viewer::initGui()
{

	gui->initialize(window, true);
	nanogui::Window *nanoguiWindow = new nanogui::Window(gui, "");
	nanoguiWindow->setLayout(new nanogui::GroupLayout());

	Eigen::Vector3f highlightColor = Eigen::Vector3f(yellow.x, yellow.y, yellow.z)*0.9;

	/* =======================================================================================
	Input button: Load mesh (either in OBJ, or OFF format)
	========================================================================================== */

	new nanogui::Label(nanoguiWindow, "Input / Output", "sans-bold");

	nanogui::Button *loadMeshButton = new nanogui::Button(nanoguiWindow, "Load Mesh");
	loadMeshButton->setFixedHeight(25);
	loadMeshButton->setCallback([] {
		std::vector<std::pair<std::string, std::string> > filetype;
		std::pair <std::string, std::string> file1;
		file1 = std::make_pair(std::string("obj"),std::string("OBJ file")); 
		filetype.push_back(file1);
		file1 = std::make_pair(std::string("off"),std::string("OFF file")); 
		filetype.push_back(file1);
		objPath = nanogui::file_dialog(filetype, false);
		if (objPath.empty()) return;
		try {
			clearScene();
			std::string error;
			std::size_t found = objPath.find("obj");
			if(found !=std::string::npos) {
				MeshIO::readOBJ(objPath, *mesh, error);
			} else {
				found = objPath.find("off");
				if(found !=std::string::npos) {
					MeshIO::readOFF(objPath, model, error);
					mesh = &model;
				} else {
					reportError("Unable to load file: " + objPath);
				}
			}
			applyNorm = false;
			applyQuasi = false;
			applyRefMCF = false;
			applyNormCheckBox->setChecked(applyNorm);
			applyQuasiCheckBox->setChecked(applyQuasi);
			applyRefMCFCheckBox->setChecked(applyRefMCF);
			initModel();
			initRenderMeshes();
			updateCameraZoom(0);

		} catch (...) {
			reportError("Unable to load file: " + objPath);
		}
	});

	/* =======================================================================================
	Output button: Export mesh (either in OBJ, or OFF format)
	========================================================================================== */

	nanogui::Button *exportMeshButton = new nanogui::Button(nanoguiWindow, "Export Mesh");
	exportMeshButton->setFixedHeight(25);
	exportMeshButton->setCallback([] {
		std::string file = nanogui::file_dialog({{"obj", "Wavefront OBJ file"},{"off", "OFF file"}}, true);
		if (file.empty()) return;
		try {
			if (!MeshIO::write(file, *mesh)) {
				reportError("Unable to write file: " + file);
			}

		} catch (...) {
			reportError("Unable to write file: " + file);
		}
	});

	/* =======================================================================================
	Checkbox: Toggle on/off wireframe on the mesh
	========================================================================================== */

	new nanogui::Label(nanoguiWindow, "Display controls", "sans-bold");

	wireframeCheckBox = new nanogui::CheckBox(nanoguiWindow, "Show Wireframe");
	wireframeCheckBox->setCallback([](int box) { showWireframe = !showWireframe; });
	wireframeCheckBox->setChecked(showWireframe);
	wireframeCheckBox->setFixedHeight(25);

	/* =======================================================================================
	Checkbox: Toggle on/off visualization of cone points on the mesh
	========================================================================================== */

	showConeCheckBox = new nanogui::CheckBox(nanoguiWindow, "Show cone points");
	showConeCheckBox->setCallback([](int box) { showCone = !showCone; });
	showConeCheckBox->setChecked(showCone);
	showConeCheckBox->setFixedHeight(25);

	/* =======================================================================================
	Combobox: Define representation of the meshes
	========================================================================================== */

//	plotLabel = new nanogui::Label(nanoguiWindow, "Plot", "sans-bold");
	plotComboBox = new nanogui::ComboBox(nanoguiWindow, {"      Constant      ",
							     "       Shaded       ",
							     "       Normal       ",
							     "Conformal Distortion",
							     "   Area Distortion  "});

	plotComboBox->setSelectedIndex(static_cast<int>(plotType));
	plotComboBox->setFixedHeight(25);
	plotComboBox->setCallback([](int box) {
		switch (box) {
			case 0:
				plotType = PlotType::constant;
				break;
			case 1:
				plotType = PlotType::shaded;
				break;
			case 2:
				plotType = PlotType::normals;
				break;
			case 3:
				plotType = PlotType::quasiConformalError;
				break;
			case 4:
				plotType = PlotType::areaScaling;
				break;
		}

		if (loadedModel) {
			colorsNeedUpdate = true;
			update(false);
		}
	});

	/* =======================================================================================
	Section: parametrization
	========================================================================================== */

	VertexIter pole;
	new nanogui::Label(nanoguiWindow, "Parameterization", "sans-bold");
	paramComboBox = new nanogui::ComboBox(nanoguiWindow, {  " None           ",
								" Tutte: Uniform ",
								" Tutte: Cotan ",
								" Tutte: Mean Value ",
								" Ricci flow   ",
								" Yamabe flow  ",
								" Yamabe flow (IDT) ",
								" cMCF flow    ",
								" Willmore flow",
								});
	paramComboBox->setSelectedIndex(0);
	paramComboBox->setFixedHeight(25);
	paramComboBox->setCallback([&pole](int box) {

		if(loadedModel) {
			if(box == 1) {
				int v_type = 0;
				int tutte_type = 1;
				VertexIter pole2 = tutte.planarTutte(*mesh, v_type, tutte_type, &SUCCESS);
				state->mappedToSphere = true;
				updateVertexHandle(pole2);
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				cameras[1]->reset();
			} 
			if(box == 2) {
				int v_type = 0;
				int tutte_type = 2;
				VertexIter pole2 = tutte.planarTutte(*mesh, v_type, tutte_type, &SUCCESS);
				state->mappedToSphere = true;
				updateVertexHandle(pole2);
				applyNorm = false;
				applyRefMCF = false;
				applyQuasi = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				cameras[1]->reset();
			} 
			if(box == 3) {
				int v_type = 0;
				int tutte_type = 3;
				VertexIter pole2 = tutte.planarTutte(*mesh, v_type, tutte_type, &SUCCESS);
				if(SUCCESS) state->mappedToSphere = true;
				updateVertexHandle(pole2);
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				cameras[1]->reset();
			} 
			if(box == 4) {
				int v_type = 0;
				nR = 0;
				VertexIter pole2 = ricci.euclideanRicci(*mesh, v_type, &SUCCESS);
				if(SUCCESS) state->mappedToSphere = true;
				updateVertexHandle(pole2);
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				cameras[1]->reset();
			} 
			if(box == 5) {
				int v_type = 0;
				nY = 0;
				VertexIter pole2 = yamabe.yamabeFlow(*mesh, v_type, &SUCCESS);
				if(SUCCESS) state->mappedToSphere = true;
				updateVertexHandle(pole2);
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				cameras[1]->reset();
			} 
			if(box == 6) {
				int v_type = 0;
				nYD = 0;
				VertexIter pole2 = yamabeIDT.yamabeFlowIDT(*mesh, v_type, &SUCCESS);
				if(SUCCESS) state->mappedToSphere = true;
				updateVertexHandle(pole2);
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				cameras[1]->reset();
			} 
			if(box == 7) {
				state->mappedToSphere = true;
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				int niter=500;
				double TOL = 1.e-6;
				double dt=0.005;
				double S, Q;
				double Q_old = 0;
				Vector r;
				ntype = 0;
				int f_type = 0;
				int type = 0;
				mcf.initFlow(*mesh, f_type, dt);
				cameras[1]->reset();
				update(true);
				for(int i = 0; i < niter; i++)
				{
					r = mcf.solveOneStep(*mesh, i, dt, type);
					S = r[0];
					Q = r[1];
					cameras[1]->reset();
					update(true);
					if(std::abs(Q-Q_old) < TOL) break;
					Q_old = Q;
				}
				mcf.stopFlow(*mesh);
				cameras[1]->reset();
				update(true);
			}
			if(box == 8) {
				state->mappedToSphere = true;
				applyNorm = false;
				applyQuasi = false;
				applyRefMCF = false;
				applyNormCheckBox->setChecked(applyNorm);
				applyQuasiCheckBox->setChecked(applyQuasi);
				applyRefMCFCheckBox->setChecked(applyRefMCF);
				plotType = PlotType::quasiConformalError;
				plotComboBox->setSelectedIndex(static_cast<int>(plotType));
				int niter=50;
				double TOL1 = 1.e-6, TOL2 = 1.e-3;
				double dt=0.2;
				double S, Q;
				double Q_old = 0;
				int btype = 0;
				nE = 0;
				nLS = 0;
				nS = 0;
				Vector r;
				willmore.initFlow(*mesh, dt);
				cameras[1]->reset();
				update(true);

				for(int i = 0; i < niter; i++)
				{
					r = willmore.solveOneStep(*mesh, i, dt, btype);
					S = r[0];
					Q = r[1];
					cameras[1]->reset();
					update(true);
					if(std::abs(Q-Q_old) < TOL1 || (std::abs(S-1.0) < TOL2) ) break;
					Q_old = Q;
				}
				willmore.stopFlow(*mesh);
				cameras[1]->reset();
				update(true);

			}
		}
		update(true);
		return pole;
	});

	/* =======================================================================================
	Checkbox: postprocessing on the mesh
	========================================================================================== */

	new nanogui::Label(nanoguiWindow, "Post processing", "sans-bold");

	applyNormCheckBox = new nanogui::CheckBox(nanoguiWindow, "Normalize mesh");
	applyNormCheckBox->setCallback([](int box) { 
		applyNorm = !applyNorm; 
		if(applyNorm) {
			Normalize::normalize(*mesh);
			ConformalError::checkConformal(*mesh);
			updateVertexHandlesUVs();
			cameras[1]->reset();
			update(true);
		}
	});
	applyNormCheckBox->setChecked(applyNorm);
	applyNormCheckBox->setFixedHeight(25);

	applyQuasiCheckBox = new nanogui::CheckBox(nanoguiWindow, "Quasi conformal flow");
	applyQuasiCheckBox->setCallback([](int box) { 
		applyQuasi = !applyQuasi; 
		if(applyQuasi) {
			beltrami.beltramiFlow(*mesh);
			ConformalError::checkConformal(*mesh);
			updateVertexHandlesUVs();
			cameras[1]->reset();
			update(true);
		}
	});
	applyQuasiCheckBox->setChecked(applyQuasi);
	applyQuasiCheckBox->setFixedHeight(25);

	applyRefMCFCheckBox = new nanogui::CheckBox(nanoguiWindow, "cMCF refinement");
	applyRefMCFCheckBox->setCallback([](int box) { 
		applyRefMCF = !applyRefMCF; 
		if(applyRefMCF) {
			int niter=100;
			double TOL = 1.e-6;
			double dt=0.005;
			double S, Q;
			double Q_old = 0.;
			Vector r;
			int f_type = 1;
			int type = 0;
			ntype = 0;
			mcf.initFlow(*mesh, f_type, dt);
			cameras[1]->reset();
			update(true);
			for(int i = 0; i < niter; i++)
			{
				r = mcf.solveOneStep(*mesh, i, dt, type);
				S = r[0];
				Q = r[1];
				cameras[1]->reset();
				update(true);
				if(std::abs(Q-Q_old) < TOL) break;
				Q_old = Q;
			}
			mcf.stopFlow(*mesh);
			ConformalError::checkConformal(*mesh);
			updateVertexHandlesUVs();
			cameras[1]->reset();
			update(true);
		}
	});
	applyRefMCFCheckBox->setChecked(applyRefMCF);
	applyRefMCFCheckBox->setFixedHeight(25);

	/* =======================================================================================
	Exit button: exit nanogui
	========================================================================================== */

	new nanogui::Label(nanoguiWindow, "");
	nanogui::Button *resetButton = new nanogui::Button(nanoguiWindow, "Exit");
	resetButton->setFixedHeight(25);
	resetButton->setCallback([] {
		clear();
	});

	// help text
	instructionText = new nanogui::Label(nanoguiWindow, "", "sans", 15);
	instructionText->setFixedSize(Eigen::Vector2i(190, 60));
	instructionText->setColor(highlightColor);

	gui->setVisible(true);
	gui->performLayout();
}

void Viewer::initGL()
{
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
}

void Viewer::initShaders()
{
	modelShader.load(modelVert, modelFrag);
	flatShader.load(flatVert, flatFrag);
	wireframeShader.load(wireframeVert, wireframeFrag);
}

void Viewer::initCameras()
{
	for (int i = 0; i < 2; i++) cameras.push_back(std::shared_ptr<Camera>(new Camera(EYE)));
}

void Viewer::initTransforms()
{
	// get transform indices
	GLuint modelIndex = glGetUniformBlockIndex(modelShader.program, "Transform");
	GLuint flatIndex = glGetUniformBlockIndex(flatShader.program, "Transform");
	GLuint wireframeIndex = glGetUniformBlockIndex(wireframeShader.program, "Transform");

	// bind
	glUniformBlockBinding(modelShader.program, modelIndex, 1);
	glUniformBlockBinding(flatShader.program, flatIndex, 1);
	glUniformBlockBinding(wireframeShader.program, wireframeIndex, 1);

	// create transform buffer
	glGenBuffers(1, &transformUbo);
	glBindBuffer(GL_UNIFORM_BUFFER, transformUbo);
	glBindBufferBase(GL_UNIFORM_BUFFER, 1, transformUbo);
	glBufferData(GL_UNIFORM_BUFFER, 4*sizeof(glm::mat4), NULL, GL_DYNAMIC_DRAW);

	// set the model matrix to the identity (will never change)
	glBufferSubData(GL_UNIFORM_BUFFER, 3*sizeof(glm::mat4), sizeof(glm::mat4), glm::value_ptr(glm::mat4()));

	glBindBuffer(GL_UNIFORM_BUFFER, 0);

	// eye
	modelShader.use();
	glUniform3f(glGetUniformLocation(modelShader.program, "eye"), 0.0, 0.0, -10.0);
}

void Viewer::initLights()
{
	// get light index
	GLuint modelIndex = glGetUniformBlockIndex(modelShader.program, "Light");

	// bind
	glUniformBlockBinding(modelShader.program, modelIndex, 2);

	// add light data
	glGenBuffers(1, &lightUbo);
	glBindBuffer(GL_UNIFORM_BUFFER, lightUbo);
	glBindBufferBase(GL_UNIFORM_BUFFER, 2, lightUbo);
	glBufferData(GL_UNIFORM_BUFFER, 2*sizeof(glm::vec4), NULL, GL_STATIC_DRAW); // std140 alignment
	// light.position
	glBufferSubData(GL_UNIFORM_BUFFER, 0, sizeof(glm::vec4), glm::value_ptr(glm::vec3(0.0, 2.0, -10.0)));
	// light.color
	glBufferSubData(GL_UNIFORM_BUFFER, sizeof(glm::vec4), sizeof(glm::vec4), glm::value_ptr(glm::vec3(1.0, 1.0, 1.0)));
	glBindBuffer(GL_UNIFORM_BUFFER, 0);
}

void Viewer::initModel()
{
	loadedModel = true;
	state->mappedToSphere = false;
	if (mesh->eulerCharacteristic() == 2) {
		state->surfaceIsClosed = true;
	}
	if (state->handles.size() > 0) {
		for (VertexHandle handle: state->handles) {
			const Vector& uv = handle.vertex->position2;
			const Vector& p = handle.vertex->position;
			addSphere(handle.sphere3D, glm::vec3(p.x, p.y, p.z), yellow);
			addSphere(handle.sphereUV, glm::vec3(uv.x, uv.y, uv.z), yellow);
		}
	}
}

glm::vec3 elementColor(unsigned int pickedId)
{
	return glm::vec3(((pickedId & 0x000000FF) >>  0)/255.0,
					 ((pickedId & 0x0000FF00) >>  8)/255.0,
					 ((pickedId & 0x00FF0000) >> 16)/255.0);
}

void Viewer::initRenderMeshes()
{
	if (loadedModel) {
		int nVertices = 0;

		state->renderMeshes.clear();

		for (int j = 0; j < 2; j++) {
			state->renderMeshes.push_back(std::shared_ptr<RenderMesh>(new RenderMesh()));
		}


		// initialize attributes for both rendermeshes
		int N = 3*mesh->faces.size();

		std::shared_ptr<Buffer> uvs(new Buffer(N));
		std::shared_ptr<Buffer> normals(new Buffer(N));
		std::shared_ptr<Buffer> barycenters(new Buffer(N));

		state->renderMeshes[0]->positions = std::shared_ptr<Buffer>(new Buffer(N));
		state->renderMeshes[0]->uvs = uvs;
		state->renderMeshes[0]->normals = normals;
		state->renderMeshes[0]->colors = std::shared_ptr<Buffer>(new Buffer(N));
		state->renderMeshes[0]->barycenters = barycenters;
		state->renderMeshes[0]->pickPositions = std::shared_ptr<Buffer>(new Buffer(6*N));
		state->renderMeshes[0]->pickColors = std::shared_ptr<Buffer>(new Buffer(6*N));
		state->renderMeshes[1]->positions = std::shared_ptr<Buffer>(new Buffer(N));
		state->renderMeshes[1]->uvs = uvs;
		state->renderMeshes[1]->normals = normals;
		state->renderMeshes[1]->colors = std::shared_ptr<Buffer>(new Buffer(N));
		state->renderMeshes[1]->barycenters = barycenters;

		std::vector<Vector> barycenter = {Vector(1.0, 0.0, 0.0), Vector(0.0, 1.0, 0.0), Vector(0.0, 0.0, 1.0)};
		for (FaceCIter f = mesh->faces.begin(); f != mesh->faces.end(); f++) {
			int index = 3*f->index;

			// set positions, normals and colors for the 1st rendermesh
			// and barycenters for both rendermeshes
			int j = 0;
			HalfEdgeCIter h = f->he;
			do {
				VertexCIter v = h->vertex;
				Vector normal = v->normal();
				for (int k = 0; k < 3; k++) {
					state->renderMeshes[0]->positions->buffer[index + j][k] = v->position[k];
					state->renderMeshes[1]->positions->buffer[index + j][k] = v->position2[k];
					state->renderMeshes[0]->colors->buffer[index + j][k] = white[k];
					state->renderMeshes[1]->colors->buffer[index + j][k] = white[k];
					normals->buffer[index + j][k] = normal[k];
					barycenters->buffer[index + j][k] = barycenter[j][k];
				}

				j++;
				h = h->next;
			} while (h != f->he);
		}

		uvs->update();
		normals->update();
		barycenters->update();
		state->renderMeshes[0]->update({POSITION, COLOR, PICK_POSITION, PICK_COLOR});
		state->renderMeshes[1]->update({COLOR});

		nVertices += (int)mesh->vertices.size();
	}
}

void Viewer::clearScene()
{
	removeVertexHandles();
	mesh = NULL;
	state = NULL;
	state = new ModelState();
	selectedMesh = 0;
	loadedModel = false;
	colorsNeedUpdate = true;
	cameras[1]->reset();
	showWireframe = false;
	applyNorm = false;
	applyQuasi = false;
	showCone = false;
	plotType = PlotType::shaded;
	plotComboBox->setSelectedIndex(static_cast<int>(plotType));
	paramComboBox->setSelectedIndex(0);
	wireframeCheckBox->setChecked(showWireframe);
	applyNormCheckBox->setChecked(applyNorm);
	applyQuasiCheckBox->setChecked(applyQuasi);
	showConeCheckBox->setChecked(showCone);
}

void Viewer::clear()
{
	clearScene();
	glDeleteBuffers(1, &transformUbo);
	glDeleteBuffers(1, &lightUbo);

	exit(0);
}

void Viewer::clearViewport(const ViewPane& pane)
{
	// set viewport
	glViewport(pane.left, pane.bottom, pane.width, pane.height);
	glScissor(pane.left, pane.bottom, pane.width, pane.height);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_SCISSOR_TEST);

	// clear scene
//	glClearColor(1.0, 1.0, 1.0, 1.0);
//	glClearColor(0.312, 0.39, 0.546, 0.1);
//	glClearColor(0.4, 0.5, 0.7, 0.05);
	glClearColor(0.64, 0.74, 0.93, 0.1);
	glClearDepth(1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDepthFunc(GL_LEQUAL);
}

void Viewer::updatePlotLabel()
{
	if (plotType == PlotType::constant || plotType == PlotType::shaded || plotType == PlotType::normals) {
		plotLabel->setCaption("Plot");

	} else if (!state->surfaceIsClosed || state->mappedToSphere) {
		if (plotType == PlotType::quasiConformalError) {
			std::stringstream label;
			Vector error = Distortion::computeQuasiConformalError(*mesh);
			label << std::setprecision(4) << "Mean:" << error[2] << "   Max:" << error[1];
			plotLabel->setCaption("Plot     " + label.str());

		} else if (plotType == PlotType::areaScaling) {
			std::stringstream label;
			Vector error = Distortion::computeAreaScaling(*mesh);
			label << std::setprecision(2) << "Min:" << exp(error[0]) << "   Max:" << exp(error[1]);
			plotLabel->setCaption("Plot     " + label.str());
		}
	} else {
		plotLabel->setCaption("Plot");
	}
}

void Viewer::updateUniforms(int index, int width, int height)
{
	// set camera transformations
	glm::mat4 projection = cameras[index]->projectionMatrix(width, height);
	glm::mat4 view = cameras[index]->viewMatrix();
	glm::mat4 view3D = cameras[0]->viewMatrix();

	glBindBuffer(GL_UNIFORM_BUFFER, transformUbo);
	// projection matrix
	glBufferSubData(GL_UNIFORM_BUFFER, 0, sizeof(glm::mat4), glm::value_ptr(projection));
	// view matrix
	glBufferSubData(GL_UNIFORM_BUFFER, sizeof(glm::mat4), sizeof(glm::mat4), glm::value_ptr(view));
	// 3D view matrix
	glBufferSubData(GL_UNIFORM_BUFFER, 2*sizeof(glm::mat4), sizeof(glm::mat4), glm::value_ptr(view3D));
	glBindBuffer(GL_UNIFORM_BUFFER, 0);

	// set check factor, check size and draw uv sphere is mapped to sphere
	if (index == 0 || (index == 1 && (plotType == PlotType::shaded || plotType == PlotType::normals))) {
		modelShader.use();
//		glUniform1f(glGetUniformLocation(modelShader.program, "patternSize"), exp(patternSize));
//		glUniform1i(glGetUniformLocation(modelShader.program, "pattern"), paramComboBox->selectedIndex());
	}

	glUniform1i(glGetUniformLocation(modelShader.program, "shade"), plotType != PlotType::normals);
}

void Viewer::updateRenderMeshes()
{
	std::shared_ptr<Buffer> positions = state->renderMeshes[1]->positions;
	for (FaceCIter f = mesh->faces.begin(); f != mesh->faces.end(); f++) {
		int index = 3*f->index;

		VertexCIter w0 = f->he->vertex;
		VertexCIter w1 = f->he->prev->vertex;
		VertexCIter w2 = f->he->next->vertex;
		const Vector& p0 = w0->position;
		const Vector& p1 = w1->position;
		const Vector& p2 = w2->position;
		double s = cross(p1 - p0, p2 - p0).z;

		int j = 0;
		HalfEdgeCIter h = f->he;
		do {
			VertexCIter w = h->vertex;
			glm::vec3 color;

			if (plotType == PlotType::constant) {
				if (s > 0.0 || state->mappedToSphere) color = skyblue;
				else color = red;

			} else if (plotType == PlotType::shaded) {
				color = white;

			} else if (plotType == PlotType::normals) {
				for (int k = 0; k < 3; k++) {
					float nk = state->renderMeshes[0]->normals->buffer[index + j][k];
					color[k] = (nk + 1)/2.0;
				}
			} else if (plotType == PlotType::quasiConformalError) {
				Vector distortion = Distortion::color(f, true);
				color = glm::vec3(distortion.x, distortion.y, distortion.z);

			} else if (plotType == PlotType::angularError) {
				Vector distortion = Distortion::color(f, true);
				color = glm::vec3(distortion.x, distortion.y, distortion.z);

			} else if (plotType == PlotType::areaScaling) {
				Vector distortion = Distortion::color(f, false);
				color = glm::vec3(distortion.x, distortion.y, distortion.z);

			}
			for (int k = 0; k < 3; k++) {
				positions->buffer[index + j][k] = w->position2[k];
				state->renderMeshes[0]->colors->buffer[index + j][k] = color[k];
				state->renderMeshes[1]->colors->buffer[index + j][k] = color[k];
			}

			j++;
			h = h->next;
		} while (h != f->he);

		if (state->handles.size() > 0) {
			for (VertexHandle handle: state->handles) {
				const Vector& uv = handle.vertex->position2;
				const Vector& p = handle.vertex->position;
				addSphere(handle.sphere3D, glm::vec3(p.x, p.y, p.z), yellow);
				addSphere(handle.sphereUV, glm::vec3(uv.x, uv.y, uv.z), yellow);
			}
		}
	}

	// only update uvs for selected mesh
	positions->update(); // buffer is shared by both rendermeshes
	if (colorsNeedUpdate) {
		state->renderMeshes[0]->update({COLOR});
		state->renderMeshes[1]->update({COLOR});
	}

	colorsNeedUpdate = false;
}

void Viewer::updateViewPanes()
{
	surfacePane = ViewPane(framebufferWidth*0.17, 0, framebufferWidth*0.415, framebufferHeight);
	uvPane = ViewPane(framebufferWidth*0.585, 0, framebufferWidth*0.415, framebufferHeight);
	fullPane = ViewPane(0, 0, framebufferWidth, framebufferHeight);
}

void Viewer::updateCameraZoom(int idx_cam)
{
	double radius = 0.0;
	if(idx_cam == 0) {
		for (VertexCIter w = mesh->vertices.begin(); w != mesh->vertices.end(); w++) {
			radius = std::max(radius, w->position.norm());
		}
	} else {
		for (VertexCIter w = mesh->vertices.begin(); w != mesh->vertices.end(); w++) {
			radius = std::max(radius, w->position2.norm());
		}
	}

	cameras[idx_cam]->radius = EYE*radius;
	cameras[idx_cam]->viewNeedsUpdate = true;
}

void Viewer::update(bool updateZoom)
{
	// compute distortion
	switch (plotType) {
		case PlotType::constant:
			colorsNeedUpdate = true;
			break;
		case PlotType::shaded:
			colorsNeedUpdate = true;
			break;
		case PlotType::normals:
			colorsNeedUpdate = true;
			break;
		case PlotType::quasiConformalError:
			Distortion::computeQuasiConformalError(*mesh);
			colorsNeedUpdate = true;
			break;
		case PlotType::angularError:
			Distortion::computeAngularDistortion(*mesh);
			colorsNeedUpdate = true;
			break;
		case PlotType::areaScaling:
			Distortion::computeAreaScaling(*mesh);
			colorsNeedUpdate = true;
			break;
		default:
			break;
	}

	// update meshes and instructions
	updateRenderMeshes();
	if (updateZoom) {
		updateCameraZoom(0);
		updateCameraZoom(1);
	}
	draw();
}

void Viewer::updateSelectedMesh()
{
	// set current mesh and state; reset camera and update instructions
	cameras[0]->reset();
	cameras[1]->reset();
	updateCameraZoom(0);
	updateCameraZoom(1);

}

bool intersect(Vector u1, Vector u2, Vector v1, Vector v2)
{
	double ux = u2.x - u1.x;
	double uy = u2.y - u1.y;
	double vx = v2.x - v1.x;
	double vy = v2.y - v1.y;

	double den = -vx*uy + ux*vy;
	double s = (-uy*(u1.x - v1.x) + ux*(u1.y - v1.y))/den;
	double t = (vx*(u1.y - v1.y) - vy*(u1.x - v1.x))/den;

	return s >= 0 && s <= 1 && t >= 0 && t <= 1;
}

bool selfIntersects(const std::vector<Vector>& gamma)
{
	int n = (int)gamma.size();
	for (int i = 1; i < n; i++) {
		for (int j = i + 2; j < n; j++) {
			if (intersect(gamma[i], gamma[(i + 1)%n], gamma[j], gamma[(j + 1)%n])) {
				return true;
			}
		}
	}

	return intersect(gamma[n - 2], gamma[n - 1], gamma[0], gamma[1]);
}

Vector Viewer::ndc(int x, int y, int width, int height) {
	double s = 0.17;
	double t = 0.585;
	if (state->surfaceIsClosed && !state->mappedToSphere) {
		s = 0.0;
		t = 0.0;
	}

	return Vector(2.0*(x - width*s)/(width*(1.0 - t)) - 1.0,
				  2.0*(x - width*t)/(width*(1.0 - t)) - 1.0,
				  1.0 - 2.0*y/height);
}

void Viewer::cursorPosCallback(GLFWwindow *w, double x, double y)
{
	// call gui callback event
	if (!gui->cursorPosCallbackEvent(x, y) && loadedModel) {
		glfwGetWindowSize(w, &windowWidth, &windowHeight);
		if (mouseDown) {
			// camera rotation, panning
			Vector current = ndc(x, y, windowWidth, windowHeight);
			if (current[2] > -1 && current[2] < 1) {
				float dy = current[2] - mouse[2];

				if (current[0] > -1 && current[0] < 1) {
					float dx = current[0] - mouse[0];

					if (altClick) {
						cameras[0]->pan(dx, dy);
						cameras[1]->pan(dx, dy);
					}
					else {
						cameras[0]->rotate(-dx*M_PI, dy*M_PI);
						if (state->mappedToSphere) {
							cameras[1]->rotate(-dx*M_PI, dy*M_PI);

						} else {
							Vector u = origMouse - Vector(2, 0, 0);
							Vector v = current - Vector(2, 0, 0);
							u.y = 0.0;
							v.y = 0.0;
							u.x /= uvPane.height;
							v.x /= uvPane.height;
							u.z /= uvPane.width;
							v.z /= uvPane.width;
							double theta = atan2(cross(u, v).y, dot(u,v));
							cameras[1]->worldUp.x = cos(theta)*origUp.x - sin(theta)*origUp.y;
							cameras[1]->worldUp.y = sin(theta)*origUp.x + cos(theta)*origUp.y;
							cameras[1]->viewNeedsUpdate = true;
						}
					}

				} else if (current[1] > -1 && current[1] < 1) {
					float dx = current[1] - mouse[1];

					if (altClick) {
						cameras[0]->pan(dx, dy);
						cameras[1]->pan(dx, dy);

					} else {
						cameras[0]->rotate(-dx*M_PI, dy*M_PI);
						if (state->mappedToSphere) {
							cameras[1]->rotate(-dx*M_PI, dy*M_PI);

						} else {
							Vector u = origMouse - Vector(2, 0, 0);
							Vector v = current - Vector(2, 0, 0);
							u.y = 0.0;
							v.y = 0.0;
							u.x /= uvPane.height;
							v.x /= uvPane.height;
							u.z /= uvPane.width;
							v.z /= uvPane.width;
							double theta = atan2(cross(u, v).y, dot(u,v));
							cameras[1]->worldUp.x = cos(theta)*origUp.x - sin(theta)*origUp.y;
							cameras[1]->worldUp.y = sin(theta)*origUp.x + cos(theta)*origUp.y;
							cameras[1]->viewNeedsUpdate = true;
						}
					}
				}
			}
		}

		mouse = ndc(x, y, windowWidth, windowHeight);
	}
}


void Viewer::mouseButtonCallback(GLFWwindow *w, int button, int action, int modifiers)
{
	// call gui callback event
	if (!gui->mouseButtonCallbackEvent(button, action, modifiers) && loadedModel) {
		// track mouse click
		double x, y;
		glfwGetCursorPos(w, &x, &y);
		mouse = ndc(x, y, windowWidth, windowHeight);
		origMouse = mouse;
		origUp.x = cameras[1]->worldUp.x;
		origUp.y = cameras[1]->worldUp.y;
		origUp.z = cameras[1]->worldUp.z;

		ctrlClick = (bool)(modifiers & GLFW_MOD_CONTROL);
		altClick = (bool)(modifiers & GLFW_MOD_ALT);

		if (button == GLFW_MOUSE_BUTTON_LEFT && action == GLFW_PRESS) {
			clickX = gui->pixelRatio()*x;
			clickY = gui->pixelRatio()*y;
			mouseDown = true;
			shiftClick = (bool)(modifiers & GLFW_MOD_SHIFT);
		} else if (button == GLFW_MOUSE_BUTTON_LEFT && action == GLFW_RELEASE) {
			unclickX = gui->pixelRatio()*x;
			unclickY = gui->pixelRatio()*y;

			mouseDown = false;
			shiftClick = false;
                }

		

	}

}

void Viewer::keyCallback(GLFWwindow *w, int key, int scancode, int action, int mods)
{
	// call gui callback event
	gui->keyCallbackEvent(key, scancode, action, mods);

	if (key == GLFW_KEY_ESCAPE && action == GLFW_PRESS) {
		glfwSetWindowShouldClose(w, GLFW_TRUE);

	} else if (loadedModel) {
		if (key == GLFW_KEY_LEFT && action == GLFW_RELEASE) {
			selectedMesh--;
			if (selectedMesh < 0) selectedMesh = 0;
			updateSelectedMesh();

		} else if (key == GLFW_KEY_RIGHT && action == GLFW_RELEASE) {
			selectedMesh++;
			if (selectedMesh >0) selectedMesh = 0;
			updateSelectedMesh();
		}
	}
}

void Viewer::scrollCallback(GLFWwindow *w, double x, double y)
{
	// call gui callback event
	gui->scrollCallbackEvent(x, y);

	if (mouse[0] > -1 && mouse[0] < 1) cameras[0]->zoom(0.1*y);
	else if (mouse[1] > -1 && mouse[1] < 1) cameras[1]->zoom(0.1*y);
}

void Viewer::charCallback(GLFWwindow *w, unsigned int codepoint)
{
	// call gui callback event
	gui->charCallbackEvent(codepoint);
}

void Viewer::framebufferSizeCallback(GLFWwindow *w, int width, int height)
{
	// call gui callback event
	gui->resizeCallbackEvent(width, height);

	// update drawing
	draw();
}

void Viewer::drawRenderMeshes(const ViewPane& pane, int index)
{
	// update uniforms
	updateUniforms(index, pane.width, pane.height);

	// clear viewport
	clearViewport(pane);

	// draw model
	// draw all meshes in 3D pane and only the selected mesh in the UV pane
	int i = 0;
	if (index == 0 || (index == 1 && i == selectedMesh)) {
		// update per mesh uniforms
		modelShader.use();
		bool drawPattern = !state->surfaceIsClosed || state->mappedToSphere;
		glUniform1f(glGetUniformLocation(modelShader.program, "patternFactor"), drawPattern ? 0.5 : 0.0);
		glUniform1i(glGetUniformLocation(modelShader.program, "drawUVSphere"), state->mappedToSphere);
		glUniform1f(glGetUniformLocation(modelShader.program, "ambient"), i != selectedMesh ? 0.4 : 0.1);

		// draw mesh
		state->renderMeshes[index]->draw(modelShader);

		// draw vertex handles
		if(showCone) {
			if(index==0) {
				for (VertexHandle handle: state->handles) {
					handle.sphere3D->draw(flatShader);
				}
			} else {
				for (VertexHandle handle: state->handles) {
					handle.sphereUV->draw(flatShader);
				}
			}
		}

		// draw wireframe
		if (showWireframe) state->renderMeshes[index]->draw(wireframeShader);
	}
}

void Viewer::drawScene()
{
	clearViewport(fullPane);

	if (loadedModel) {
		drawRenderMeshes(surfacePane, 0);
		if (!state->surfaceIsClosed || state->mappedToSphere) {
			drawRenderMeshes(uvPane, 1);
		}
	}
}

void Viewer::draw()
{
	glfwGetFramebufferSize(window, &framebufferWidth, &framebufferHeight);
	updateViewPanes();
	pickingEnabled = false;

	// draw scene
	drawScene();

	// draw gui
	gui->drawWidgets();

	// swap buffers
	glfwSwapBuffers(window);
}

void Viewer::reportError(const std::string& error)
{
	new nanogui::MessageDialog(gui, nanogui::MessageDialog::Type::Warning, "Error", error);
}

/* ===============================================================================================
   Add a sphere
   =============================================================================================== */

void Viewer::addSphere(std::shared_ptr<RenderMesh>& sphere,
					   const glm::vec3& center, const glm::vec3& color)
{
	sphere = std::shared_ptr<RenderMesh>(new RenderMesh());
	updateSphere(sphere, center, color);
}

void Viewer::addSphere(std::vector<std::shared_ptr<RenderMesh>>& spheres,
			   const glm::vec3& center, const glm::vec3& color)
{
	std::shared_ptr<RenderMesh> sphere(new RenderMesh());
	updateSphere(sphere, center, color);
	spheres.push_back(sphere);
}

/* ===============================================================================================
   Update display of a sphere
   =============================================================================================== */

void Viewer::updateSphere(std::shared_ptr<RenderMesh> sphere, const glm::vec3& center, const glm::vec3& color)
{
	const int nSlices = 10;
	const int nStacks = 10;
	const float dTheta = 2*M_PI/nSlices;
	const float dPhi = M_PI/nSlices;
	const float radius = 0.04;

	if (!sphere->positions) sphere->positions = std::shared_ptr<Buffer>(new Buffer());
	else sphere->positions->buffer.clear();
	if (!sphere->colors) sphere->colors = std::shared_ptr<Buffer>(new Buffer());
	else sphere->colors->buffer.clear();

	double phi = 0.0;
	for (int i = 0; i < nSlices; i++) {
		double theta = 0.0;
		for (int j = 0; j < nStacks; j++) {
			glm::vec3 n00(cos(theta + 0*dTheta)*sin(phi + 0*dPhi),
						  sin(theta + 0*dTheta)*sin(phi + 0*dPhi),
						  cos(phi + 0*dPhi));
			glm::vec3 n01(cos(theta + 1*dTheta)*sin(phi + 0*dPhi),
						  sin(theta + 1*dTheta)*sin(phi + 0*dPhi),
						  cos(phi + 0*dPhi));
			glm::vec3 n10(cos(theta + 0*dTheta)*sin(phi + 1*dPhi),
						  sin(theta + 0*dTheta)*sin(phi + 1*dPhi),
						  cos(phi + 1*dPhi));
			glm::vec3 n11(cos(theta + 1*dTheta)*sin(phi + 1*dPhi),
						  sin(theta + 1*dTheta)*sin(phi + 1*dPhi),
						  cos(phi + 1*dPhi));
			glm::vec3 p00 = center + radius*n00;
			glm::vec3 p01 = center + radius*n01;
			glm::vec3 p10 = center + radius*n10;
			glm::vec3 p11 = center + radius*n11;

			sphere->positions->buffer.push_back(p00);
			sphere->positions->buffer.push_back(p01);
			sphere->positions->buffer.push_back(p10);
			sphere->positions->buffer.push_back(p10);
			sphere->positions->buffer.push_back(p01);
			sphere->positions->buffer.push_back(p11);
			for (int k = 0; k < 6; k++) sphere->colors->buffer.push_back(color);

			theta += dTheta;
		}
		phi += dPhi;
	}

	sphere->update({POSITION, COLOR});
}

/* ===============================================================================================
   Add handle to one vertex
   =============================================================================================== */

void Viewer::addVertexHandle(VertexIter v, double newAngle)
{
	// check if vertex v already has a handle
	bool handleExists = false;
	VertexHandle existingHandle;
	for (VertexHandle handle: state->handles) {
		glm::vec3 color = yellow;

		if (handle.vertex == v) {
			existingHandle = handle;
			color = orange;
			state->selectedVertexHandle = handle.vertex;
			handleExists = true;
		}

		for (int i = 0; i < (int)handle.sphere3D->colors->buffer.size(); i++) {
			handle.sphere3D->colors->buffer[i] = color;
			handle.sphereUV->colors->buffer[i] = color;
		}

		handle.sphere3D->update({COLOR});
		handle.sphereUV->update({COLOR});
	}

	if (handleExists) {
		// if vertex v already has a handle, just move it to the front of the list
		state->handles.remove(existingHandle);
		state->handles.push_front(existingHandle);

	} else {
		// otherwise, create a new handle for vertex v (putting it at the front of the list)
		state->selectedVertexHandle = v;

		VertexHandle handle;
		handle.vertex = v;
		const Vector& uv = v->position2;
		const Vector& p = v->position;
		addSphere(handle.sphere3D, glm::vec3(p.x, p.y, p.z), orange);
		addSphere(handle.sphereUV, glm::vec3(uv.x, uv.y, uv.z), orange);
		state->handles.push_front(handle);
	}
}
/* ===============================================================================================
   Remove handle to one vertex
   =============================================================================================== */

void Viewer::removeVertexHandle(VertexIter v)
{
	// check if the vertex has a handle
	for (VertexHandle handle: state->handles) {
		if (handle.vertex == v) {
			state->handles.remove(handle);
			if (state->selectedVertexHandle == v) state->selectedVertexHandle = mesh->vertices.end();

			// update interface
			if (state->selectedVertexHandle == mesh->vertices.end() || state->handles.size() == 0) {
				 state->selectedVertexHandle = mesh->vertices.end();
			}

			break;
		}
	}
}

/* ===============================================================================================
   Remove all vertex handles
   =============================================================================================== */

void Viewer::removeVertexHandles()
{
	// clear all cones and set cone angles to 0
	state->handles.clear();
	state->selectedVertexHandle = mesh->vertices.end();
}

/* ===============================================================================================
   Update one vertex handle
   =============================================================================================== */

void Viewer::updateVertexHandle(VertexIter v)
{
	if (state->handles.size() > 0) {
		VertexHandle handle = state->handles.front();
	}

	if (ctrlClick) {
		removeVertexHandle(v);

	} else {
		addVertexHandle(v, 0.0);
	}
}

/* ===============================================================================================
   Update vertex handles
   =============================================================================================== */

void Viewer::updateVertexHandlesUVs()
{
	for (VertexHandle handle: state->handles) {
		VertexIter v = handle.vertex;
		glm::vec3 color = v == state->selectedVertexHandle ? orange : yellow;
		const Vector& uv = v->position2;
		updateSphere(handle.sphereUV, glm::vec3(uv.x, uv.y, uv.z), color);
	}
}

/* ===============================================================================================
   Get vertex handles
   =============================================================================================== */

std::vector<VertexIter> Viewer::getHandleVertices()
{
	std::vector<VertexIter> handleVertices;
	for (VertexHandle handle: state->handles) {
		handleVertices.push_back(handle.vertex);
	}

	return handleVertices;
}
