/*================================================================================================
  bestfit.h
  Version 1: 6/16/2023

Copyright (c) Patrice Koehl.

>>> SOURCE LICENSE >>>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

>>> END OF LICENSE >>>

================================================================================================== */

#ifndef _BESTFIT_
#define _BESTFIT_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>


/*================================================================================================
  Class
================================================================================================== */

  template <typename T>
  class BESTFIT {

	public:

		T bestfitm(T *coord, int nat1, T *coord2, int nat2, int nat, int *list1,
		int *list2, T *mass1, T *mass2);

	private:

		T deter3(T *a);
		void transpose3(T *a, T *b);

  };


/*================================================================================================
 mass-weighted bestfit between two conformations of a molecule
================================================================================================== */

  template <typename T>
  T BESTFIT<T>::bestfitm(T *coord1, int nat1, T *coord2, int nat2, int nat, int *list1,
		int *list2, T *mass1, T *mass2)
  {

	T rmsd=0;

/*================================================================================================
 	Find center of mass
================================================================================================== */

	T xc1[3], xc2[3];
	for(int i = 0; i < 3; i++) {xc1[i]=0; xc2[i]=0;};

	T xmass1=0, xmass2=0;
	for(int i = 0; i < nat; i++) {
		for(int j = 0; j < 3; j++) {
			xc1[j] += coord1[j + 3*list1[i]]*mass1[list1[i]];
			xc2[j] += coord2[j + 3*list2[i]]*mass2[list2[i]];
		}
		xmass1 += mass1[list1[i]];
		xmass2 += mass2[list2[i]];
	}

	for(int i = 0; i < 3; i++) {xc1[i] /= xmass1; xc2[i] /= xmass2;};

/*================================================================================================
 	Calculate covariance matrix
================================================================================================== */

	T covar[9];
	T val;

	for(int i = 0; i < 3; i++) {
		for(int j = 0; j < 3; j++) {
			val = 0;
			for(int k = 0; k < nat; k++) {
				val += (coord1[i+3*list1[k]]-xc1[i])
				*(coord2[j+3*list2[k]]-xc2[j])*mass1[list1[k]];
			}
			covar[i+3*j]=val;
		}
	}

/*================================================================================================
 	Calculate determinant of covariance matrix
================================================================================================== */

	T det = deter3(covar);

	if(det==0) {
		std::cout << "Error in bestfit: det(covar) = 0" << std::endl;
		return rmsd;
	}

	T sign = 1.0;
	if(det < 0) sign = -1;


/*================================================================================================
 	Compute svd of covariance matrix
================================================================================================== */

	char JOBU = 'A'; char JOBV = 'A';
	int m = 3;
	int info;
	int lwork = 100;
	T work[100];
	T S[3], U[9], VT[9], V[9];

	eig_dgesvd_(&JOBU, &JOBV, &m, &m, covar, &m, S, U, &m, VT, &m, work, &lwork, &info);
	transpose3(VT, V);

	if(info !=0) {
		std::cout << "Error in bestfit: SVD fails " << std::endl;
		return rmsd;
	}

/*================================================================================================
	Calculate bestfit rotational matrix r
================================================================================================== */

	T tiny = 1.e-14;
	T r[9];
	T norm;

	if(S[1] > tiny) {
		if(S[2] <= tiny) {
			sign = 1.0;
			U[0+6] = U[1]*U[2+3] - U[2]*U[1+3];
			U[1+6] = U[2]*U[0+3] - U[0]*U[2+3];
			U[2+6] = U[0]*U[1+3] - U[1]*U[0+3];
			V[0+6] = V[1]*U[2+3] - V[2]*U[1+3];
			V[1+6] = V[2]*U[0+3] - V[0]*U[2+3];
			V[2+6] = V[0]*U[1+3] - V[1]*U[0+3];
		}
		for(int i = 0; i < 3; i++) {
			for(int j = 0; j < 3; j++) {
				r[i+3*j] = U[i]*V[j]+U[i+3]*V[j+3] + sign*U[i+6]*V[j+6];
			}
		}
	} else {
		work[0] = U[1]*V[2+3] - U[2]*V[1+3];
		work[1] = U[2]*V[0+3] - U[0]*V[2+3];
		work[2] = U[0]*V[1+3] - U[1]*V[0+3];
		norm = 0;
		for(int i = 0; i < 3; i++) norm += work[i]*work[i];
		if(norm !=0) {
			for(int i = 0; i < 3; i++) work[i] = U[i]+V[i];
		} else {
			work[0] = - U[1]; work[1] = U[0]; work[2] = 0;
		}
		norm = 0;
		for(int i = 0; i < 3; i++) norm += work[i]*work[i];
		norm = std::sqrt(norm);
		for(int i = 0; i < 3; i++) work[i] /= norm;
		for(int i = 0; i < 3; i++) {
			for(int j = 0; j < 3; j++) {
				r[i+3*j] = 2*work[i]*work[j];
			}
			r[i+3*i] -= 1;
		}
	}

	det = deter3(r);

	if(det < 0) {
		std::cout << "Warning: det rotation matrix negative" << std::endl;
	}

/*================================================================================================
	Apply rotation matrix on coord2
================================================================================================== */

	T c[3];
	for(int i = 0; i < nat2; i++) {
		for(int j = 0; j < 3; j++) {
			c[j] = 0;
			for(int k = 0; k < 3; k++) {
				c[j] += r[j+3*k]*(coord2[k+3*i]-xc2[k]);
			}
		}
		for(int j = 0; j < 3; j++) coord2[j+3*i] = c[j] + xc1[j];
	}

/*================================================================================================
	Compute rmsd
================================================================================================== */

	rmsd = 0;
	for(int i = 0; i < nat; i++) {
		for(int j = 0; j< 3; j++) {
			val = coord2[j+3*list2[i]]-coord1[j+3*list1[i]];
			rmsd+= val*val;
		}
	}

	rmsd = std::sqrt(std::abs(rmsd)/nat);

	return rmsd;

  }

/*================================================================================================
  Compute a 3x3 determinant
================================================================================================== */

  template <typename T>
  T BESTFIT<T>::deter3(T *a)
  {

	T val0 = a[0]*(a[4]*a[8]-a[5]*a[7]);
	T val1 = a[1]*(a[3]*a[8]-a[5]*a[6]);
	T val2 = a[2]*(a[3]*a[7]-a[4]*a[6]);

	T det = val0 - val1 + val2;

	return det;
  }

/*================================================================================================
  Transpose a 3x3 matrix
================================================================================================== */

  template <typename T>
  void BESTFIT<T>::transpose3(T *a, T *b)
  {
	b[0] = a[0];
	b[1] = a[3];
	b[2] = a[6];
	b[3] = a[1];
	b[4] = a[4];
	b[5] = a[7];
	b[6] = a[2];
	b[7] = a[5];
	b[8] = a[8];
  }

#endif
