/*********************************************************************************
 *	The Vector class: tools to manipulate 3D vectors
 *********************************************************************************/

#ifndef VECTOR_H
#define VECTOR_H

#include "math.h"

/*********************************************************************************
 *	The class
 *********************************************************************************/

  template <typename T>
  class Vector {

		public:

			// initializes all components to zero
			Vector();

			// initializes with specified components
			Vector(T x, T y, T z = 0.0);

			// copy constructor
			Vector(const Vector& v);

			// access
			T& operator[](int index);
			const T& operator[](int index) const;

			// math
			Vector operator*(T s) const;
			Vector operator/(T s) const;
			Vector operator+(const Vector& v) const;
			Vector operator-(const Vector& v) const;
			Vector operator-() const;
			T operator*(const Vector& v) const;

			Vector& operator*=(T s);
			Vector& operator/=(T s);
			Vector& operator+=(const Vector& v);
			Vector& operator-=(const Vector& v);

			Vector operator^(const Vector& v) const;

			// returns Euclidean length
			T norm() const;

			// returns Euclidean length squared
			T norm2() const;

			// normalizes vector
			void normalize();

			// returns unit vector in the direction of this vector
			Vector unit() const;

			// members
			T x, y, z;
};

/*********************************************************************************
 * Global functions associated with vectors
 *********************************************************************************/

template <typename T>
Vector<T> operator*(T s, const Vector<T>& v);

template <typename T>
T dot(const Vector<T>& u, const Vector<T>& v);

template <typename T>
Vector<T> cross(const Vector<T>& u, const Vector<T>& v);

/*********************************************************************************
 * Constructor (set coords to 0)
 *********************************************************************************/

template <typename T>
inline Vector<T>::Vector():
x(0.0),
y(0.0),
z(0.0)
{

}

/*********************************************************************************
 * Constructor (with coords)
 *********************************************************************************/

template <typename T>
inline Vector<T>::Vector(T x_, T y_, T z_):
x(x_),
y(y_),
z(z_)
{

}

/*********************************************************************************
 * Constructor (with vector)
 *********************************************************************************/

template <typename T>
inline Vector<T>::Vector(const Vector<T>& v):
x(v.x),
y(v.y),
z(v.z)
{

}

/*********************************************************************************
 * Access one coordinate using []
 *********************************************************************************/

template <typename T>
inline T& Vector<T>::operator[](int index)
{
	return (&x)[index];
}

/*********************************************************************************
 * Access one coordinate using []
 *********************************************************************************/

template <typename T>
inline const T& Vector<T>::operator[](int index) const
{
	return (&x)[index];
}

/*********************************************************************************
 * Scale
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::operator*(T s) const
{
	return Vector<T>(x*s, y*s, z*s);
}

/*********************************************************************************
 * Divide by constant
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::operator/(T s) const
{
	return (*this)*(1.0/s);
}

/*********************************************************************************
 * Add two vectors
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::operator+(const Vector<T>& v) const
{
	return Vector<T>(x + v.x, y + v.y, z + v.z);
}

/*********************************************************************************
 * Dot product (set as "*")
 *********************************************************************************/

template <typename T>
inline T Vector<T>::operator*(const Vector<T>& v) const
{
	return x*v.x + y*v.y + z*v.z;
}

/*********************************************************************************
 * Cross product
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::operator^(const Vector<T>& v) const
{
	return Vector<T>(y*v.z - z*v.y, z*v.x - x*v.z, x*v.y - y*v.x);
}

/*********************************************************************************
 * Difference   
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::operator-(const Vector<T>& v) const
{
	return Vector<T>(x - v.x, y - v.y, z - v.z);
}

/*********************************************************************************
 * negate a vector (multiply by -1)
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::operator-() const
{
	return Vector<T>(-x, -y, -z);
}

/*********************************************************************************
 * Use of *=
 *********************************************************************************/

template <typename T>
inline Vector<T>& Vector<T>::operator*=(T s)
{
	x *= s;
	y *= s;
	z *= s;

	return *this;
}

/*********************************************************************************
 * Use of /=
 *********************************************************************************/

template <typename T>
inline Vector<T>& Vector<T>::operator/=(T s)
{
	(*this) *= (1.0/s);

	return *this;
}

/*********************************************************************************
 * Use of +=
 *********************************************************************************/

template <typename T>
inline Vector<T>& Vector<T>::operator+=(const Vector<T>& v)
{
	x += v.x;
	y += v.y;
	z += v.z;

	return *this;
}

/*********************************************************************************
 * Use of -=
 *********************************************************************************/

template <typename T>
inline Vector<T>& Vector<T>::operator-=(const Vector<T>& v)
{
	x -= v.x;
	y -= v.y;
	z -= v.z;

	return *this;
}

/*********************************************************************************
 * Norm of a vector
 *********************************************************************************/

template <typename T>
inline T Vector<T>::norm() const
{
	return sqrt(norm2());
}

/*********************************************************************************
 * Norm^2 of a vector
 *********************************************************************************/

template <typename T>
inline T Vector<T>::norm2() const
{
	return dot(*this, *this);
}

/*********************************************************************************
 * Normalize a vector
 *********************************************************************************/

template <typename T>
inline void Vector<T>::normalize()
{
	(*this) /= norm();
}

/*********************************************************************************
 * Define unit vector (same as normalize)
 *********************************************************************************/

template <typename T>
inline Vector<T> Vector<T>::unit() const
{
   return (*this) / norm();
}

/*********************************************************************************
 * Scale by a constant
 *********************************************************************************/

template <typename T>
inline Vector<T> operator*(T s, const Vector<T>& v)
{
	return v*s;
}

/*********************************************************************************
 * Dot product
 *********************************************************************************/

template <typename T>
inline T dot(const Vector<T>& u, const Vector<T>& v)
{
	return u.x*v.x + u.y*v.y + u.z*v.z;
}

/*********************************************************************************
 * cross product
 *********************************************************************************/

template <typename T>
inline Vector<T> cross(const Vector<T>& u, const Vector<T>& v)
{
	return Vector<T>(u.y*v.z - u.z*v.y, u.z*v.x - u.x*v.z, u.x*v.y - u.y*v.x);
}

#endif
