/*================================================================================================
  chebDavED.h
  Version 1: 08/10/2024

  Purpose: Finds the top N eigenvalues / eigenvectors of a real symmetric matrix

  Method: Block Chebyshev-Davidson algorithm, with explicit deflation

  Y. Zhou and Y. Saad, A Chebyshev-Davidson algorithm for large symmetric
  eigenproblems, SIAM J. Matrix Anal. Appl., 29, 954-971 (2007)

  Y. Zhou. A block Chebyshev-Davidson method with inner-outer restart for
  large eigenvalue problems. J. Comput. Phys, 229, 9188-9200 (2010)

Copyright (c) Patrice Koehl.

>>> SOURCE LICENSE >>>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

>>> END OF LICENSE >>>

================================================================================================== */

#ifndef _CHEBDAVED_
#define _CHEBDAVED_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <iostream>
#include <chrono>
#include <cstdlib>

/*================================================================================================
 define opA
================================================================================================== */

typedef void (*opA) (int nrow, int ncol, T *x, int ldx, T *y, int ldy, void *mvparam);

/*================================================================================================
  Define a class for computing eigenvectors / eigenvalues of a symmetric sparse matrix
  stored as sum of outer products
================================================================================================== */

template <typename T>
  class CHEBDAVED {

	public:

		// driver for computing some eigenpairs of a symmetric sparse matrix
		// Explicit Deflation approach    
		void chebDavED(opA opA0, opA opAD, eig_info<T> * info, int N, 
			T *eigVal, T *eigVect, int *k_conv, T *work,
			std::vector<std::vector<double>>& edinfo);

		// driver for computing some eigenpairs of a symmetric sparse matrix
		// Explicit Deflation approach with Communication Avoiding
		void chebDavCA(opA opA0, opA opAD, eig_info<T> * info, int N, 
			T *eigVal, T *eigVect, int *k_conv, T *work,
			std::vector<std::vector<double>>& edinfo);

	private:


	protected:

  };


/*================================================================================================
  chebDavED

	Purpose:
	========
	Find the NE smallest eigenvalues of a Hessian, using the Block-Chebyshev-Davidson
	     algorithm and an explicit Hotelling deflation scheme

	Arguments:
	==========

	op	(input) pointer
		On entry, pointer to the Matrix-Vector operator

	info	(input) pointer to info struct
		On entry, structure containing the parameters for the run

	N	(input) integer
		On entry, size of the matrix

	eigVal	(output) array of float or double, of size  >= k_conv
		On exit, the converged eigenvalue. Should be dimensioned >= NE

	eigVect	(output) array of float or double, of size  >= k_conv*N
		On exit, the converged eigenvector. Should be dimensioned >= NE*N

	k_conv	(output) integer
		The actual number of eigenpairs that have converged

	work1	(input) float or double array for chebDav

	work2	(input) float or double array of size at least 2*NEV + NEV*Nblocks

	edinfo	(output) STL vector
		On exit, information about each cycle of explicit deflation:
			- time (matrix vector)
			- time (orthogonalization)
			- time (total)
			- time (total wall time)
			- NMV  number of Matrix x Vector multiply
			- NORTH number of orthogonalization
			- Nconf number of eigenpairs that have converged
			- niter Number of iterations
================================================================================================== */

template <typename T>
 void CHEBDAVED<T>::chebDavED(opA opA0, opA opAD, eig_info<T> * info, int N, 
			T *eigVal, T *eigVect, int *k_conv, T *work,
			std::vector<std::vector<double>>& edinfo)
  {

/*================================================================================================
	Declare some variables
================================================================================================== */

	clock_t clk0, clk1;
	timeval tim;
	double t1, t2, u1, u2;

	std::vector<double> ed; // contains information about

	edinfo.clear();

	int dim_max = info->nev + info->act_max;
	int act_max = info->act_max;

/*================================================================================================
	Initialize
================================================================================================== */

	clk0 = clock();
	gettimeofday(&tim,NULL);
	t1 = tim.tv_sec;
	u1 = tim.tv_usec;

/*================================================================================================
	First call to chebDav: no deflation yet
================================================================================================== */

	int nconv = 0;
	dav.chebDav(opA0, opA0, info, N, eigVal, eigVect, &nconv, work);

/*================================================================================================
	Store information about this run
================================================================================================== */
	
	ed.push_back(info->clk_op); ed.push_back(info->clk_orth);
	ed.push_back(info->clk_tot); ed.push_back(info->walltime);
	ed.push_back(info->mvp); ed.push_back(info->north);
	ed.push_back(nconv); ed.push_back(info->niter);
	edinfo.push_back(ed);

/*================================================================================================
	now performs deflation cycles
================================================================================================== */

	opparams<T> *par;
	par         = (opparams<T> *) info->mvparams;

	int nev_tot = 0;

	for(int i_ed = 1; i_ed < info->ned; i_ed++) {

/*================================================================================================
		Adjust last number of eigenpairs to be found in previous cycles found more
================================================================================================== */

		if(i_ed==info->ned-1) info->nev_d = info->nev - nev_tot - nconv;

		info->act_max = std::min(act_max, dim_max - info->nev_d - nev_tot - nconv);

/*================================================================================================
		Define shifts for deflation
================================================================================================== */

		for(int i = 0; i < nconv ; i++) {
			par->sigma[nev_tot+i] = eigVal[0] + info->anrm - eigVal[nev_tot+i];
		}

/*================================================================================================
		Update parameters
================================================================================================== */

		nev_tot		 += nconv;
		par->need	  = nev_tot;
		info->mvp	  = 0;
		info->north 	  = 0;
		info->clk_op	  = 0;
		info->clk_orth	  = 0;
		info->clk_tot	  = 0;
		info->flag_start  = 1;
		info->filter_type = 1;

/*================================================================================================
		Now call chebDav
================================================================================================== */

		dav.chebDav(opA0, opAD, info, N, &eigVal[nev_tot], &eigVect[N*nev_tot], 
		&nconv, work);

/*================================================================================================
		Store information about this cycle
================================================================================================== */

		ed.clear();
		ed.push_back(info->clk_op); ed.push_back(info->clk_orth);
		ed.push_back(info->clk_tot); ed.push_back(info->walltime);
		ed.push_back(info->mvp); ed.push_back(info->north);
		ed.push_back(nconv); ed.push_back(info->niter);
		edinfo.push_back(ed);

/*================================================================================================
		End cycles
================================================================================================== */

	}

	nev_tot += nconv;
	*k_conv = nev_tot;

/*================================================================================================
	Store total time for chebDavED (CPU time and wall time) in info
================================================================================================== */

	ed.clear();
	clk1 = clock();
	info->clk_tot = (clk1-clk0);

	gettimeofday(&tim,NULL);
	t2 = tim.tv_sec;
	u2 = tim.tv_usec;
	info->walltime = (t2-t1) + (u2-u1)*1.e-6;

}

/*================================================================================================
  chebDavCA

	Purpose:
	========
	Find the NE smallest eigenvalues of a Hessian, using the Block-Chebyshev-Davidson
	     algorithm and an explicit Hotelling deflation scheme and communication avoiding

	Arguments:
	==========

	op	(input) pointer
		On entry, pointer to the Matrix-Vector operator

	info	(input) pointer to info struct
		On entry, structure containing the parameters for the run

	N	(input) integer
		On entry, size of the matrix

	eigVal	(output) array of float or double, of size  >= k_conv
		On exit, the converged eigenvalue. Should be dimensioned >= NE

	eigVect	(output) array of float or double, of size  >= k_conv*N
		On exit, the converged eigenvector. Should be dimensioned >= NE*N

	k_conv	(output) integer
		The actual number of eigenpairs that have converged

	work1	(input) float or double array for chebDav

	work2	(input) float or double array of size at least 2*NEV + NEV*Nblocks

	edinfo	(output) STL vector
		On exit, information about each cycle of explicit deflation:
			- time (matrix vector)
			- time (orthogonalization)
			- time (total)
			- time (total wall time)
			- NMV  number of Matrix x Vector multiply
			- NORTH number of orthogonalization
			- Nconf number of eigenpairs that have converged
			- Niter number of iterations
================================================================================================== */

template <typename T>
 void CHEBDAVED<T>::chebDavCA(opA opA0, opA opAD, eig_info<T> * info, int N, 
			T *eigVal, T *eigVect, int *k_conv, T *work,
			std::vector<std::vector<double>>& edinfo)
  {

/*================================================================================================
	Declare some variables
================================================================================================== */

	clock_t clk0, clk1;
	timeval tim;
	double t1, t2, u1, u2;

	std::vector<double> ed; // contains information about

	edinfo.clear();

	int dim_max = info->nev + info->act_max;
	int act_max = info->act_max;

/*================================================================================================
	Initialize
================================================================================================== */

	clk0 = clock();
	gettimeofday(&tim,NULL);
	t1 = tim.tv_sec;
	u1 = tim.tv_usec;

/*================================================================================================
	First call to chebDav: no deflation yet
================================================================================================== */

	int nev_tot = 0;

	int nconv=0;
	dav.chebDav(opA0, opA0, info, N, eigVal, eigVect, &nconv, work);

/*================================================================================================
	Store information about this run
================================================================================================== */
	
	ed.push_back(info->clk_op); ed.push_back(info->clk_orth);
	ed.push_back(info->clk_tot); ed.push_back(info->walltime);
	ed.push_back(info->mvp); ed.push_back(info->north);
	ed.push_back(nconv); ed.push_back(info->niter);
	edinfo.push_back(ed);

/*================================================================================================
	now performs deflation cycles
================================================================================================== */

	opparams<T> *par;
	par         = (opparams<T> *) info->mvparams;

	for(int i_ed = 1; i_ed < info->ned; i_ed++) {

/*================================================================================================
		Adjust last number of eigenpairs to be found in previous cycles found more
================================================================================================== */

		if(i_ed==info->ned-1) info->nev_d = std::max(0, info->nev - nev_tot - nconv);
		if(info->nev_d==0) {
			info->ned--;
			break;
		}

		info->act_max = std::min(act_max, dim_max - info->nev_d - nev_tot - nconv);

/*================================================================================================
		Define shifts for deflation
================================================================================================== */

		for(int i = 0; i < nconv ; i++) {
			par->sigma[nev_tot+i] = eigVal[0] + info->anrm - eigVal[nev_tot+i];
		}

/*================================================================================================
		Update parameters
================================================================================================== */

		nev_tot		 += nconv;
		par->need	  = nev_tot;
		info->mvp	  = 0;
		info->north 	  = 0;
		info->clk_op	  = 0;
		info->clk_orth	  = 0;
		info->clk_tot	  = 0;
		info->flag_start  = 1;
		info->filter_type = 2;

/*================================================================================================
		Now call chebDav
================================================================================================== */

		dav.chebDav(opA0, opAD, info, N, &eigVal[nev_tot], &eigVect[N*nev_tot], 
		&nconv, work);

/*================================================================================================
		Store information about this cycle
================================================================================================== */

		ed.clear();
		ed.push_back(info->clk_op); ed.push_back(info->clk_orth);
		ed.push_back(info->clk_tot); ed.push_back(info->walltime);
		ed.push_back(info->mvp); ed.push_back(info->north);
		ed.push_back(nconv); ed.push_back(info->niter);
		edinfo.push_back(ed);

/*================================================================================================
		End cycles
================================================================================================== */

	}

	nev_tot += nconv;
	*k_conv = nev_tot;

/*================================================================================================
	Store total time for chebDavED (CPU time and wall time) in info
================================================================================================== */

	ed.clear();
	clk1 = clock();
	info->clk_tot = (clk1-clk0);

	gettimeofday(&tim,NULL);
	t2 = tim.tv_sec;
	u2 = tim.tv_usec;
	info->walltime = (t2-t1) + (u2-u1)*1.e-6;

}

#endif
