/*================================================================================================
  Hessian.h
  Version 1: 9/29/2019

  Purpose: Sets of routine for defining the Hessian for the elastic potential and 
	   computing Hessian - Vect products

Copyright (c) Patrice Koehl.

================================================================================================== */

#ifndef _HESSIAN_H_
#define _HESSIAN_H_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>
#include <cstring>
#include <vector>
#include "Edges.h"

/*================================================================================================
 Class for Hessian
================================================================================================== */

  template <typename T>
  class Hessian {

	public:

		// build non-bonded part of Hessian
		void buildH(std::vector<Atoms<T> >& atoms, std::vector<Edges<T> >& List, 
		T kconst, int potential, hessianMat<T>* csrHessian);

		// build Go backbone potential
		void buildGO(std::vector<Atoms<T> >& atoms, hessianMat<T>* csrHessian);

		// build full Hessian matrix
		void fullHessian(hessianMat<T>* csrHessian, T *hessian);

		// Diagonalize full Hessian
		void fullEigen(int natoms, T *eigVal, T *eigVect);

		// Rescale eigenvectors to account for mass
		void rescaleEigVect(std::vector<Atoms<T> >& atoms, int NE, T *eigVect);

		// Compute GO potential
		void goEnergy(std::vector<Atoms<T> >& atoms, T *newconf, int N,
			T *U_bond, T *U_angle, T *U_dih, T *U_Go);

	private:

		// update full Hessian matrix
		void update_hessian(int N, T *hessian, int i, int j, T *u, T *v, T fact);
  };


/*================================================================================================
 The Hessian matrix is stored in csr format (with arrays ia, ja, and val)
 pairs (i,j) and (j,i) are both stored
================================================================================================== */

template <typename T>
void Hessian<T>::buildH(std::vector<Atoms<T> >& atoms, std::vector<Edges<T> >& List, 
	T kconst, int potential, hessianMat<T>* csrHessian)
{
	int i1, j1;
	T dx, dy, dz, d;
	T fact;
	T sqr_k = std::sqrt( kconst );

	/*==========================================================================================
	Find Hessian dimension;
	Note that we store ia (row_pointers) and ja (col_pointers) based on atoms
	while the value is a small 3D vector
	============================================================================================ */

	int Natom = atoms.size();
	int Npair = List.size();
	int NNZ   = 2*Npair;

	csrHessian->nrows = Natom;
	csrHessian->nnz   = NNZ;

	csrHessian->ia  = new int[Natom+1];
	csrHessian->ja  = new int[NNZ];
	csrHessian->val = new T[3*NNZ];

	memset(csrHessian->ia, 0, (Natom+1)*sizeof(int));

	/*==========================================================================================
    	First pass: count non-zeros per row and build row_pointer array (i.e., ia)
	============================================================================================ */
    
	std::vector<int> nnz_per_row(Natom, 0);
	for (int pair = 0; pair < Npair; pair++)
	{
		i1 = List[pair].atm1;
		j1 = List[pair].atm2;
		nnz_per_row[i1]++;
		nnz_per_row[j1]++;
	}

	for(int i = 0; i < Natom; i++) {
		csrHessian->ia[i+1] = csrHessian->ia[i] + nnz_per_row[i];
	}

	/*==========================================================================================
    	Second pass: fill in matrix
	============================================================================================ */
    
	// pointer at the beginning of each row
	std::vector<int> current_pos(Natom);
	for(int i = 0; i < Natom; i++) current_pos[i] = csrHessian->ia[i];

	// fill col_pointer and values
	int pos_i, pos_j;
	for (int pair = 0; pair < Npair; pair++)
	{
		i1 = List[pair].atm1;
		j1 = List[pair].atm2;

		dx = atoms[i1].coord[0] - atoms[j1].coord[0];
		dy = atoms[i1].coord[1] - atoms[j1].coord[1];
		dz = atoms[i1].coord[2] - atoms[j1].coord[2];

		fact = std::sqrt(std::sqrt(atoms[i1].mass * atoms[j1].mass));

		d = dx*dx + dy*dy + dz*dz;
		if(potential == 1) d = std::sqrt(d);
		d = sqr_k/(d*fact);

        	// Add upper triangular entry (i, j)
		pos_i = current_pos[i1]++;
		csrHessian->ja[pos_i] = j1;
		csrHessian->val[3*pos_i + 0] = dx*d;
		csrHessian->val[3*pos_i + 1] = dy*d;
		csrHessian->val[3*pos_i + 2] = dz*d;

        	// Add lower triangular entry (i, j)
		pos_j = current_pos[j1]++;
		csrHessian->ja[pos_j] = i1;
		csrHessian->val[3*pos_j + 0] = dx*d;
		csrHessian->val[3*pos_j + 1] = dy*d;
		csrHessian->val[3*pos_j + 2] = dz*d;

	}
    
	/*==========================================================================================
	Sort column indices within each row for proper CSR format
	This is not needed if List is really sorted in lexicographic order, but for safety...
	============================================================================================ */
    
	int start, end, nval, j;
	std::vector<T> values;
	for(int i = 0; i < Natom; i++) {

		start = csrHessian->ia[i];
		end   = csrHessian->ia[i+1];
		nval  = end-start;

		values.clear();
		for(int k = 0; k < 3*nval; k++) {
			values.push_back(csrHessian->val[3*start+k]);
		} 
        
		// Create pairs of (column, value)
		std::vector<std::pair<int, int> > row_data;
		for (int k = start; k < end; k++) {
			row_data.push_back(std::make_pair(csrHessian->ja[k], k-start));
		}

		// Sort by column index
		std::sort(row_data.begin(), row_data.end());
        
		// Write back sorted data
		for (int k = 0; k < (int) row_data.size(); k++) {
			csrHessian->ja[start+k] = row_data[k].first;
			j = row_data[k].second;
			for(int l = 0; l < 3; l++) {
				csrHessian->val[3*start + 3*k + l] = values[3*j+l];
			}
		}

	}
}
	
/*================================================================================================
 From tensor to full matrix: update with one interaction
================================================================================================== */

template <typename T>
void Hessian<T>::update_hessian(int N, T *hessian, int i, int j, T *u, T *v, T fact)
{
	int i1=3*i;
	int j1=3*j;
	hessian[i1*N + j1 ] += fact*u[0]*v[0];
	hessian[i1*N + j1 + 1] += fact*u[0]*v[1];
	hessian[i1*N + j1 + 2] += fact*u[0]*v[2];
	hessian[i1*N + N + j1 ] += fact*u[1]*v[0];
	hessian[i1*N + N + j1 + 1] += fact*u[1]*v[1];
	hessian[i1*N + N + j1 + 2] += fact*u[1]*v[2];
	hessian[i1*N + 2*N + j1 ] += fact*u[2]*v[0];
	hessian[i1*N + 2*N + j1 + 1] += fact*u[2]*v[1];
	hessian[i1*N + 2*N + j1 + 2] += fact*u[2]*v[2];
}

/*================================================================================================
 fullEigen : builds full eigen from its sparse representation, and diagonalize it using
	     LAPACK algorithms

================================================================================================== */

template <typename T>
void Hessian<T>::fullEigen(int natoms, T *eigVal, T *eigVect) 
{
/*================================================================================================
	Input:
		natoms    : number of atoms
		List	  : list of all interaction pairs
		eigVect   : Full hessian
	Output:
		eigVal    : the 3*natoms eigenvalues
		eigVect   : the corresponding eigenvectors
================================================================================================== */

/*================================================================================================
	Declare some variables
================================================================================================== */

	int lwork, liwork, info;
	int isizeopt;
	int N = 3*natoms;

	T sizeopt;

	char U = 'U';
	char V = 'V';

/*================================================================================================
	Create all temporary arrays
================================================================================================== */

	lwork = -1;
	liwork = -1;
	eig_dsyevd_(&V, &U, &N, eigVect, &N, eigVal, &sizeopt, &lwork, &isizeopt, &liwork, &info);

	lwork = (int) sizeopt;
	liwork = isizeopt;
	int *iwork = new int[liwork];
	T *dwork = new T[lwork];

/*================================================================================================
	Compute eigenvalues / eigenvectors
================================================================================================== */

	eig_dsyevd_(&V, &U, &N, eigVect, &N, eigVal, dwork, &lwork, iwork, &liwork, &info);

/*================================================================================================
	Clean up all temporary arrays
================================================================================================== */

	delete [] dwork;
	delete [] iwork;

}

/*================================================================================================
 Rescale eigvect: rescale eigenvectors based on mass of atoms
================================================================================================== */

template <typename T>
void Hessian<T>::rescaleEigVect(std::vector<Atoms<T> >& atoms, int NE, T *eigVect)
{
	int N = 3*atoms.size();
	T alpha;
	T *Temp = new T[N];

	for(int i = 0; i < N/3; i++) {
		alpha = 1.0/std::sqrt(atoms[i].mass);
		Temp[3*i] = alpha;
		Temp[3*i+1] = alpha;
		Temp[3*i+2] = alpha;
	}

	for(int i = 0; i < NE; i++)
	{
		for(int j = 0; j < N; j++) {
			eigVect[N*i + j] = eigVect[N*i +j]*Temp[j];
		}
	}

	delete [] Temp;
}

/*================================================================================================
 Tensor vectors that defines Hessian for bond, angle, and dihedral terms for Go potential
================================================================================================== */

template <typename T>
void Hessian<T>::buildGO(std::vector<Atoms<T> >& atoms, hessianMat<T>* csrHessian)
{
	int i1, j1, k1, l1;
	T dx, dy, dz, d;
	T fact;

	int natoms = atoms.size();

	csrHessian->type   = 2;

	csrHessian->bonds  = new T[3*natoms];
	csrHessian->angles = new T[9*natoms];
	csrHessian->diheds = new T[12*natoms];

	/*==========================================================================================
	Bond contribution
	============================================================================================ */

	T kconst = 200*0.36;
	T sqr_k = std::sqrt(kconst);

	for (int pair = 0; pair < natoms-1; pair++)
	{
		i1 = pair;
		j1 = pair+1;

		if(atoms[i1].chainid == atoms[j1].chainid)
		{
			dx = atoms[i1].coord[0] - atoms[j1].coord[0];
			dy = atoms[i1].coord[1] - atoms[j1].coord[1];
			dz = atoms[i1].coord[2] - atoms[j1].coord[2];

			fact = std::sqrt(std::sqrt(atoms[i1].mass * atoms[j1].mass));

			d = dx*dx + dy*dy + dz*dz;
			d = std::sqrt(d);
			d = sqr_k/(d*fact);

			csrHessian->bonds[3*pair + 0]   = dx*d;
			csrHessian->bonds[3*pair + 1]   = dy*d;
			csrHessian->bonds[3*pair + 2]   = dz*d;
		} else {
			csrHessian->bonds[3*pair + 0]   = 0;
			csrHessian->bonds[3*pair + 1]   = 0;
			csrHessian->bonds[3*pair + 2]   = 0;
		}
	}

	/*==========================================================================================
	angle contribution
	============================================================================================ */

	T r1, r2, a1, a2, a3;
	T cosine, sine, alpha, beta, beta1, beta2;
	kconst = 40*0.36;
	sqr_k = std::sqrt(kconst);
	T d1[3], d2[3], u[3], v[3];

	for (int ang = 0; ang < natoms-2; ang++)
	{
		i1 = ang;
		j1 = ang+1;
		k1 = ang+2;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) ) 
		{
			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
			}
			r1 = d1[0]*d1[0] + d1[1]*d1[1] + d1[2]*d1[2];
			r2 = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			d  = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			r1 = std::sqrt(r1);
			r2 = std::sqrt(r2);
			cosine = d/(r1*r2);
			sine   = std::sqrt(1-cosine*cosine);
			alpha  = sqr_k/(sine*r1*r2);
			beta1  = sqr_k*cosine/(sine*r1*r1);
			beta2  = sqr_k*cosine/(sine*r2*r2);

			for(int i = 0; i < 3; i++)
			{
				u[i] = alpha*d2[i]-beta1*d1[i];
				v[i] = alpha*d1[i]-beta2*d2[i];
			}

			csrHessian->angles[9*ang]   = u[0];
			csrHessian->angles[9*ang+1] = u[1];
			csrHessian->angles[9*ang+2] = u[2];
			csrHessian->angles[9*ang+3] = -u[0]-v[0];
			csrHessian->angles[9*ang+4] = -u[1]-v[1];
			csrHessian->angles[9*ang+5] = -u[2]-v[2];
			csrHessian->angles[9*ang+6] = v[0];
			csrHessian->angles[9*ang+7] = v[1];
			csrHessian->angles[9*ang+8] = v[2];
		} else {
			for (int i = 0; i < 9 ; i++ ) {
				csrHessian->angles[9*ang+i]=0;
			}
		}
	}

	/*==========================================================================================
	dihedral contribution
	============================================================================================ */

	T r, rm, rn, a4;
	T kconst1 = 0.36;
	T kconst3 = 0.18;
	sqr_k = std::sqrt(kconst1+9*kconst3);
	T d3[3], xm[3], xn[3], t[3];

	for (int dih = 0; dih < natoms-3; dih++)
	{
		i1 = dih;
		j1 = dih+1;
		k1 = dih+2;
		l1 = dih+3;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) &&
		(atoms[i1].chainid == atoms[l1].chainid))
		{
			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				a4 = atoms[l1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
				d3[i]   = a4 - a3;
			}
			r  = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			a1 = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			a2 = d2[0]*d3[0] + d2[1]*d3[1] + d2[2]*d3[2];
			xm[0] = d1[1]*d2[2] - d1[2]*d2[1];
			xm[1] = -d1[0]*d2[2] + d1[2]*d2[0];
			xm[2] = d1[0]*d2[1] - d1[1]*d2[0];
			xn[0] = d2[1]*d3[2] - d2[2]*d3[1];
			xn[1] = -d2[0]*d3[2] + d2[2]*d3[0];
			xn[2] = d2[0]*d3[1] - d2[1]*d3[0];
			rm    = xm[0]*xm[0] + xm[1]*xm[1] + xm[2]*xm[2];
			rn    = xn[0]*xn[0] + xn[1]*xn[1] + xn[2]*xn[2];
			r  = std::sqrt(r);
			alpha = a1/(r*r);
			beta  = a2/(r*r);

			for( int i = 0; i < 3; i++)
			{
				u[i] = r*xm[i]/rm;
				t[i] = -r*xn[i]/rn;
				v[i] = (alpha-1.0)*u[i] -beta*t[i];
			}

			csrHessian->diheds[12*dih]    = sqr_k*u[0];
			csrHessian->diheds[12*dih+1]  = sqr_k*u[1];
			csrHessian->diheds[12*dih+2]  = sqr_k*u[2];
			csrHessian->diheds[12*dih+3]  = sqr_k*v[0];
			csrHessian->diheds[12*dih+4]  = sqr_k*v[1];
			csrHessian->diheds[12*dih+5]  = sqr_k*v[2];
			csrHessian->diheds[12*dih+6]  = -sqr_k*(u[0]+v[0]+t[0]);
			csrHessian->diheds[12*dih+7]  = -sqr_k*(u[1]+v[1]+t[1]);
			csrHessian->diheds[12*dih+8]  = -sqr_k*(u[2]+v[2]+t[2]);
			csrHessian->diheds[12*dih+9]  = sqr_k*t[0];
			csrHessian->diheds[12*dih+10] = sqr_k*t[1];
			csrHessian->diheds[12*dih+11] = sqr_k*t[2];
		} else
		{
			for (int i = 0; i < 12; i++) {
				csrHessian->diheds[12*dih+i] = 0;
			}
		}
	}
}

/*+================================================================================================
 Define full hessian matrix
================================================================================================== */

template <typename T>
void Hessian<T>::fullHessian(hessianMat<T>* csrHessian, T *hessian)
{
	int natoms = csrHessian->nrows;
  
	int N = 3*natoms;
	std::memset(hessian, 0, N*N*sizeof(T));

	int i1, j1, k1, l1;
	int j, start, end;
	T u[3], v[3], w[3], t[3];
	T kconst = 1;

	if(csrHessian->type==2) {
		for(int atom = 0; atom < natoms-1; atom++)
		{
			i1 = atom;
			j1 = atom + 1;
			for (int i = 0; i < 3; i++)
			{
				u[i] = csrHessian->bonds[3*atom+i];
				v[i] = -csrHessian->bonds[3*atom+i];
			}
			update_hessian(N, hessian, i1, j1, u, v, kconst);
			update_hessian(N, hessian, j1, i1, v, u, kconst);
		}

		for(int atom = 0; atom < natoms-2; atom++)
		{
			i1 = atom;
			j1 = atom + 1;
			k1 = atom + 2;
			for (int i = 0; i < 3; i++)
			{
				u[i] = csrHessian->angles[9*atom+i];
				v[i] = csrHessian->angles[9*atom+3+i];
				w[i] = csrHessian->angles[9*atom+6+i];
			}
			update_hessian(N, hessian, i1, j1, u, v, kconst);
			update_hessian(N, hessian, j1, i1, v, u, kconst);
			update_hessian(N, hessian, i1, k1, u, w, kconst);
			update_hessian(N, hessian, k1, i1, w, u, kconst);
			update_hessian(N, hessian, j1, k1, v, w, kconst);
			update_hessian(N, hessian, k1, j1, w, v, kconst);
		}

		for(int atom = 0; atom < natoms-3; atom++)
		{
			i1 = atom;
			j1 = atom + 1;
			k1 = atom + 2;
			l1 = atom + 3;
			for (int i = 0; i < 3; i++)
			{
				u[i] = -csrHessian->diheds[12*atom+i];
				v[i] = -csrHessian->diheds[12*atom+3+i];
				w[i] = -csrHessian->diheds[12*atom+6+i];
				t[i] = -csrHessian->diheds[12*atom+9+i];
			}
			update_hessian(N, hessian, i1, j1, u, v, kconst);
			update_hessian(N, hessian, j1, i1, v, u, kconst);
			update_hessian(N, hessian, i1, k1, u, w, kconst);
			update_hessian(N, hessian, k1, i1, w, u, kconst);
			update_hessian(N, hessian, i1, l1, u, t, kconst);
			update_hessian(N, hessian, l1, i1, t, u, kconst);
			update_hessian(N, hessian, j1, k1, v, w, kconst);
			update_hessian(N, hessian, k1, j1, w, v, kconst);
			update_hessian(N, hessian, j1, l1, v, t, kconst);
			update_hessian(N, hessian, l1, j1, t, v, kconst);
			update_hessian(N, hessian, k1, l1, w, t, kconst);
			update_hessian(N, hessian, l1, k1, t, w, kconst);
		}
	}

	for(int i = 0; i < natoms; i++) {
		start = csrHessian->ia[i];
		end   = csrHessian->ia[i+1];

		for (int k = start; k < end; k++) {

			j = csrHessian->ja[k];
			for(int l = 0; l < 3; l++) {
				u[l] = csrHessian->val[3*k+l];
				v[l] = -u[l];
			}

			update_hessian(N, hessian, i, j, u, v, kconst);
		}
	}

	T sum;
	int Ncol;
	for(int i = 0; i < natoms; i++)
	{
		for(int l = 0; l < 3; l++) {
			Ncol = 3*i+l;
			for(int k = 0; k < 3; k++)
			{
				sum = 0.0;
				for(int j = k; j < N; j = j + 3) 
				{
					sum = sum + hessian[Ncol*N+j];
				}
				hessian[Ncol*N + 3*i + k] = -sum;
			}
		}
	}

}

/*================================================================================================
 goEnergy: computes the bonded energy for the Go potential
================================================================================================== */

template <typename T>
void Hessian<T>::goEnergy(std::vector<Atoms<T> >& atoms, T *newconf, int N,
	T *U_bond, T *U_angle, T *U_dih, T *U_Go)
{
	int i1, j1, k1, l1;
	T dx, dy, dz, d, r0, r;

	T K_bond  = 200*0.36;
	T K_angle = 40*0.36;
	T K_dih1  = 0.36;
	T K_dih3  = 0.18;

/*================================================================================================
 	Bond energy for Go potential
================================================================================================== */

	T bond = 0;
	int nb = 0;
	for (int nbond = 0; nbond < N-1; nbond++)
	{
		i1 = nbond;
		j1 = nbond+1;

		if(atoms[i1].chainid == atoms[j1].chainid)
		{
			dx = atoms[i1].coord[0] - atoms[j1].coord[0];
			dy = atoms[i1].coord[1] - atoms[j1].coord[1];
			dz = atoms[i1].coord[2] - atoms[j1].coord[2];

			r0 = dx*dx + dy*dy + dz*dz;
			r0 = std::sqrt(r0);

			dx = newconf[3*i1] - newconf[3*j1];
			dy = newconf[3*i1+1] - newconf[3*j1+1];
			dz = newconf[3*i1+2] - newconf[3*j1+2];

			r  = dx*dx + dy*dy + dz*dz;
			r = std::sqrt(r);

			bond = bond + (r-r0)*(r-r0);
			nb++;
		}
	}
	bond = K_bond*bond;
	*U_bond = std::sqrt(bond/nb);

/*================================================================================================
 	Angle energy for Go potential
================================================================================================== */

	T a1, a2, a3, a4;
	T r1, r2, cosine;
	T theta0, theta;
	T d1[3], d2[3], d3[3];

	T ang = 0;
	for (int nang = 0; nang < N-2; nang++)
	{
		i1 = nang;
		j1 = nang+1;
		k1 = nang+2;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) ) 
		{

			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
			}
			r1 = d1[0]*d1[0] + d1[1]*d1[1] + d1[2]*d1[2];
			r2 = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			d  = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			r1 = std::sqrt(r1);
			r2 = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta0 = std::acos(cosine);

			for (int i = 0; i < 3; i++)
			{
				a1 = newconf[3*i1+i];
				a2 = newconf[3*j1+i];
				a3 = newconf[3*k1+i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
			}
			r1 = d1[0]*d1[0] + d1[1]*d1[1] + d1[2]*d1[2];
			r2 = d2[0]*d2[0] + d2[1]*d2[1] + d2[2]*d2[2];
			d  = d1[0]*d2[0] + d1[1]*d2[1] + d1[2]*d2[2];
			r1 = std::sqrt(r1);
			r2 = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta = std::acos(cosine);

			if( std::abs(theta-theta0) > 6) {
				std::cout << "Angle problem: i,j,k: " << i1 << " " << j1 << " " << k1 << " theta0, theta: " << theta0 << " " << theta << std::endl;
			}

			ang = ang + (theta-theta0)*(theta-theta0);
		}
	}
	ang = K_angle*ang;
	*U_angle = ang;

/*================================================================================================
 	Dihedral energy for Go potential
================================================================================================== */

	T xm[3], xn[3];
	T dih = 0;

	for (int ndih = 0; ndih < N-3; ndih++)
	{
		i1 = ndih;
		j1 = ndih+1;
		k1 = ndih+2;
		l1 = ndih+3;

		if((atoms[i1].chainid == atoms[j1].chainid) && (atoms[i1].chainid == atoms[k1].chainid) &&
		(atoms[i1].chainid == atoms[l1].chainid))
		{
			for (int i = 0; i < 3; i++)
			{
				a1 = atoms[i1].coord[i];
				a2 = atoms[j1].coord[i];
				a3 = atoms[k1].coord[i];
				a4 = atoms[l1].coord[i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
				d3[i]   = a4 - a3;
			}
			xm[0] = d1[1]*d2[2] - d1[2]*d2[1];
			xm[1] = -d1[0]*d2[2] + d1[2]*d2[0];
			xm[2] = d1[0]*d2[1] - d1[1]*d2[0];
			xn[0] = d2[1]*d3[2] - d2[2]*d3[1];
			xn[1] = -d2[0]*d3[2] + d2[2]*d3[0];
			xn[2] = d2[0]*d3[1] - d2[1]*d3[0];
			r1    = xm[0]*xm[0] + xm[1]*xm[1] + xm[2]*xm[2];
			r2    = xn[0]*xn[0] + xn[1]*xn[1] + xn[2]*xn[2];
			d     = xm[0]*xn[0] + xm[1]*xn[1] + xm[2]*xn[2];
			r1    = std::sqrt(r1);
			r2    = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta0  = std::acos(cosine);

			for (int i = 0; i < 3; i++)
			{
				a1 = newconf[3*i1+i];
				a2 = newconf[3*j1+i];
				a3 = newconf[3*k1+i];
				a4 = newconf[3*l1+i];
				d1[i]   = a2 - a1;
				d2[i]   = a2 - a3;
				d3[i]   = a4 - a3;
			}
			xm[0] = d1[1]*d2[2] - d1[2]*d2[1];
			xm[1] = -d1[0]*d2[2] + d1[2]*d2[0];
			xm[2] = d1[0]*d2[1] - d1[1]*d2[0];
			xn[0] = d2[1]*d3[2] - d2[2]*d3[1];
			xn[1] = -d2[0]*d3[2] + d2[2]*d3[0];
			xn[2] = d2[0]*d3[1] - d2[1]*d3[0];
			r1    = xm[0]*xm[0] + xm[1]*xm[1] + xm[2]*xm[2];
			r2    = xn[0]*xn[0] + xn[1]*xn[1] + xn[2]*xn[2];
			d     = xm[0]*xn[0] + xm[1]*xn[1] + xm[2]*xn[2];
			r1    = std::sqrt(r1);
			r2    = std::sqrt(r2);
			cosine = d/(r1*r2);
			theta  = std::acos(cosine);

			dih = dih + K_dih1*std::cos(theta-theta0) +
			K_dih3*std::cos(3*(theta-theta0));
		}
	}
	*U_dih = dih;

	*U_Go = bond + ang + dih;

}
#endif
