/*================================================================================================
  pebbleComponent

  Implementation of the component pebble game algorithm proposed in:

        Audrey Lee and Ileana Streinu, "Pebble game algorithms and sparse graphs",
        Discrete Mathematics 308 (2008) 1425 – 1437

WARNING: this version only works for the low range, i.e. l in [0, k])

Copyright (c) Patrice Koehl.

>>> SOURCE LICENSE >>>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

>>> END OF LICENSE >>>

================================================================================================== */

#ifndef _PEBBLECOMPONENT_
#define _PEBBLECOMPONENT_

/*================================================================================================
 Includes
================================================================================================== */

#include <iostream>
#include <vector>
#include <thread>
#include <atomic>
#include <sys/time.h>

/*================================================================================================
 pebbleComponent class
================================================================================================== */

class pebbleComponent {

	public:

		// Constructor
		pebbleComponent(int n_, int k_, int l_, int nthreads_);

		int n;
		int k, l;
		int nc;
		int stamp;
		int nthreads;

		double time_component;
		double time_collect;

		int nfind_tot, nfind_success;
		double nmarked;
		double size;

		std::vector<int> parent;
		std::vector<std::vector<int> > bfs_queues;

		// directed edges in diGraph maintained by the pebble game
		std::vector<std::vector<int> > out_edges; 

		// number of free pebbles at each node
		std::vector<int> pebbles;     

		// component membership
		std::vector<int> component;     

		// Precomputed reverse adjacency list for faster reverse DFS
		std::vector<std::vector<int> > in_edges;

		// markup for vertices (used for component identification
		std::vector<std::atomic<int> > markup;

		// threads
		std::vector<std::thread> threads;

		std::vector<int> depth;

		// Insert edge if possible
		bool insert_edge(int u, int v);

		// Greedy BFS to find and free a pebble starting at root
		bool find_pebble(int root, int other);

		// Path reversal and pebble collection, when found
		void path_reversal(int root, int u);

		// Try to collect up to `need` pebbles from u and v, 
		// respecting k-pebble limit on either u or v
		int collect_pebbles(int u, int v, int need);

		// Component identification methods
		bool identify_new_component(int u, int v);
		bool identify_new_component_parallel(int u, int v);

		// BFS over the reversed graph
		void bfs_reverse_graph(int start, int tid);
		
		// count pebbles
		void count_pebbles(int *npebbles, int *nvertices);

};
	
/*================================================================================================
 pebbleComponent constructor
================================================================================================== */

 pebbleComponent::pebbleComponent(int n_, int k_, int l_, int nthreads_) : 
		n(n_), k(k_), l(l_), nc(1), stamp(0), nthreads(nthreads_),
		parent(n_, -1), bfs_queues(nthreads_, std::vector<int>(n_)), out_edges(n_), pebbles(n_, k_), 
		component(n_, 0), in_edges(n_), markup(n_), threads(nthreads_), depth(n_)
 {

	time_component = 0;
	time_collect = 0;

	nfind_tot = 0;
	nfind_success = 0;
	size = 0;
	nmarked = 0;

	for(int i = 0; i < n_; i++) markup[i].store(0, std::memory_order_relaxed);

	if(k != 0 && l != 0) {
		if (k <= 0) {
			std::cout << "k must be positive" << std::endl;
			exit(1);
		}
		if (l < 0) {
			std::cout << "l must be non-negative" << std::endl;
			exit(1);
		}
		if (l > 2*k) {
			std::cout << "Warning: l should not be less than 2*k" << std::endl;
		}
	}
  }

/*================================================================================================
  Insert edge, if possible
================================================================================================== */

 bool pebbleComponent::insert_edge(int u, int v) 
 {
	if(stamp > 1e6) {
		for(int i = 0; i < n; i++) markup[i].store(0, std::memory_order_relaxed);
		stamp = 0;
	}

	timeval tim;
	double t1, t2;

	if (u == v) return false;
	if (component[u]!=0 && (component[u]==component[v])) return false;
	if( collect_pebbles(u, v, l + 1) >= (l + 1) ) {

		/*================================================================================
		insert edge, assign its direction and consume pebble on one vertex
		================================================================================== */

		if (pebbles[u] > 0) {
			pebbles[u]--;
			out_edges[u].push_back(v);
			in_edges[v].push_back(u);
		} else if (pebbles[v] > 0) {
			pebbles[v]--;
			out_edges[v].push_back(u);
			in_edges[u].push_back(v);
		} else {
			// This should never happen
			std::cout << "Error: no pebble available on u or v after successful collection!" << std::endl;
			return false;
		}

		/*================================================================================
		Define new component created by (u, v)
		================================================================================== */

		gettimeofday(&tim,NULL);
		t1 = tim.tv_sec + tim.tv_usec*1.e-6;

		bool found = identify_new_component_parallel(u, v);

		gettimeofday(&tim,NULL);
		t2 = tim.tv_sec + tim.tv_usec*1.e-6;
		time_component += (t2-t1);

		if(found) nc++;

		return true;
	} else {
		return false;
	}
 }

/*================================================================================================
  DFS in reverse graph D1 (with all edge directions reversed)
================================================================================================== */

 void pebbleComponent::bfs_reverse_graph(int start, int tid)
 {

	auto& queue = bfs_queues[tid];

	int front = 0;
	int back = 0;

	if (markup[start].load(std::memory_order_relaxed) != stamp+1) {
		markup[start].store(stamp+1, std::memory_order_relaxed);
		queue[back++] = start;
	}
	
	while (front < back) {
		int u = queue[front++];
		
		// Use precomputed in_edges for reverse graph traversal
		for (int v : in_edges[u]) {
			if (markup[v].load(std::memory_order_relaxed) !=stamp+1) {
				markup[v].store(stamp+1, std::memory_order_relaxed);
				queue[back++] = v;
			}
		}
	}
 }

/*================================================================================================
  Greedy BFS to find and free a pebble starting at root
================================================================================================== */

bool pebbleComponent::find_pebble(int root, int other) 
{
	auto& queue = bfs_queues[0];

	int front = 0;
	int back = 0;
		
	int u;
		
	// Push root
	queue[back++] = root;
		
	while (front < back) {
		u = queue[front++];
		
		for(int v : out_edges[u]) {
			if (markup[v].load(std::memory_order_relaxed) != stamp) {
				parent[v] = u;
				if (v != root && v != other && pebbles[v] > 0) {
					path_reversal(root, v);
					nmarked += back;
					return true;
				}
				markup[v].store(stamp, std::memory_order_relaxed);
				queue[back++] = v;
			}
		}
	}

	return false;
}

/*================================================================================================
  Try to collect up to `need` pebbles from u and v, respecting k-pebble limit
================================================================================================== */

 int pebbleComponent::collect_pebbles(int u, int v, int need) 
 {

	timeval tim;    
	double t1, t2;

	int total = pebbles[u] + pebbles[v];

	gettimeofday(&tim,NULL);
	t1 = tim.tv_sec + tim.tv_usec*1.e-6;

	bool success;
	while (total < need) {

		success = false;

		// Try to find and free pebble for u if it can still hold more
		if (pebbles[u] < k) {
			stamp++;
			nfind_tot++;
			if (find_pebble(u, v)) {
				pebbles[u]++;
				total++;
				nfind_success++;
				success = true;
			}
		}

		// Try v if u failed and v can hold more
		if (pebbles[v] < k) {
			stamp++;
			nfind_tot++;
			if (find_pebble(v, u)) {
				pebbles[v]++;
				total++;
				nfind_success++;
				success = true;
			}
		}

		if (!success) break; // cannot collect more pebbles without violating limits
	}

	gettimeofday(&tim,NULL);
	t2 = tim.tv_sec + tim.tv_usec*1.e-6;
	time_collect += (t2-t1);

	return total;
 }

/*================================================================================================
  identify_new_component
================================================================================================== */

bool pebbleComponent::identify_new_component(int u, int v)
{

/*================================================================================================
	Step 1: If more than l pebbles are present on u and v, return empty set
================================================================================================== */

	if (pebbles[u] + pebbles[v] > l) {
		return false;
	}
	
/*================================================================================================
	Step 2: Compute Reach(u,v) with early termination in case it finds a vertex with a pebble
================================================================================================== */

	stamp++;

	auto& queue = bfs_queues[0];
	int front = 0;
	int back = 0;
	
	// Start BFS from both u and v
	queue[back++] = u;
	queue[back++] = v;
	markup[u].store(stamp, std::memory_order_relaxed); markup[v].store(stamp, std::memory_order_relaxed);
	
	while (front < back) {
		int current = queue[front++];
		
		// Early termination: if we find a vertex (other than u,v) with pebbles, return empty
		if (current != u && current != v && pebbles[current] >= 1) {
			return false; // Early termination
		}
		
		// Continue BFS
		for (int neighbor : out_edges[current]) {
			if (markup[neighbor].load(std::memory_order_relaxed) != stamp) {
				markup[neighbor].store(stamp, std::memory_order_relaxed);
				queue[back++] = neighbor;
			}
		}
	}
	
/*================================================================================================
	Step 3: If we reach here, no vertices with pebble were found in Reach(u,v)
		Continue with the reverse DFS step
================================================================================================== */

	for (int w = 0; w < n; w++) {
		// Check if w is not in Reach(u,v) and has at least one pebble
		if (markup[w].load(std::memory_order_relaxed) != stamp && pebbles[w] >= 1) {
			bfs_reverse_graph(w, 0);
		}
	}
	
/*================================================================================================
	Step 4: Return V' - the set of non-visited vertices from all reverse searches
================================================================================================== */

	int nfound = 0;
	for (int w = 0; w < n; w++) {
		if(markup[w].load(std::memory_order_relaxed) != stamp+1) {
			component[w]=nc;
			nfound++;
		}
	}

	stamp++;
	
/*================================================================================================
	Filter out degenerate cases
================================================================================================== */

	if(nfound == n || nfound == 0) return false;
	
	return true;
}

/*================================================================================================
  identify_new_component_parallel
================================================================================================== */

bool pebbleComponent::identify_new_component_parallel(int u, int v)
{

/*================================================================================================
	Step 1: If more than l pebbles are present on u and v, return empty set
================================================================================================== */

	if (pebbles[u] + pebbles[v] > l) {
		return false;
	}
	
/*================================================================================================
	Step 2: Compute Reach(u,v) with early termination in case it finds a vertex with a pebble
================================================================================================== */

	stamp++;

	auto& queue = bfs_queues[0];
	int front = 0;
	int back = 0;
	
	// Start BFS from both u and v
	queue[back++] = u;
	queue[back++] = v;
	markup[u].store(stamp, std::memory_order_relaxed); markup[v].store(stamp, std::memory_order_relaxed);
	
	while (front < back) {
		int current = queue[front++];
		
		// Early termination: if we find a vertex (other than u,v) with pebbles, return empty
		if (current != u && current != v && pebbles[current] >= 1) {
			return false; // Early termination
		}
		
		// Continue BFS
		for (int neighbor : out_edges[current]) {
			if (markup[neighbor].load(std::memory_order_relaxed) != stamp) {
				markup[neighbor].store(stamp, std::memory_order_relaxed);
				queue[back++] = neighbor;
			}
		}
	}
	
/*================================================================================================
	Step 3: If we reach here, no vertices with pebble were found in Reach(u,v)
		Continue with the reverse BFS step
================================================================================================== */

	// Collect vertices that need reverse BFS
	std::vector<int> vertices_to_process;
	for (int w = 0; w < n; w++) {
		if (markup[w].load(std::memory_order_relaxed) != stamp && pebbles[w] >= 1) {
			vertices_to_process.push_back(w);
		}
	}
    
	if ((int) vertices_to_process.size() < nthreads) {

		// If few vertices, run sequentially
		for (int w : vertices_to_process) {
			bfs_reverse_graph(w, 0);
		}

	} else {

	        // Run in parallel
        
		int vertices_per_thread = vertices_to_process.size() / nthreads;
		threads.clear();
		
		for (int t = 0; t < nthreads; t++) {
			int start_idx = t * vertices_per_thread;
			int end_idx = (t == nthreads - 1) ? vertices_to_process.size() : start_idx + vertices_per_thread;

			threads.emplace_back([this, start_idx, end_idx, vertices_to_process, t] {
				for (int i = start_idx; i < end_idx ; i++) {
					this->bfs_reverse_graph(vertices_to_process[i], t);
				}
			});
		}
		for (auto& t : threads) {
			t.join();
		}

	}

/*================================================================================================
	Step 4: Return V' - the set of non-visited vertices from all reverse searches
================================================================================================== */

	int nfound = 0;
	for (int w = 0; w < n; w++) {
		if(markup[w].load(std::memory_order_relaxed) != stamp+1) {
			component[w]=nc;
			nfound++;
		}
	}

	stamp++;
	
/*================================================================================================
	Filter out degenerate cases
================================================================================================== */

	if(nfound == n || nfound == 0) return false;
	
	return true;
}

/*================================================================================================
  Path reversal
================================================================================================== */

 void pebbleComponent::path_reversal(int root, int u)
 {

	int current = u;
	int p;
	int n1 = 0;

	// Backtrack and reverse edge directions
	while (current != root) {
		p = parent[current];
		auto it1 = std::find(out_edges[p].begin(), out_edges[p].end(), current);
		*it1 = out_edges[p].back();
		out_edges[p].pop_back();
		auto it2 = std::find(in_edges[current].begin(), in_edges[current].end(), p);
		*it2 = in_edges[current].back();
		in_edges[current].pop_back();
		out_edges[current].push_back(p);
		in_edges[p].push_back(current);
		current = p;
		n1++;
	}

	pebbles[u]--;
	size += n1;

 }

/*================================================================================================
 count:
	# of remaining pebbles
	# of vertices with pebbles
================================================================================================== */

 void pebbleComponent::count_pebbles(int *npebbles, int *nvertices)
 {
	int n1 = 0;
	int n2 = 0;
	for(int i = 0; i < n; i++) {
		n1 += pebbles[i];
		if(pebbles[i] > 0) n2++;
	}

	*npebbles = n1;
	*nvertices = n2;
 }

#endif
