/*================================================================================================
  WriteResults.h
  Version 1: 12/1/2017

Copyright (c) Patrice Koehl.

>>> SOURCE LICENSE >>>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

>>> END OF LICENSE >>>

================================================================================================== */

#ifndef _WRITERESULTS_H_
#define _WRITERESULTS_H_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>
#include <random>
#include <string>
#include <fstream>
#include <iostream>
#include <sstream>

/*================================================================================================
 class   
================================================================================================== */

 template <typename T>
 class WriteResults {

	public:

 		// writeBfact: computed and experimental B-factors
		void writeBfact(std::string filename, std::vector<Atoms<T>>& atoms, T *bfact);

		// writeOverlap: contribution of each mode to: b-factor, overlap, rms
		void writeBfact2(std::string filename, int nmode1, int nmodes, T *rms_bfact, T *correl_bfact);

 		// writeCorrel: write matrix of correlation of motions
		void writeCorrel(std::string filename, int Natoms, T *correl);

		// writeOverlap: contribution of each mode to: b-factor, overlap, rms
		void writeOverlap(std::string filename, int nmode1, int nmodes, int check_bfact, T *rms_bfact, 
		T *correl_bfact, T rms0, T *rms, T *overlap, T *quality, T *cadist);

		// writeEig: write Eigenvalues and Eigenvectors in a binary file
		void writeEig(std::string filename, T *eigVal, T *eigVect, int Natoms, int nmodes);

		void writeTraj(std::string filename, std::vector<Atoms<T>>& atoms, int Natoms, 
			int imode, T eigval, T *eigvect, T rms_target, int Nperiod, int Npose);

		void trajCIF(std::string filename, std::vector<Atoms<T>>& atoms, int Natoms, 
			int imode, T eigval, T *eigvect, T rms_target, int Nperiod, int Npose);

		// Write a pml file
		void write_pml(std::string FileName, std::vector<Atoms<T>>& atoms, std::vector<Edges<T>>& List);

	private:

		std::string letters="ABCDEFGHIJKLMNOPQRSTUVWXYZ";

};


/*================================================================================================
 writeBfact: computed and experimental B-factors
================================================================================================== */

 template <typename T>
void WriteResults<T>::writeBfact(std::string filename, std::vector<Atoms<T>>& atoms, T *bfact)
{
	std::ofstream output;
	output.open(filename);
	int Natoms = atoms.size();

	for(int i = 0; i < Natoms; i++)
	{
		output << atoms[i].line.substr(0,30) << "  " << std::setw(10) << atoms[i].bfact << "  ";
		output << std::setw(10) << bfact[i] << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeCorrel: write matrix of correlation of motions
================================================================================================== */

 template <typename T>
void WriteResults<T>::writeCorrel(std::string filename, int Natoms, T *correl)
{
	std::ofstream output;
	output.open(filename);

	for(int i = 0; i < Natoms; i++)
	{
		for(int j = 0; j < Natoms; j++)
		{
			output << correl[Natoms*i+j] << " ";
		}
		output << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeBfact2: contribution of each mode to: b-factor
================================================================================================== */

 template <typename T>
void WriteResults<T>::writeBfact2(std::string filename, int nmode1, int nmodes, T *rms_bfact, 
	T *correl_bfact)
{
	std::ofstream output;
	output.open(filename);

	for(int i = nmode1; i < nmodes; i++)
	{
		output << i +1 << "  ";
		output << std::setw(10) << rms_bfact[i] << "  " << std::setw(10) << correl_bfact[i] << "  ";
		output << std::endl;
	}
 }

/*================================================================================================
 writeOverlap: contribution of each mode to: b-factor, overlap, rms
================================================================================================== */

 template <typename T>
void WriteResults<T>::writeOverlap(std::string filename, int nmode1, int nmodes, int check_bfact, T *rms_bfact, 
	T *correl_bfact, T rms0, T *rms, T *overlap, T *quality, T *cadist)
{
	std::ofstream output;
	output.open(filename);

//	int j = 0;
//	T zero=0;
//	output << j << "  ";
//	if(check_bfact != 0) {
//		output << std::setw(10) << zero << "  " << std::setw(10) << zero << "  ";
//	}
//	output << std::setw(10) << zero << "  " << std::setw(10) << rms0 << std::endl;
	T norm;
	int inc = 1;
	for(int i = nmode1; i < nmodes; i++)
	{
		output << i +1 << "  ";
		if(check_bfact != 0) {
			output << std::setw(10) << rms_bfact[i] << "  " << std::setw(10) << correl_bfact[i] << "  ";
		}
		output << std::setw(10) << overlap[i] << "  ";
		norm = eig_dnrm2_(&i, overlap, &inc);
		output << std::setw(10) << norm << "  ";
		output << std::setw(10) << rms[i] << "  ";
//		output << std::setw(10) << quality[i] << "  ";
//		output << std::setw(10) << cadist[i] <<std::endl;
		output << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeEig: write Eigenvalues and Eigenvectors in a binary file
================================================================================================== */

 template <typename T>
void WriteResults<T>::writeEig(std::string filename, T *eigVal, T *eigVect, int Natoms, int nmodes)
{
	std::ofstream output(filename, std::ios::out | std::ios::binary);

	output.write ((char*)&Natoms, sizeof (int));
	output.write ((char*)&nmodes, sizeof (int));
	output.write ((char*)eigVal, nmodes*sizeof (T));
	output.write ((char*)eigVect, 3*Natoms*nmodes*sizeof (T));

	output.close();
}

/*================================================================================================
 writeTraj: writes trajectory for one normal mode
================================================================================================== */

 template <typename T>
void WriteResults<T>::writeTraj(std::string filename, std::vector<Atoms<T>>& atoms, 
		int Natoms, int imode, T eigval, T *eigvect, T rms_target, int Nperiod, int Npose)
{
	std::ofstream outfile;
	outfile.open(filename);

	int natoms = atoms.size();
	int *idx1 = new int[natoms];
	int *idx2 = new int[natoms];

	for(int iatm = 0; iatm < natoms; iatm++) {
		idx1[iatm] = atoms[iatm].index;
	}

	for(int iatm = 0; iatm < natoms; iatm++) {
		idx2[idx1[iatm]] = iatm;
	}

	T rms1;
	T rms = 0;
	for(int i = 0; i < Natoms; i++) {
		rms1 = 0.0;
		for(int j = 0; j < 3; j++) {
			rms1 += eigvect[3*i+j]*eigvect[3*i+j];
		}
		rms += std::sqrt(rms1);
	}
	rms = rms/Natoms;

	outfile << "REMARK" << std::endl;
	outfile << "REMARK Normal mode considered: " << imode+1 << std::endl;
	outfile << "REMARK" << std::endl;

	T KB_AKMA = 0.00198719;
	T TempN = 293.0;
	T fac = std::sqrt( 2.0*KB_AKMA*TempN);
	T twopi = 2.0*std::acos(-1.0);

	T Tref = 33.35/eigval;
	T delta_t = (Nperiod*Tref)/Npose;

	std::string info;
	std::string newline;
	std::string cname, cname0;
	T newcoord;
	T add;

	fac = fac*rms_target/rms;
	fac = fac/std::sqrt(eigval);

	for(int ipose = 0; ipose < Npose; ipose ++)
	{
		outfile << "REMARK time = " << delta_t*ipose << std::endl;
		outfile << "MODEL " << ipose+1 << std::endl;

		cname0 = atoms[idx2[0]].line.substr(21,2);

		for(int j = 0; j < Natoms; j++)
        	{
			int i = idx2[j];
			cname = atoms[i].line.substr(21,2);
			if(cname != cname0) {
				outfile << "TER" << std::endl;
				cname0 = cname;
			}
			info = atoms[i].line;
			std::ostringstream coord;
			coord.precision(3);
			coord << std::fixed;
			for(int k = 0; k < 3; k++)
			{
				newcoord = atoms[i].coord[k];
				add = eigvect[3*i+k]*fac*
				std::sin(eigval*twopi*delta_t*ipose/33.35);
				newcoord += add;
				coord <<std::fixed << std::setw(8) << newcoord;
			}
			newline = info.substr(0,30) + coord.str();
			
			outfile << newline << std::endl;
        	}
		outfile << "TER" << std::endl;
		outfile << "ENDMDL" << std::endl;
	}
        outfile.close();

	delete [] idx1;
	delete [] idx2;

}

/*================================================================================================
 	writeCIF: write trajectory a CIF file
 ================================================================================================== */

template <typename T>
void WriteResults<T>::trajCIF(std::string filename, std::vector<Atoms<T>>& atoms, int Natoms, 
		int imode, T eigval, T *eigvect, T rms_target, int Nperiod, int Npose)
{
	std::ofstream output;
	output.open(filename);

	int natoms = atoms.size();
	std::string info1, info3;

	output << "data_" << std::endl;
	output << "loop_" << std::endl;
	output << "_atom_site.group_PDB " << std::endl;
	output << "_atom_site.id " << std::endl;
	output << "_atom_site.label_atom_id " << std::endl;
	output << "_atom_site.label_alt_id " << std::endl;
	output << "_atom_site.label_comp_id " << std::endl;
	output << "_atom_site.label_asym_id " << std::endl;
	output << "_atom_site.label_entity_id " << std::endl;
	output << "_atom_site.label_seq_id " << std::endl;
	output << "_atom_site.Cartn_x " << std::endl;
	output << "_atom_site.Cartn_y " << std::endl;
	output << "_atom_site.Cartn_z " << std::endl;
	output << "_atom_site.occupancy" << std::endl;
	output << "_atom_site.B_iso_or_equiv" << std::endl;
	output << "_atom_site.auth_seq_id " << std::endl;
	output << "_atom_site.auth_comp_id " << std::endl;
	output << "_atom_site.auth_asym_id " << std::endl;
	output << "_atom_site.auth_atom_id " << std::endl;
	output << "_atom_site.pdbx_PDB_model_num" << std::endl;

	int *idx1 = new int[natoms];
	int *idx2 = new int[natoms];

	for(int iatm = 0; iatm < natoms; iatm++) {
		idx1[iatm] = atoms[iatm].index;
	}

	for(int iatm = 0; iatm < natoms; iatm++) {
		idx2[idx1[iatm]] = iatm;
	}

	T crd[3];
	T occup, bfact;

	std::string chain="AA";
	std::string chain1, chain_old;
	chain_old = " ";
	int ires = 0;
	std::string s;

	T rms1;
	T rms = 0;
	for(int i = 0; i < Natoms; i++) {
		rms1 = 0.0;
		for(int j = 0; j < 3; j++) {
			rms1 += eigvect[3*i+j]*eigvect[3*i+j];
		}
		rms += std::sqrt(rms1);
	}
	rms = rms/Natoms;

	T KB_AKMA = 0.00198719;
	T TempN = 293.0;
	T fac = std::sqrt( 2.0*KB_AKMA*TempN);
	T twopi = 2.0*std::acos(-1.0);

	T Tref = 33.35/eigval;
	T delta_t = (Nperiod*Tref)/Npose;

	T newcoord;
	T add;

	fac = fac*rms_target/rms;
	fac = fac/std::sqrt(eigval);
	fac = fac/10.;

	int imodel=0;

	for(int ipose = 0; ipose < Npose; ipose ++)
	{
		ires = 0;
		for(int iatm1 = 0; iatm1 < natoms; iatm1++) {

			int iatm = idx2[iatm1];

			for(int k = 0; k < 3; k++) {
				newcoord = atoms[iatm].coord[k];
				add = eigvect[3*iatm1+k]*fac*
				std::sin(eigval*twopi*delta_t*ipose/33.35);
				crd[k] = newcoord+add;
			}

			if(iatm1==0) {
				chain_old = atoms[iatm].line.substr(21,2);
				ires = 0;
			}
			chain1 = atoms[iatm].line.substr(21,2);
			if(chain1!=chain_old) {
				chain_old = chain1;
				ires=0;
			}
			ires++;
			int info2 = 1;
			if(imodel==0) {
				chain = chain1;
			} else {
				if(chain1.substr(1,1)==" ") {
				        chain = chain1.substr(0,1) + "-" + std::to_string(imodel);
				} else {
				        chain = chain1 + "-" + std::to_string(imodel);
				}
			}
			info1 = atoms[iatm].line.substr(13,4);
			info3 = atoms[iatm].line.substr(17,4);
			occup = 1.0;
			bfact = atoms[iatm].bfact;

			output << "ATOM ";
			output << iatm1+1 << " ";
			output << info1;
			output << ". ";
			output << info3;
			output << chain;
			output << " ";
			output << info2;
			output << " ";
			output << ires;
			output << " ";
			output << std::fixed << std::setprecision(3) << std::setw(8) << crd[0];
			output << " ";
			output << std::fixed << std::setprecision(3) << std::setw(8) << crd[1];
			output << " ";
			output << std::fixed << std::setprecision(3) << std::setw(8) << crd[2];
			output << " ";
			output << std::fixed << std::setprecision(2) << std::setw(6) << occup;
			output << " ";
			output << std::fixed << std::setprecision(2) << std::setw(6) << bfact;
			output << " ";
			output << ires;
			output << " ";
			output << info3;
			output << " ";
			output << chain;
			output << " ";
			output << info1;
			output << std::to_string(ipose+1);
			output << std::endl;

			if(atoms[iatm].endmdl==1) imodel++;

		}

	}

//	output << "#" << std::endl;

	output.close();
}

/* ===============================================================================================
   Write pml file to visualize elastic network
   =============================================================================================== */

  template <typename T>
   void WriteResults<T>::write_pml(std::string FileName, std::vector<Atoms<T>>& atoms, std::vector<Edges<T>>& List)
  {

	std::ofstream outfile(FileName.c_str());

/* 	==========================================================================================
   	Header for pymol file
   	========================================================================================== */

	outfile << "python" << std::endl;
	outfile << " " << std::endl;

	outfile << "from pymol.cgo import *" << std::endl;
	outfile << "from pymol import cmd" << std::endl;
	outfile << " " << std::endl;

/* 	==========================================================================================
   	Write all active atoms as spheres
   	========================================================================================== */

	int natoms = atoms.size();

	outfile << "vertex = [" << "\n" << "  COLOR, 0.0, 0.0, 1.0," << std::endl;
	outfile << " " << std::endl;

	double radius = 0.5;
	double x, y, z;
	for(int i = 0; i < natoms; i++) 
	{
		x = atoms[i].coord[0];
                y = atoms[i].coord[1];
                z = atoms[i].coord[2];

		if(atoms[i].mass != 0){
			outfile  << "   SPHERE, " << std::setprecision(8) << x << ", " << 
			y << ", " << z <<", " << radius << "," << std::endl;
		}
	}
	outfile << "   ]" << std::endl;
	outfile << " " << std::endl;
	outfile << " " << std::endl;

/* 	==========================================================================================
   	Write all active links as lines
   	========================================================================================== */

	int nedges = List.size();
//	outfile << "links = [" << "\n" << "   BEGIN, LINES," << "\n" << "   COLOR, 1.0, 0.0, 0.0," << std::endl;
	outfile << "links = [" << "\n" << "   BEGIN, LINES," << "\n" << std::endl;
	outfile << " " << std::endl;

	int i1,j1;
	for(int pair = 0; pair < nedges; pair++)
	{
		i1 = List[pair].atm1;
		j1 = List[pair].atm2;

		x = atoms[i1].coord[0];
                y = atoms[i1].coord[1];
                z = atoms[i1].coord[2];
		outfile  << "   VERTEX, " << std::setprecision(8) << x << ", " << y << ", " << z << "," <<std::endl;
		x = atoms[j1].coord[0];
                y = atoms[j1].coord[1];
                z = atoms[j1].coord[2];
		outfile  << "   VERTEX, " << std::setprecision(8) << x << ", " << y << ", " << z << "," <<std::endl;
		outfile << " " << std::endl;

	}
	outfile << " " << std::endl;
	outfile << "   END" << "\n"<< "   ]" << std::endl;
	outfile << " " << std::endl;
	outfile << " " << std::endl;

	outfile << "cmd.load_cgo(vertex,\"vertex\")" << std::endl;
	outfile << "cmd.load_cgo(links,\"links\")" << std::endl;

	outfile << " " << std::endl;
	outfile << "python end" << std::endl;

	outfile.close();

  }

#endif
