#include "GLWidget.h"
#include <QDebug>
#include <fstream>
#include <vector>

GLWidget::GLWidget(const QGLFormat &format, QWidget *parent) :
    QGLWidget(format, parent)
{
    setFocusPolicy(Qt::ClickFocus);
    setMinimumSize(256, 256);
    setWindowTitle("Volume Ray-casting");
}

void GLWidget::initializeGL()
{
#ifndef __APPLE__
    glewExperimental = GL_TRUE;
    GLenum err = glewInit();
    if (GLEW_OK != err)
        qDebug() << glewGetErrorString(err);
#endif

    glClearColor(1.0, 1.0, 1.0, 0.0);
    glEnable(GL_BLEND);
    glBlendEquation(GL_FUNC_ADD);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_CULL_FACE);
    glCullFace(GL_FRONT);
    glDisable(GL_DEPTH_TEST);
    glColorMask(true, true, true, true);
    glDepthMask(true);

    // initialize transformation
    rotation = Mat3f::identity();
    translation = Vec3f(0.0f, 0.0f, -2.0f);

    // load sample data volume
    volumeDim.x = 32;
    volumeDim.y = 32;
    volumeDim.z = 32;
    int volumeSize = volumeDim.x * volumeDim.y * volumeDim.z;
    std::vector<unsigned char> volumeData;
    volumeData.resize(volumeSize);
    std::ifstream ifs("SampleVolume.raw", std::ios::binary);
    if(!ifs.is_open())
    {
        qDebug() << "Failed to open sample volume!";
        close();
    }

    ifs.read(reinterpret_cast<char *>(&volumeData.front()), volumeSize * sizeof(unsigned char));
    ifs.close();

    // prepare transfer function data
    std::vector<Vec4f> transferFuncData;
    transferFuncData.push_back(Vec4f(1.0f, 0.0f, 0.0f, 0.0f));
    transferFuncData.push_back(Vec4f(1.0f, 1.0f, 0.0f, 0.1f));
    transferFuncData.push_back(Vec4f(0.0f, 1.0f, 0.0f, 0.2f));
    transferFuncData.push_back(Vec4f(0.0f, 1.0f, 1.0f, 0.3f));
    transferFuncData.push_back(Vec4f(0.0f, 0.0f, 1.0f, 0.4f));
    transferFuncData.push_back(Vec4f(1.0f, 0.0f, 1.0f, 0.5f));

    // create OpenGL textures
    volumeTex = GLTexture::create();
    volumeTex->updateTexImage3D(GL_R8, volumeDim, GL_RED, GL_UNSIGNED_BYTE, &volumeData.front());

    transferFuncTex = GLTexture::create();
    transferFuncTex->updateTexImage2D(GL_RGBA32F, Vec2i(int(transferFuncData.size()), 1), GL_RGBA, GL_FLOAT, &transferFuncData.front());

    // create full screen rectangle for volume ray-casting
    std::vector<Vec2f> rectVertices;
    rectVertices.push_back(Vec2f(0.0f, 0.0f));
    rectVertices.push_back(Vec2f(0.0f, 1.0f));
    rectVertices.push_back(Vec2f(1.0f, 1.0f));
    rectVertices.push_back(Vec2f(1.0f, 0.0f));
    rectVertexBuffer = GLArrayBuffer::create(GL_ARRAY_BUFFER);
    rectVertexBuffer->update(rectVertices.size() * sizeof(Vec2f), &rectVertices.front(), GL_STATIC_DRAW);
    rectVertexArray = GLVertexArray::create();

    // create OpenGL shaders
    volumeRayCastingProgram = GLShaderProgram::create("VolumeRayCasting.vert", "VolumeRayCasting.frag");
    if(volumeRayCastingProgram == NULL)
        close();

    // assign vertex buffer to the shader attribute input called "Vertex"
    volumeRayCastingProgram->setVertexAttribute("Vertex", rectVertexArray, rectVertexBuffer, 2, GL_FLOAT, false);
}

void GLWidget::resizeGL(int w, int h)
{
    glViewport(0, 0, w, h);
    projectionMatrix = Mat4f::perspective(Math::pi<float>() * 0.25f, float(w) / float(h), 0.01f, 10.0f);
}

void GLWidget::paintGL()
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    Vec3f volumeScale = Vec3f(volumeDim);
    volumeScale = volumeScale / volumeScale.norm();
    modelViewMatrix = Mat4f(rotation, translation);
    Mat4f transform = Mat4f::ortho(0, 1, 0, 1, -1, 1);
    Mat4f invModelView = modelViewMatrix.inverse();
    volumeRayCastingProgram->setUniform("volumeScale", volumeScale);
    volumeRayCastingProgram->setUniform("transform", transform);
    volumeRayCastingProgram->setUniform("invModelView", invModelView);
    volumeRayCastingProgram->setUniform("aspect", float(width()) / float(height()));
    volumeRayCastingProgram->setUniform("cotFOV", projectionMatrix.elem[1][1]);
    volumeRayCastingProgram->setTexture("volumeTex", volumeTex);
    volumeRayCastingProgram->setTexture("transferFuncTex", transferFuncTex);
    volumeRayCastingProgram->begin();
    rectVertexArray->drawArrays(GL_TRIANGLE_FAN, 4);
    volumeRayCastingProgram->end();
}

void GLWidget::mouseMoveEvent(QMouseEvent *event)
{
    QPointF pos = event->localPos();
    float dx = pos.x() - mousePos.x();
    float dy = pos.y() - mousePos.y();
    mousePos = pos;

    Qt::MouseButtons mouseButtons = event->buttons();

    if(mouseButtons & Qt::LeftButton)
    {
        rotation = Mat3f::fromAxisAngle(Vec3f::unitY(), dx * 0.005f) * rotation;
        rotation = Mat3f::fromAxisAngle(Vec3f::unitX(), dy * 0.005f) * rotation;
        rotation.orthonormalize();
    }

    if(mouseButtons & Qt::RightButton)
    {
        translation.z += dy * 0.005f;
        translation.z = clamp(translation.z, -9.0f, 1.0f);
    }

    if(mouseButtons & Qt::MidButton)
    {
        float scale = -std::min(translation.z, 0.0f) * 0.001f + 0.000025f;
        translation.x += dx * scale;
        translation.y -= dy * scale;
    }

    updateGL();
}

void GLWidget::mousePressEvent(QMouseEvent *event)
{
    mousePos = event->localPos();
}

void GLWidget::mouseReleaseEvent(QMouseEvent *event)
{
    mousePos = event->localPos();
}

void GLWidget::wheelEvent(QWheelEvent * event)
{
    QPoint numPixels = event->pixelDelta();
    QPoint numDegrees = event->angleDelta() / 8;
    float dx = 0.0f, dy = 0.0f;
    if(!numPixels.isNull())
    {
        dx = numPixels.x();
        dy = numPixels.y();
    }
    else if(!numDegrees.isNull())
    {
        dx = numDegrees.x() / 15;
        dy = numDegrees.y() / 15;
        dy = numDegrees.y() * -1;
    }

    Qt::KeyboardModifiers km = QApplication::keyboardModifiers();

    if(km & Qt::ControlModifier)
    {
        rotation = Mat3f::fromAxisAngle(Vec3f::unitY(), dx * 0.005f) * rotation;
        rotation = Mat3f::fromAxisAngle(Vec3f::unitX(), dy * 0.005f) * rotation;
    }
    else if(km & Qt::AltModifier)
    {
        float scale = -std::min(translation.z, 0.0f) * 0.001f + 0.000025f;
        translation.x += dx * scale;
        translation.y -= dy * scale;
    }
    else// if(km & Qt::ShiftModifier)
    {
        translation.z += dy * 0.005f;
        translation.z = clamp(translation.z, -9.0f, 1.0f);
    }

    updateGL();
}
