#ifndef lint
static char     sccsid[] = "@(#)linda_calls.c";
static char     Creator[] = "C 1991 G. Sxoinas csi.forth.gr \n";
static char     Mail[] = "sxoinas@ariadne.bitnet or sxoinas@csi.forth.gr";
#endif
/*
 * This file is part of the POSYBL (PrOgramming SYstem for distriButed
 * appLications, a free Linda implementation for Unix Networks) and contains
 * the linda function calls.
 * 
 * There are no restrictions on this code; however, if you make any changes, I
 * request that you document them so that I do not get credit or blame for your
 * modifications.
 * 
 * Written by Ioannis Schoinas (sxoinas@csd.uch.gr or sxoinas@csi.forth.gr),
 * Computer Science Department, University of Crete, Heraclion, Crete, Greece.
 * 
 * Note: Linda is a trademark of SCA, Inc.
 * 
 */
#include <stdio.h>
#include <varargs.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <rpcsvc/rstat.h>

#include "linda_defs.h"

extern char    *skip_field();

#define  INITIAL_FIELD_BUFFER_SIZE 32
#define  FIELD_BUFFER_INCREMENT 4
#define  HIGH_FIELD_BUFFER_WATERMARK 64


static int      LocalSocket;
static struct timeval LocalCallTimeOut = {LINDA_RETRY_LOCAL, 0};
static struct sockaddr_un LocalAddress;
static NODEINFO NodesInfoBuffer[32];

static int      SystemInitialized = FALSE;
static int      Pid, Uid, RunGroup;
static char    *LocalServerSocketName;
static int      NodesNumber, CurrentNode = 0;
static union
{
  TUPLE_MESSAGE   Tuple;
  RESPONSE_MESSAGE Response;
}               Buffer;
static TEMPLATE_MESSAGE TemplateBuffer;
static struct buffer
{
  FIELD          *buffer;
  int             num, lim;
}               FieldsBuffer =
{
  NULL, 0, 0
};

static
int
Null()
{
  return TRUE;
}

static
init_field_buffer()
{
  register int    size;

  if (FieldsBuffer.buffer == NULL)
  {
    size = sizeof(FIELD) * INITIAL_FIELD_BUFFER_SIZE;
    FieldsBuffer.buffer = (FIELD *) malloc(size);
    FieldsBuffer.lim = INITIAL_FIELD_BUFFER_SIZE;
  }
}

static
int
put_field_ptr(ptr)
  register FIELD *ptr;
{
  register int    newsize, pos;

  if (FieldsBuffer.num == FieldsBuffer.lim)
  {
    FieldsBuffer.lim += FIELD_BUFFER_INCREMENT;
    newsize = FieldsBuffer.lim * sizeof(FIELD);
    FieldsBuffer.buffer = (FIELD *) realloc(FieldsBuffer.buffer, newsize);
  }
  pos = FieldsBuffer.num;
  FieldsBuffer.buffer[pos] = *ptr;
  FieldsBuffer.num++;
  return (pos);
}

static void
reset_field_buffer()
{
  register int    newsize;

  if (FieldsBuffer.num > HIGH_FIELD_BUFFER_WATERMARK)
  {
    FieldsBuffer.lim = HIGH_FIELD_BUFFER_WATERMARK;
    newsize = HIGH_FIELD_BUFFER_WATERMARK * sizeof(FIELD);
    FieldsBuffer.buffer = (FIELD *) realloc(FieldsBuffer.buffer, newsize);
  }
  FieldsBuffer.num = 0;
}

static int
compare(i, j)
  register NODEINFO *i, *j;
{
  return (i->load - j->load);
}

static
void
UpdateLoadInfo()
{
  register int    i;
#ifdef DYNAMIC
  struct statstime statp;
#endif

  for (i = 0; i < NodesNumber; i++)
  {
#ifdef DYNAMIC
    if (rstat(NodesInfoBuffer[i].name, &statp) == 0)
      NodesInfoBuffer[i].load = (-((float) statp.avenrun[0] + (float) statp.avenrun[1] / 5.0)) * NodesInfoBuffer[i].arch;
    else
      NodesInfoBuffer[i].load = 10E+20;
#else
    NodesInfoBuffer[i].load = NodesInfoBuffer[i].arch;
#endif
  }
  qsort(NodesInfoBuffer, NodesNumber, sizeof(NodesInfoBuffer[0]), compare);
  CurrentNode = 0;
}

void
init(rungroup, nodefile)	/* Defines the RunGroup under which the program
				 * will run */
  int             rungroup;
  char           *nodefile;
{
  int             port;

  SystemInitialized = TRUE;
  LocalServerSocketName = (char *) FindLocalName(SOCKNAME_LOCAL);
  Pid = getpid();
  Uid = getuid();
  RunGroup = rungroup;
  LocalSocket = Rpc_UnixSTCreate(False, NULL);
  LocalAddress.sun_family = AF_UNIX;
  strcpy(LocalAddress.sun_path, LocalServerSocketName);
  port = GetPort(LocalSocket, &LocalAddress);
  LoadNodes(nodefile, NodesInfoBuffer, &NodesNumber);
  UpdateLoadInfo();
  init_field_buffer();
}

void
in(va_alist) va_dcl		/* Linda's in operation */
{
  va_list         pvar;
  register FIELD *field;
  register int    i, msglen, FieldsNumber = FieldsBuffer.num;
  register Rpc_Stat error;
  register char   exist_formals = FALSE, *ptr;
  register char  *msgdata = (char *) TemplateBuffer.message_data;
  register RESPONSE_MESSAGE *reply = &Buffer.Response;

  va_start(pvar);
  if (!SystemInitialized)
    init(0, NULL);
  /* make header */
  TemplateBuffer.header.pid = Pid;
  TemplateBuffer.header.uid = Uid;
  TemplateBuffer.header.rungroup = RunGroup;

  /* make key */
  TemplateBuffer.message_data[0] = FieldsNumber;
  msglen = sizeof(int);
  field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
  if (isformal(field->id))
  {
    fprintf(stderr, "in: First field must be always actual, failed...\n");
    reset_field_buffer();
    return;
  }
  TemplateBuffer.message_data[1] = actual(field->id);
  msglen += sizeof(int);
  msglen += copydata(msgdata + msglen, field);
  for (i = 1; i < FieldsNumber; i++)
  {
    field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
    *((int *) (msgdata + msglen)) = actual(field->id);
    msglen += sizeof(int);
  }
  va_end(pvar);
  TemplateBuffer.header.keybytes = msglen;

  /* make data */
  va_start(pvar);
  va_arg(pvar, int);
  for (i = 1; i < FieldsNumber; i++)
  {
    field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
    *((int *) (msgdata + msglen)) = field->id;
    msglen += sizeof(int);
    if (!isformal(field->id))
      msglen += copydata(msgdata + msglen, field);
    else
      exist_formals = TRUE;
  }
  va_end(pvar);
  TemplateBuffer.header.databytes = msglen - TemplateBuffer.header.keybytes;
  msglen += sizeof(TEMPLATE_HEADER);

  /* call the LindaIn procedure in the local Tuple Manager */
  if ((error = Rpc_Call(LocalSocket, &LocalAddress, (Rpc_Proc) LINDA_IN,
			msglen, &TemplateBuffer,
			sizeof(RESPONSE_MESSAGE), reply,
			LINDA_NRETRY_LOCAL, &LocalCallTimeOut)) != RPC_SUCCESS)
  {
    fprintf(stderr, "in: Rpc_Call: %s\n", Rpc_ErrorMessage(error));
    exit(1);
  }
  ptr = (char *) reply->message_data;

  /* Copy data from the tuple to the user variables */
  if (exist_formals)
  {
    va_start(pvar);
    va_arg(pvar, int);
    for (i = 1; i < FieldsNumber; i++)
    {
      field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
      if (isformal(field->id))
	rcopydata(field, ptr + sizeof(int));
      ptr = skip_field(ptr);
    }
    va_end(pvar);
  }
  reset_field_buffer();
}

void
rd(va_alist) va_dcl		/* Linda's rd operation */
{
  va_list         pvar;
  register FIELD *field;
  register int    i, msglen, FieldsNumber = FieldsBuffer.num;
  register Rpc_Stat error;
  register char   exist_formals = FALSE, *ptr;
  register char  *msgdata = (char *) TemplateBuffer.message_data;
  register RESPONSE_MESSAGE *reply = &Buffer.Response;

  va_start(pvar);
  if (!SystemInitialized)
    init(0, NULL);

  /* make header */
  TemplateBuffer.header.pid = Pid;
  TemplateBuffer.header.uid = Uid;
  TemplateBuffer.header.rungroup = RunGroup;
  TemplateBuffer.message_data[0] = FieldsNumber;
  msglen = sizeof(int);

  /* make key */
  field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
  TemplateBuffer.message_data[1] = actual(field->id);
  msglen += sizeof(int);
  msglen += copydata(msgdata + msglen, field);
  if (isformal(field->id))
  {
    fprintf(stderr, "rd: First field must be always actual, failed...\n");
    reset_field_buffer();
    return;
  }
  for (i = 1; i < FieldsNumber; i++)
  {
    field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
    *((int *) (msgdata + msglen)) = actual(field->id);
    msglen += sizeof(int);
  }
  va_end(pvar);
  TemplateBuffer.header.keybytes = msglen;

  /* make data */
  va_start(pvar);
  va_arg(pvar, int);
  for (i = 1; i < FieldsNumber; i++)
  {
    field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
    *((int *) (msgdata + msglen)) = field->id;
    msglen += sizeof(int);
    if (!isformal(field->id))
      msglen += copydata(msgdata + msglen, field);
    else
      exist_formals = TRUE;
  }
  va_end(pvar);
  TemplateBuffer.header.databytes = msglen - TemplateBuffer.header.keybytes;
  msglen += sizeof(TEMPLATE_HEADER);

  /* Call the local Tuple Manager */
  if ((error = Rpc_Call(LocalSocket, &LocalAddress, (Rpc_Proc) LINDA_READ,
			msglen, &TemplateBuffer,
			sizeof(RESPONSE_MESSAGE), reply,
			LINDA_NRETRY_LOCAL, &LocalCallTimeOut)) != RPC_SUCCESS)
  {
    fprintf(stderr, "rd: Rpc_Call: %s\n", Rpc_ErrorMessage(error));
    exit(1);
  }
  ptr = (char *) reply->message_data;

  /* Copy the tuple's data to the user variables */
  if (exist_formals)
  {
    va_start(pvar);
    va_arg(pvar, int);
    for (i = 1; i < FieldsNumber; i++)
    {
      field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
      if (isformal(field->id))
	rcopydata(field, ptr + sizeof(int));
      ptr = skip_field(ptr);
    }
    va_end(pvar);
  }
  reset_field_buffer();
}

void
out(va_alist) va_dcl		/* Linda's out operation */
{
  va_list         pvar;
  register int    i, FieldsNumber = FieldsBuffer.num, msglen;
  register Rpc_Stat error;
  register FIELD *field;
  register char  *msgdata = (char *) Buffer.Tuple.message_data;

  va_start(pvar);
  if (!SystemInitialized)
    init(0, NULL);

  /* make header */
  Buffer.Tuple.header.uid = Uid;
  Buffer.Tuple.header.rungroup = RunGroup;

  /* make key */
  Buffer.Tuple.message_data[0] = FieldsNumber;
  msglen = sizeof(int);
  for (i = 0; i < FieldsNumber; i++)
  {
    field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
    if (isformal(field->id))
    {
      fprintf(stderr, "out: Formal field not allowed\n");
      reset_field_buffer();
      return;
    }
    *((int *) (msgdata + msglen)) = field->id;
    msglen += sizeof(int);
    if (i == 0)
      msglen += copydata(msgdata + msglen, field);
  }
  va_end(pvar);
  Buffer.Tuple.header.keybytes = msglen;

  /* make data */
  va_start(pvar);
  va_arg(pvar, int);
  for (i = 1; i < FieldsNumber; i++)
  {
    field = (FIELD *) & FieldsBuffer.buffer[va_arg(pvar, int)];
    *((int *) (msgdata + msglen)) = field->id;
    msglen += sizeof(int);
    msglen += copydata(msgdata + msglen, field);
  }
  va_end(pvar);
  Buffer.Tuple.header.databytes = msglen - Buffer.Tuple.header.keybytes;
  msglen += sizeof(TUPLE_HEADER);

  if ((error = Rpc_Call(LocalSocket, &LocalAddress, (Rpc_Proc) LINDA_OUT,
			msglen, &Buffer.Tuple,
	      0, NULL, LINDA_NRETRY_LOCAL, &LocalCallTimeOut)) != RPC_SUCCESS)
  {
    fprintf(stderr, "out: Rpc_Call: %s\n", Rpc_ErrorMessage(error));
    exit(1);
  }
  reset_field_buffer();
}

eval_nl(va_alist) va_dcl	/* eval(node, program, argv[1],
				 * argv[2],...,NULL) */
{
  va_list         pvar;
  register int    i;
  register char  *fname, *node, *str, *command[64];

  va_start(pvar);
  if (!SystemInitialized)
    init(0, NULL);
  command[0] = (char *) va_arg(pvar, char *);
  command[1] = "-n";
  command[2] = (char *) FixName(command[0], (char *) va_arg(pvar, char *),
				NodesInfoBuffer, NodesNumber);
  for (i = 0; (str = (char *) va_arg(pvar, char *)) != NULL; i++)
    command[i + 3] = (char *) FixName(command[0], (char *) va_arg(pvar, char *),
				      NodesInfoBuffer, NodesNumber);
  command[i + 3] = NULL;
  fprintf(stderr, "Evaluating %s on node: %s\n", command[2], command[0]);
  if (vfork() == 0)
    execvp("rsh", command);
  else
  {
    for (i = 0; command[i] != NULL; i++)
      free(command[i + 2]);
  }
  va_end(pvar);
}

eval_nv(node, argv)		/* eval(node, argv) */
  char           *node, **argv;
{
  register int    i;
  register char  *command[64];

  if (!SystemInitialized)
    init(0, NULL);
  command[0] = node;
  command[1] = "-n";
  for (i = 0; argv[i] != NULL; i++)
    command[i + 2] = (char *) FixName(node, argv[i],
				      NodesInfoBuffer, NodesNumber);
  command[i + 2] = NULL;
  fprintf(stderr, "Evaluating %s on node: %s\n", command[2], command[0]);
  if (vfork() == 0)
    execvp("rsh", command);
  else
  {
    for (i = 0; argv[i] != NULL; i++)
      free(command[i + 2]);
  }
}

eval_v(argv)			/* eval(argv) */
  char          **argv;
{
  if (!SystemInitialized)
    init(0, NULL);
  if (NodesInfoBuffer[CurrentNode].load < 10E+20)
  {
#ifdef DEBUG
    printf("Node: %s Load: %f\n", NodesInfoBuffer[CurrentNode].name, NodesInfoBuffer[CurrentNode].load);
#endif
    eval_nv(NodesInfoBuffer[CurrentNode].name, argv);
    if (++CurrentNode == NodesNumber)
      UpdateLoadInfo();
  }
  else
  {
    UpdateLoadInfo();
    eval_v(argv);
  }
}

eval_l(va_alist) va_dcl		/* eval(program, argv[1],...,NULL) */
{
  va_list         pvar;
  register char  *ptr[64], *str;
  register int    i = 0;

  va_start(pvar);
  if (!SystemInitialized)
    init(0, NULL);
  do
  {
    str = (char *) va_arg(pvar, char *);
    ptr[i++] = str;
  } while (str != NULL);
  eval_v(ptr);
  va_end(pvar);
}

/* From here to the end of file follow the field description functions */

int
lchar(c)
  register char   c;
{
  FIELD           f;

  init_field_buffer();
  f.id = CHAR;
  *(char *) f.data = c;
  return (put_field_ptr(&f));
}

int
lshort(s)
  register short  s;
{
  FIELD           f;

  init_field_buffer();
  f.id = SHORT;
  *(short *) f.data = s;
  return (put_field_ptr(&f));
}

int
lint(i)
  register int    i;
{
  FIELD           f;

  init_field_buffer();
  f.id = INT;
  *(int *) f.data = i;
  return (put_field_ptr(&f));
}

int
lfloat(i)
  register double i;
{
  FIELD           f;

  init_field_buffer();
  f.id = FLOAT;
  *(float *) f.data = (float) i;
  return (put_field_ptr(&f));
}

int
ldouble(i)
  double          i;
{
  FIELD           f;

  init_field_buffer();
  f.id = DOUBLE;
  *((double *) f.data) = i;
  return (put_field_ptr(&f));
}

int
lstring(s)
  register char  *s;
{
  FIELD           f;

  init_field_buffer();
  f.id = CHAR_STRING;
  f.data[0] = (int) s;
  return (put_field_ptr(&f));
}

int
lnchar(a, l)
  register char  *a;
  register int    l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_CHAR;
  f.data[0] = (int) a;
  f.data[1] = l;
  return (put_field_ptr(&f));
}

int
lnshort(a, l)
  register short *a;
  register int    l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_SHORT;
  f.data[0] = (int) a;
  f.data[1] = l;
  return (put_field_ptr(&f));
}


int
lnint(a, l)
  register int   *a, l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_INT;
  f.data[0] = (int) a;
  f.data[1] = l;
  return (put_field_ptr(&f));
}

int
lndouble(a, l)
  register double *a;
  register int    l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_DOUBLE;
  f.data[0] = (int) a;
  f.data[1] = l;
  return (put_field_ptr(&f));
}

int
lnfloat(a, l)
  register float *a;
  register int    l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_FLOAT;
  f.data[0] = (int) a;
  f.data[1] = l;
  return (put_field_ptr(&f));
}

int
qlchar(c)
  register char  *c;
{
  FIELD           f;

  init_field_buffer();
  f.id = CHAR_Q;
  f.data[0] = (int) c;
  return (put_field_ptr(&f));
}

int
qlshort(s)
  register short *s;
{
  FIELD           f;

  init_field_buffer();
  f.id = SHORT_Q;
  f.data[0] = (int) s;
  return (put_field_ptr(&f));
}

int
qlint(i)
  register int   *i;
{
  FIELD           f;

  init_field_buffer();
  f.id = INT_Q;
  f.data[0] = (int) i;
  return (put_field_ptr(&f));
}

int
qlfloat(i)
  register float *i;
{
  FIELD           f;

  init_field_buffer();
  f.id = FLOAT_Q;
  f.data[0] = (int) i;
  return (put_field_ptr(&f));
}

int
qldouble(i)
  register double *i;
{
  FIELD           f;

  init_field_buffer();
  f.id = DOUBLE_Q;
  f.data[0] = (int) i;
  return (put_field_ptr(&f));
}

int
qlstring(s)
  register char **s;
{
  FIELD           f;

  init_field_buffer();
  f.id = CHAR_STRING_Q;
  f.data[0] = (int) s;
  return (put_field_ptr(&f));
}

int
qlnchar(a, l)
  register char **a;
  register int   *l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_CHAR_Q;
  f.data[0] = (int) a;
  f.data[1] = (int) l;
  return (put_field_ptr(&f));
}

int
qlnshort(a, l)
  register short **a;
  register int   *l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_SHORT_Q;
  f.data[0] = (int) a;
  f.data[1] = (int) l;
  return (put_field_ptr(&f));
}

int
qlnint(a, l)
  register int  **a;
  register int   *l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_INT_Q;
  f.data[0] = (int) a;
  f.data[1] = (int) l;
  return (put_field_ptr(&f));
}

int
qlnfloat(a, l)
  register float **a;
  register int   *l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_FLOAT_Q;
  f.data[0] = (int) a;
  f.data[1] = (int) l;
  return (put_field_ptr(&f));
}

int
qlndouble(a, l)
  register double **a;
  register int   *l;
{
  FIELD           f;

  init_field_buffer();
  f.id = ARRAY_OF_DOUBLE_Q;
  f.data[0] = (int) a;
  f.data[1] = (int) l;
  return (put_field_ptr(&f));
}
