

// ESim sample program:  Stop-and-Wait protocol

// usage:  stop_and_wait time_limit debug_flag alpha timeout p mean_delay
// where
//
//    alpha = (one-way) propagation delay (frame transmit time = 1.0)
//    timeout = time station A waits before giving up and retransmitting
//    p = probability that Station B finds frame erroneous
//    mean_delay = mean of delay before B can start sending back to A

#include <stdio.h>
#include <string>
#include <cstdlib>

#include <ESimDefs.h>

int NextFrameToBeCreated = 0;  
int TotFrames = 0;  // number of jobs done so far
int TotTries = 0;

float TotWait = 0.0;  // total wait for those jobs

float Alpha,Timeout,P,MeanDelay;

class FrameElt: public ESElt  {
   public:
      int FrameNumber;
      float StartTime;  // time at which this frame is first sent
      int NTries;  // number of attempts at sending this frame so far
      int Trashed;  // 1 means erroneous
      int TimeoutEltNumber;  // ESEltNumber for the paired timeout
      FrameElt();
      void ESPrintElt();
};

void FrameElt::ESPrintElt()

{  ESElt::ESPrintElt();
   printf("  frame %d, number of tries = %d, trashed = %d\n",
      this->FrameNumber,this->NTries,this->Trashed);
}

FrameElt::FrameElt() : ESElt() { ; }

class TimeoutElt: public ESElt  {
   public:
      int FrameEltNumber;  // ESEltNumber for the paired timeout
      TimeoutElt();
      void ESPrintElt()
         {  ESElt::ESPrintElt(); }
};

TimeoutElt::TimeoutElt() : ESElt() { ; }

FrameElt Frame;
TimeoutElt TmOt;

// send the frame and set the timer; if this is the first xmit of the
// frame, set New = 1

void SendFrame(int New)

{  strcpy(Frame.ESName,"arrive at B");
   Frame.ESEvntTime = ESSim::ESSimTime + 1 + Alpha;
   if (New)  {
      Frame.FrameNumber = NextFrameToBeCreated++;
      Frame.StartTime = Frame.ESEvntTime;
      Frame.NTries = 1;
   }
   else Frame.NTries++;
   Frame.Trashed = 0;
   Frame.ESInsertInSchedList();
   // set up the paired timeout
   TmOt.ESEvntTime = ESSim::ESSimTime + Timeout;
   strcpy(TmOt.ESName,"timeout");
   TmOt.ESInsertInSchedList(); 
   // pair them together to enable one to cancel the other
   Frame.TimeoutEltNumber = TmOt.ESEltNumber;
   TmOt.FrameEltNumber = Frame.ESEltNumber;
}

void DoArrivalAtB()

{  strcpy(ESElt::ESCurrEvnt->ESName,"done with delay");
   Frame.ESEvntTime = ESSim::ESSimTime + ESSim::ESExpon(MeanDelay);
   Frame.ESInsertInSchedList();
}

void DoDelayDone()

{  Frame.Trashed = (ESSim::ESRnd() < P);
   strcpy(ESElt::ESCurrEvnt->ESName,"arrive back at A");
   Frame.ESEvntTime = ESSim::ESSimTime + Alpha;
   Frame.ESInsertInSchedList();
}

void DoArrivalBackToA()

{  ESElt::ESCancel(Frame.TimeoutEltNumber);
   if (!Frame.Trashed)  {
      TotFrames++;
      TotWait += ESSim::ESSimTime - Frame.StartTime;
      TotTries += Frame.NTries;
      SendFrame(1);
   }
   else SendFrame(0);

}
 
void DoTimeout()

{  ESElt::ESCancel(TmOt.FrameEltNumber);
   SendFrame(0);
}

void ESElt::ESEvntHandler() 

{  

   if (ESElt::ESEvntMatch("arrive at B")) 
      {DoArrivalAtB(); return;}
   if (ESElt::ESEvntMatch("done with delay")) 
      {DoDelayDone(); return;}
   if (ESElt::ESEvntMatch("arrive back at A")) 
      {DoArrivalBackToA(); return;}
   if (ESElt::ESEvntMatch("timeout")) 
      {DoTimeout(); return;}
   printf("shouldn't reach this point\n");
   exit(1);
}

int main(int Argc, char **Argv) 

{  float Tmp;

   ESSim::ESInit(Argv);

   sscanf(Argv[3],"%f",&Alpha);
   sscanf(Argv[4],"%f",&Timeout);
   sscanf(Argv[5],"%f",&P);
   sscanf(Argv[6],"%f",&MeanDelay);

   // set up first xmit
   SendFrame(1);
   
   ESSim::ESMainLoop(Argv);
   
   printf("total frames sent = %d\n",TotFrames); 
   printf("mean wait = %f\n",TotWait/TotFrames); 
   printf("mean number of xmits per frame = %f\n",(float) TotTries/TotFrames); 
}
