

// JSim sample program:  Stop-and-Wait protocol; paired with
// FrameElt.java, representing one transmitted frame, and
// TimeOutElt.java, representing a timer for a timeout period

// Network node A is sending to Network node B.  It takes 1.0 time to
// get the frame onto the network, and alpha time for the leading edge
// of the frame to reach B.  (The trailing edge arrives 1.0 time later.)
// When B receives the frame, there will be a delay before B can check
// it.  Then B will check for errors, and send back to A either an
// acknowledgement, if the frame is OK, or an error report, if not OK.
// B's response is so short, it takes negligible time to get onto the
// link, but still takes 1.0 time to get back to A.  As soon as A starts
// sending the frame, it starts a timer.  If the timeout is exceeded, or
// if A receives a report from B that the frame was in error, A will
// send the frame again.

// usage:  java StopAndWait timelimit debugflag alpha timeout p meandelay
//
// where
//
//    alpha = (one-way) propagation delay (frame transmit time = 1.0)
//    timeout = time station A waits before giving up and retransmitting
//    p = probability that Station B finds frame erroneous
//    meandelay = mean of delay before B can start sending back to A

public class Main // required

{

   public static int NextFrameToBeCreated = 0;  
   public static int TotFrames = 0;  // number of jobs done so far
   public static int TotTries = 0;  // number of tries among all jobs so far

   public static float TotWait = (float) 0.0;  // total wait for those jobs

   public static float Alpha,Timeout,P,MeanDelay;

   public static FrameElt Frame;
   public static TimeoutElt TmOt;

   public static void main(String[] Argv) 

   {  float Tmp;

      JSSim.JSInit(Argv);  // required init

      // get command-line arguments
      Alpha = Float.parseFloat(Argv[3]);
      Timeout = Float.parseFloat(Argv[4]);
      P = Float.parseFloat(Argv[5]);
      MeanDelay = Float.parseFloat(Argv[6]);

      // set up first xmit, to initialize event list  
      // need to create objects first
      Frame = new FrameElt();
      TmOt = new TimeoutElt();
      SendFrame(true);
   
      JSSim.JSMainLoop(Argv);  // required; here is where the work is done
   
      // simulation done, so print out results
      System.out.println("total frames sent = "+TotFrames); 
      System.out.println("mean wait = "+TotWait/TotFrames); 
      System.out.println("mean number of xmits per frame = "
         +(float) TotTries/TotFrames); 
   }

   // send the frame and set the timer; New = true means that this is
   // the first time we are tyring to xmit this frame

   static void SendFrame(boolean New)

   {  Frame.JSName = "arrive at B";
      Frame.JSEvntTime = JSSim.JSSimTime + 1 + Alpha;
      if (New)  {
         Frame.FrameNumber = NextFrameToBeCreated++;
         Frame.StartTime = JSSim.JSSimTime;
         Frame.NTries = 1;
      }
      else Frame.NTries++;
      Frame.Trashed = false;
      Frame.JSInsertInSchedList();
      // set up the paired timeout
      TmOt.JSEvntTime = JSSim.JSSimTime + Timeout;
      TmOt.JSName = "timeout";
      TmOt.JSInsertInSchedList(); 
      // pair them together to enable one to cancel the other
      Frame.TimeoutEltNumber = TmOt.JSEltNumber;
      TmOt.FrameEltNumber = Frame.JSEltNumber;
   }

   static void DoArrivalAtB()

   {  Frame.JSName = "done with delay";
      Frame.JSEvntTime = JSSim.JSSimTime + JSSim.JSExpon(MeanDelay);
      Frame.JSInsertInSchedList();
   }

   static void DoDelayDone()

   {  Frame.Trashed = (JSSim.JSRnd() < P);
      Frame.JSName = "arrive back at A";
      Frame.JSEvntTime = JSSim.JSSimTime + Alpha;
      Frame.JSInsertInSchedList();
   }

   static void DoArrivalBackToA()

   {  // we beat the timeout, so cancel it
      JSElt.JSCancel(Frame.TimeoutEltNumber);
      if (Frame.Trashed)  
         SendFrame(false);  // failed, try sending again
      else  {
         // succeeded
         // need to do some bookkeeping 
         TotFrames++;
         TotWait += JSSim.JSSimTime - Frame.StartTime;
         TotTries += Frame.NTries;
         // done, so send the next frame
         SendFrame(true);  
      }
   }
 
   static void DoTimeout()

   {  // timed out, so cancel the scheduled arrival of this frame
      JSElt.JSCancel(TmOt.FrameEltNumber);
      SendFrame(false);  // try sending it again
   }

   public static void EvntHandler()  // application-specific; required

   {  

      if (JSSim.JSCurrEvnt.JSEvntMatch("arrive at B")) 
         {DoArrivalAtB(); return;}
      if (JSSim.JSCurrEvnt.JSEvntMatch("done with delay")) 
         {DoDelayDone(); return;}
      if (JSSim.JSCurrEvnt.JSEvntMatch("arrive back at A")) 
         {DoArrivalBackToA(); return;}
      if (JSSim.JSCurrEvnt.JSEvntMatch("timeout")) 
         {DoTimeout(); return;}
      System.out.println("shouldn't reach this point");
      System.exit(1);
   }

}

