

/*

I-Tuplets "active tuples" version of Linda.  See

   http://heather.cs.ucdavis.edu/~matloff/uct.html 

for usage.

Overall design by N. Matloff, with implementation by M. Foster, D.
Standring and R. Sweeney, in June 1999.  Updates by N. Matloff in 2000.

Here are the major data structures:

class Tuple {
public:
  Tuple();
  ~Tuple();
  int op;
  int datasize;
  char * key;
  int keysize;
  char * data;
  int (*info)[3];
};

This class represents a single tuple:  

   op  --  operation, one of 

              enum {OUT, IN, RD, EXEC};

   datasize  --  length of the array "data" in bytes

   key  --  string of data types, e.g. "rsip"

   keysize  --  the length of "key" in bytes

   data  --  the entire tuple, in the raw form transmitted between
             the application clients and the tuple server

   info  --  a variable-length array of 3-int arrays, one for each
             component of the tuple:

             info[i][0]  --  type, one of

	        enum {INT, INTA, FLOAT, FLOATA, STRING, PTR};

             info[i][1]  --  size of data for this component in bytes

             info[i][2]  --  offset of this component within the
	                     array "data"
 

class List {
public:
  void Insert (Tuple * t);
  void Remove ();
  Tuple * Iterate();
  void Reset();
  Tuple * FindTuple(Tuple * tester, int op);
  List();
  ~List();

  struct node {
    Tuple * tuple;
    node * next;
    node * prev;
  };

  node * head;
  node * index;
};


This class represents a link, via the field "index", into the tuple
space.  The tuple space is partitioned into linear linked lists,
according to key size, with the global library[i] being an instance of
List and pointing to all tuples with key size i.  

Roles of the member functions:

   Reset()  --  moves "index" to the head of this linked tuple list

   Iterate()  --  moves "index" to the next item within this linked
                  tuple list

   Remove()  --  removes tuple pointed to by "index" from this linked 
                 tuple list

   Insert(Tuple *t)  --  inserts *t into this linked tuple list

   FindTuple(Tuple *tester, int op)  --  try to match *t to a tuple in
                 this linked tuple list; if match is found, take action
		 appropriate to op, e.g. call Remove() if op = IN but
		 not if op = RD

Call sequence for a client:

   UCTInit()  --  mainly calls UCTConnectTM() and (user-supplied) Worker()

   UCTConnectTM()  --  sets up connection to tuple manager

   Worker()  --  user-supplied, specific to application

   in()/out()/etc.  --  

      EncodePacketClnt()  --  encode and send packet to tuple server

      case IN, RD:  call ReadData() and DecodePacketClnt() to get the 
                    tuple (probably after a wait)

      case OUT:  call ReadData() to wait for ACK

Call sequence for the server, i.e. tuple manager:

   UCTInit()  --  mainly calls UCTTupleManager()

   UCTTupleManager()  --  does the main work of the server; accepts
                          a connection from each client, then enters
			  into a "while" loop:

			     while(...)
			        for each client
			          check client for message 
				  if there
				    read, calling read() and then 
			               ReadData(), putting result in 
				       HeaderBuffer 
				    call ProcessMsg()
                                call UpdatePendingList(), which will
                                   check each pending message to see
				   if it can now be satisfied

   ProcessMsg()  --  call DecodePacketSrvr(), and:

                        case IN, RD:  call FindTuple() to look for match;
			              if found, call EncodePacketSrvr()
				      to send result back to client;
				      else assign this tuple to
				      PendingMsg[I], where I is the
				      number of the client

                        case OUT:  call Insert(), call WriteData() to 
			           ACK client

   UpdatePendingList()  --  for each client
                               if PendingList[client] != 0
			          call FindTuple() to check for match
				  if found, call EncodePacketSrvr() to
				     send back to client

   EncodePacketSrvr()  --  encodes the tuple, and if key[0] ='r', i.e.
                           remote op, then send response back to client
			   (in 'l' case, it is a TMEXEC tuple, called
			   locally at the server)

The variable-length aspect of the Linda functions in(), out(), etc. 
is implemented via <vararg.h> and the associated C library.


*/




#include <iostream.h> 
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include "UCTInclude.h"
#include "UCTGlobal.h"
#include "UCTTupleMgmt.h"
#include "UCTPacketMgmt.h"

/* function prototypes */
void ProcessMsg(int, int);

/* variables global to manager */
/* DoneList is an array of size equal to the number of worker nodes, which
   stores a 0 in the workers index if the worker is active, and 1 if retired. */

  int* DoneList;
  int WorkerCount;
  Tuple * InTuple,	//stores incoming data in form of Tuple
        * OutTuple;	//stores outgoing data in form of Tuple

/* PendingList is an array of size equal to the number of worker nodes, which
   stores and processes pending "in" and "rd" calls */
  Tuple ** PendingList;

  char outArgs[TMEXEC_RTN_SIZE];

/* Manager Functions */
/* UCTupleManager() is called by UCTInit() to initialize the Tuple
   Manager.  It will set up non-blocking socket connections with
   each worker and then enter a main loop which checks for worker
   messages and processes them. */

void UCTupleManager() {
  int * ClntDescriptor,	/* socket descriptor to client */
      SrvrDescriptor;	/* socket descriptor for server */
  struct sockaddr_in BindInfo;
  int NBytes, Count=0, Flag;

  DoneList = new int[UCTNWorkNodes];
  memset(DoneList, 0, (sizeof(int) * UCTNWorkNodes));

  WorkerCount = UCTNWorkNodes;

  PendingList = new Tuple* [UCTNWorkNodes];
  for (int i=0; i < UCTNWorkNodes; i++)
    PendingList[i] = NULL;

  if ((SrvrDescriptor = socket(AF_INET,SOCK_STREAM,0)) < 0)
    Error("Failure creating socket at server.\n");

  /* make the socket nonblocking */
//  ioctl(SrvrDescriptor,FIONBIO,&Flag); 
  BindInfo.sin_family = AF_INET;
  BindInfo.sin_port = UCTPortNum;
  BindInfo.sin_addr.s_addr = INADDR_ANY;
  if (bind(SrvrDescriptor,(sockaddr *) &BindInfo,sizeof(BindInfo)) < 0) {
    close(SrvrDescriptor);
    Error("Failure binding server socket.\n");
  }

  if (listen(SrvrDescriptor, UCTNWorkNodes+1) < 0)
    Error("Failure to listen at server.\n");

  /* first set up the client sockets */
  ClntDescriptor = new int[UCTNWorkNodes];
  for (int j=0; j < UCTNWorkNodes; j++)  {
    while (1)  {
       ClntDescriptor[j] = accept(SrvrDescriptor,0,0);
       if (ClntDescriptor[j] > 0) {
         Flag = 1;
         ioctl(ClntDescriptor[j], FIONBIO, &Flag); 
         break;
       }
    }
  }

  Flag = 0;	// Used to test if there was any activity this loop

  /* Main Manager Loop.  Loop through each active worker to see if they
     have a message for us.  After the last worker has been tested, run
     through the pending list to try and process pending "in" and "rd"
     calls.  Terminate loop when WorkerCount = 0, indicating that all workers
     have terminated. */

  while (WorkerCount) {
    if (!DoneList[Count]) {
      /* read header, which is always two ints long */
      NBytes = read(ClntDescriptor[Count], HeaderBuffer, headersize);

      /* if bytes were there to be read then process them */
      if (NBytes > 0) {
        ReadData(ClntDescriptor[Count], HeaderBuffer + NBytes, headersize - NBytes);
        Flag = 1;	//tell PendingList that activity has occurred this cycle

        /* ProcessMsg() will decode the incoming data into a UCTuple and
           then process the message.  It will also take care of returning
           data to the worker (except, of course, with out() operation. */
        ProcessMsg(ClntDescriptor[Count], Count);
      }
    }

    /* if true, we've cycled through all workers, so see if we can process
       anyone in the pending list */
    if (++Count == UCTNWorkNodes) {
      if (Flag) UpdatePendingList(PendingList, ClntDescriptor);
      Count = 0; Flag = 0;
    }
  }

  /* close the worker descriptors */
  for (int m=0; m < UCTNWorkNodes; m++)
    close (ClntDescriptor[m]);
}



/* ProcessMsg() is called by the manager to decode and process an incoming
   message. */

void ProcessMsg(int client, int count) {
  void (*fp)(char*, int, char*, int*);
  int argSize, outLen;
  char* Args;

  /* Decode the incoming packet and build a tuple out of it. */
  InTuple = DecodePacketSrvr(client);

  /* Handle tm_exec() calls. */
  if (InTuple->op == EXEC) {
    memcpy(&fp, InTuple->data, addrsize);	//copy the function pointer
    argSize = InTuple->datasize - addrsize; //calculate the argument size

    Args = new char[argSize];
    memcpy(Args, InTuple->data + addrsize, argSize); //get the function args

    fp(Args, argSize, outArgs, &outLen); //call function

    char * buffer = new char[outLen + intsize];   
    //copy return the output arguments
    memcpy(buffer, &outLen, intsize);
    memcpy(buffer+intsize, &outArgs, outLen);
    WriteData(client, buffer, outLen+intsize);

    delete [] Args;
    delete [] buffer;
    return;
  }

  /* For type OUT, insert this new tuple into the library.  Also, test if this
     was a termination msg "UCTEnd", and if so, decrement WorkerCount. */
  if (InTuple->op == OUT) {
    library[InTuple->keysize].Insert(InTuple);
    if (TestForEndMsg(InTuple)) {
      DoneList[count] = 1;
      WorkerCount--;
    }

    /* Send a receipt confirmation to the worker. */
    WriteData(client, "go", 3);
    return;
  }

  /* Otherwise, the type is IN or RD.  Find a matching UCTuple and
     send relevant data to the calling worker. If no match exists, the
     call to FindTuple will return NULL. */
  
  OutTuple = library[InTuple->keysize].FindTuple(InTuple, InTuple->op);

  /* If OutTuple is not NULL, we have data to return to the caller.
     Encode this data and ship it off. If it is NULL, place the InTuple
     in the pending queue for this worker, unless the original call was
     out(). */
  
  if (OutTuple)
    EncodePacketSrvr(client, InTuple, OutTuple);
  else PendingList[count] = InTuple;
}

void LocalProcessMsg(Tuple* t) {
  int size, offset_addr, offset_data;
  void * addr;

  if (t->op == OUT) {
    library[t->keysize].Insert(t);
    return;
  }

  OutTuple = library[t->keysize].FindTuple(t, t->op);

  if (!OutTuple) Error("calls must not block within user-specified function\n");

  for (int j=1; j < t->keysize; j++)
    if (t->key[j] == 'p') {
      offset_addr = t->info[j][2];
      offset_data = OutTuple->info[j][2];
      size = OutTuple->info[j][1];

      memcpy(&addr, t->data + offset_addr, addrsize);
    
      memcpy(addr, OutTuple->data + offset_data, size);
    }

  if (t->op == IN) delete OutTuple;
  delete t;
}
