// Learning Processing
// Daniel Shiffman
// http://www.learningprocessing.com

// Example 16-8: Brightness mirror

// Each pixel from the video source is drawn as a
// rectangle with size based on brightness.

import processing.video.*;

// Size of each cell in the grid
int blockSize = 10;

//do we use every input pixel (1), every second (2), etc.
int samplingRate = 2;

// Number of columns and rows in our system
int cols, rows;
//int vidCols=160, vidRows=120;
int vidCols=320, vidRows=240;
// Variable for capture device
Capture video;

// A variable for the color we are searching for.
color trackColor; 

// Declare two arrays with 50 elements.
int[] xpos = new int[50]; 
int[] ypos = new int[50];

void setup() {
  cols = vidCols*blockSize/samplingRate;
  rows = vidRows*blockSize/samplingRate;
  
  size(cols, rows);
  
  smooth();
  // Construct the Capture object
  video = new Capture(this,vidCols,vidRows,30);
  video.start();
  
  
  // Start off tracking for yellow
  trackColor = color(255,0,0);
  
   // Initialize all elements of each array to zero.
  for (int i = 0; i < xpos.length; i ++ ) {
    xpos[i] = 0; 
    ypos[i] = 0;
  }
 
}

void draw() {
  if (video.available()) {
    video.read();
  }
  background(0);

  video.loadPixels();

  // Begin loop for columns
  for (int i = 0; i < vidCols; i+=samplingRate) {
    // Begin loop for rows
    for (int j = 0; j < vidRows; j+=samplingRate) {

      // Where are we, pixel-wise?
      int x = i*blockSize/samplingRate;
      int y = j*blockSize/samplingRate;

      // Reversing x to mirror the image
      // In order to mirror the image, the column is reversed with the following formula:
      // mirrored column = width - column - 1
      int loc = (video.width - i - 1) + j*video.width;

      // Each rect is colored white with a size determined by brightness
      color c = video.pixels[loc];

      // A rectangle size is calculated as a function of the pixel's brightness. 
      // A bright pixel is a large rectangle, and a dark pixel is a small one.
      float sz = (brightness(c)/255.0)*blockSize; 
      rectMode(CENTER);
      fill(255);
      noStroke();
      rect(x + blockSize/2,y + blockSize/2,sz,sz);

    }
  }
  
  
  
  //Now add the drawing trace
  // Before we begin searching, the "world record" for closest color is set to a high number that is easy for the first pixel to beat.
  float worldRecord = 500; 

  // XY coordinate of closest color
  int closestX = 0;
  int closestY = 0;

  // Begin loop to walk through every pixel
  for (int x = 0; x < video.width; x ++ ) {
    for (int y = 0; y < video.height; y ++ ) {
      int loc = x + y*video.width;
      // What is current color
      color currentColor = video.pixels[loc];
      float r1 = red(currentColor);
      float g1 = green(currentColor);
      float b1 = blue(currentColor);
      float r2 = red(trackColor);
      float g2 = green(trackColor);
      float b2 = blue(trackColor);

      // Using euclidean distance to compare colors
      float d = dist(r1,g1,b1,r2,g2,b2); // We are using the dist( ) function to compare the current color with the color we are tracking.

      // If current color is more similar to tracked color than
      // closest color, save current location and current difference
      if (d < worldRecord) {
        worldRecord = d;
        closestX = x;
        closestY = y;
      }
    }
  }

  // Shift array values
  for (int i = 0; i < xpos.length-1; i ++ ) {
    // Shift all elements down one spot. 
    // xpos[0] = xpos[1], xpos[1] = xpos = [2], and so on. Stop at the second to last element.
    xpos[i] = xpos[i+1];
    ypos[i] = ypos[i+1];
  }

 if (worldRecord < 200) 
 { 
  
  // New location
  xpos[xpos.length-1] = cols - (int)(closestX*(float)blockSize/(float)samplingRate) -1; // Update the last spot in the array with the mouse location.
  ypos[ypos.length-1] = (int)(closestY*(float)blockSize/(float)samplingRate);
  
 // println(xpos[xpos.length-1] + " " + ypos[ypos.length-1]);
 }
 else
 {
  xpos[xpos.length-1] = xpos[xpos.length-2]; // Update the last spot in the array with the mouse location.
  ypos[ypos.length-1] = ypos[xpos.length-2];
 }  
 
  // Draw everything
  for (int i = 0; i < xpos.length; i ++ ) {
     // Draw an ellipse for each element in the arrays. 
     // Color and size are tied to the loop's counter: i.
    noStroke();
    fill(i*5,0,0);
    ellipse(xpos[i],ypos[i],3*i,3*i);
  }
 
}
