/***********************************************************************
GridEditor - Vrui application for interactive virtual clay modeling
using a density grid and interactive isosurface extraction.
Copyright (c) 2006-2009 Oliver Kreylos

This file is part of the Virtual Clay Editing Package.

The Virtual Clay Editing Package is free software; you can
redistribute it and/or modify it under the terms of the GNU General
Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

The Virtual Clay Editing Package is distributed in the hope that it will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with the Virtual Clay Editing Package; if not, write to the Free
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA
***********************************************************************/

#ifndef GRIDEDITOR_INCLUDED
#define GRIDEDITOR_INCLUDED

#include <string>
#include <vector>
#include <Misc/Array.h>
#include <Geometry/Point.h>
#include <Geometry/OrthogonalTransformation.h>
#include <GL/gl.h>
#include <GL/GLObject.h>
#include <GLMotif/RadioBox.h>
#include <GLMotif/Slider.h>
#include <Vrui/Tools/LocatorTool.h>
#include <Vrui/LocatorToolAdapter.h>
#include <Vrui/ToolManager.h>
#include <Vrui/Application.h>

#include "EditableGrid.h"

/* Forward declarations: */
namespace Misc {
class CallbackData;
}
namespace GLMotif {
class PopupMenu;
class PopupWindow;
class TextField;
}

class GridEditor:public Vrui::Application,GLObject
	{
	/* Embedded classes: */
	private:
	typedef Geometry::Point<float,3> Point; // Data type for affine points
	
	struct DataItem:public GLObject::DataItem
		{
		/* Elements: */
		public:
		GLuint influenceSphereDisplayListId; // ID of display list to render transparent spheres
		GLuint domainBoxDisplayListId; // ID of display list to render the editable grid's domain box
		
		/* Constructors and destructors: */
		DataItem(void);
		virtual ~DataItem(void);
		};
	
	class BaseLocator:public Vrui::LocatorToolAdapter // Base class for locators
		{
		/* Elements: */
		protected:
		GridEditor* application;
		
		/* Constructors and destructors: */
		public:
		BaseLocator(Vrui::LocatorTool* sTool,GridEditor* sApplication);
		
		/* Methods: */
		virtual void glRenderAction(GLContextData& contextData) const;
		};
	
	class EditLocator:public BaseLocator
		{
		/* Embedded classes: */
		public:
		enum EditMode // Enumerated type for editing modes
			{
			ADD,SUBTRACT,SMOOTH,DRAG
			};
		
		/* Elements: */
		private:
		EditableGrid* grid; // Pointer to the edited grid
		Vrui::Scalar influenceRadius; // Radius of selector's influence in physical coordinates
		float fudgeSize; // Half width of smooth area around brush radius in model size
		EditMode editMode; // Editing mode of this locator
		Vrui::NavTrackerState lastOrientation; // Orientation of locator at previous frame
		Point modelCenter; // Locator's position in model coordinates
		float modelRadius; // Locator's influence radius in model coordinates
		bool active; // Flag whether the locator is active
		Misc::Array<float,3> newValues; // Temporary storage for new grid values
		GLMotif::PopupWindow* settingsDialog;
		GLMotif::TextField* brushSizeValue;
		GLMotif::Slider* brushSizeSlider;
		GLMotif::TextField* fudgeSizeValue;
		GLMotif::Slider* fudgeSizeSlider;
		
		/* Constructors and destructors: */
		public:
		EditLocator(Vrui::LocatorTool* sTool,GridEditor* sApplication);
		virtual ~EditLocator(void);
		
		/* Methods: */
		virtual void motionCallback(Vrui::LocatorTool::MotionCallbackData* cbData);
		virtual void buttonPressCallback(Vrui::LocatorTool::ButtonPressCallbackData* cbData);
		virtual void buttonReleaseCallback(Vrui::LocatorTool::ButtonReleaseCallbackData* cbData);
		virtual void glRenderAction(GLContextData& contextData) const;
		void brushSizeSliderCallback(GLMotif::Slider::ValueChangedCallbackData* cbData);
		void fudgeSizeSliderCallback(GLMotif::Slider::ValueChangedCallbackData* cbData);
		void changeEditModeCallback(GLMotif::RadioBox::ValueChangedCallbackData* cbData);
		};
	
	typedef std::vector<BaseLocator*> BaseLocatorList;
	
	friend class BaseLocator;
	friend class EditLocator;
	
	/* Elements: */
	private:
	EditableGrid* grid; // Editable 3D grid
	BaseLocatorList baseLocators; // List of active locators
	
	GLMotif::PopupMenu* mainMenu; // The program's main menu
	
	/* Private methods: */
	GLMotif::PopupMenu* createMainMenu(void); // Creates the program's main menu
	
	/* Constructors and destructors: */
	public:
	GridEditor(int& argc,char**& argv,char**& appDefaults);
	virtual ~GridEditor(void);
	
	/* Methods: */
	virtual void initContext(GLContextData& contextData) const;
	virtual void toolCreationCallback(Vrui::ToolManager::ToolCreationCallbackData* cbData);
	virtual void toolDestructionCallback(Vrui::ToolManager::ToolDestructionCallbackData* cbData);
	virtual void display(GLContextData& contextData) const;
	void centerDisplayCallback(Misc::CallbackData* cbData);
	void saveGridCallback(Misc::CallbackData* cbData);
	};

#endif
