/***********************************************************************
GridEditor - Vrui application for interactive virtual clay modeling
using a density grid and interactive isosurface extraction.
Copyright (c) 2006-2009 Oliver Kreylos

This file is part of the Virtual Clay Editing Package.

The Virtual Clay Editing Package is free software; you can
redistribute it and/or modify it under the terms of the GNU General
Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

The Virtual Clay Editing Package is distributed in the hope that it will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with the Virtual Clay Editing Package; if not, write to the Free
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA
***********************************************************************/

#include "GridEditor.h"

#include <string.h>
#include <iostream>
#include <Misc/File.h>
#include <Misc/ThrowStdErr.h>
#include <Geometry/Box.h>
#include <Geometry/OrthogonalTransformation.h>
#include <GL/gl.h>
#include <GL/GLColorTemplates.h>
#include <GL/GLVertexTemplates.h>
#include <GL/GLVertex.h>
#include <GL/GLMaterial.h>
#include <GL/GLContextData.h>
#include <GL/GLModels.h>
#include <GL/GLGeometryWrappers.h>
#include <GLMotif/StyleSheet.h>
#include <GLMotif/PopupMenu.h>
#include <GLMotif/PopupWindow.h>
#include <GLMotif/RowColumn.h>
#include <GLMotif/Menu.h>
#include <GLMotif/Label.h>
#include <GLMotif/TextField.h>
#include <GLMotif/ToggleButton.h>
#include <GLMotif/WidgetManager.h>
#include <Vrui/GlyphRenderer.h>
#include <Vrui/Vrui.h>

/*************************************
Methods of class GridEditor::DataItem:
*************************************/

GridEditor::DataItem::DataItem(void)
	:influenceSphereDisplayListId(glGenLists(1)),
	 domainBoxDisplayListId(glGenLists(1))
	{
	}

GridEditor::DataItem::~DataItem(void)
	{
	glDeleteLists(influenceSphereDisplayListId,1);
	glDeleteLists(domainBoxDisplayListId,1);
	}

/****************************************
Methods of class GridEditor::BaseLocator:
****************************************/

GridEditor::BaseLocator::BaseLocator(Vrui::LocatorTool* sLocatorTool,GridEditor* sApplication)
	:Vrui::LocatorToolAdapter(sLocatorTool),
	 application(sApplication)
	{
	}

void GridEditor::BaseLocator::glRenderAction(GLContextData& contextData) const
	{
	}

/****************************************
Methods of class GridEditor::EditLocator:
****************************************/

GridEditor::EditLocator::EditLocator(Vrui::LocatorTool* sTool,GridEditor* sApplication)
	:BaseLocator(sTool,sApplication),
	 grid(application->grid),
	 influenceRadius(Vrui::getGlyphRenderer()->getGlyphSize()*Vrui::Scalar(2.5)),
	 editMode(ADD),
	 active(false),
	 newValues(grid->getNumVertices()),
	 settingsDialog(0),brushSizeValue(0),brushSizeSlider(0)
	{
	/* Calculate the fudge size: */
	fudgeSize=0.0f;
	for(int i=0;i<3;++i)
		fudgeSize+=Math::sqr(grid->getCellSize(i));
	fudgeSize=Math::sqrt(fudgeSize)*2.0f;
	
	const GLMotif::StyleSheet& ss=*Vrui::getWidgetManager()->getStyleSheet();
	
	/* Create the settings dialog: */
	settingsDialog=new GLMotif::PopupWindow("SettingsDialog",Vrui::getWidgetManager(),"Brush Settings");
	
	GLMotif::RowColumn* settings=new GLMotif::RowColumn("Settings",settingsDialog,false);
	settings->setNumMinorWidgets(2);
	
	/* Create a slider/textfield combo to change the marker sphere size: */
	GLMotif::Label* brushSizeLabel=new GLMotif::Label("BrushSizeLabel",settings,"Brush Size");
	
	GLMotif::RowColumn* brushSizeBox=new GLMotif::RowColumn("BrushSize",settings,false);
	brushSizeBox->setOrientation(GLMotif::RowColumn::HORIZONTAL);
	
	brushSizeValue=new GLMotif::TextField("BrushSizeValue",brushSizeBox,7);
	brushSizeValue->setFieldWidth(7);
	brushSizeValue->setPrecision(3);
	brushSizeValue->setValue(double(influenceRadius));
	
	GLMotif::Slider* brushSizeSlider=new GLMotif::Slider("BrushSizeSlider",brushSizeBox,GLMotif::Slider::HORIZONTAL,ss.fontHeight*10.0f);
	brushSizeSlider->setValueRange(double(influenceRadius)*0.1,double(influenceRadius)*5.0,double(influenceRadius)*0.01);
	brushSizeSlider->setValue(double(influenceRadius));
	brushSizeSlider->getValueChangedCallbacks().add(this,&GridEditor::EditLocator::brushSizeSliderCallback);
	
	brushSizeBox->manageChild();
	
	/* Create a slider/textfield combo to change the fudge size: */
	GLMotif::Label* fudgeSizeLabel=new GLMotif::Label("FudgeSizeLabel",settings,"Fudge Size");
	
	GLMotif::RowColumn* fudgeSizeBox=new GLMotif::RowColumn("BrushSize",settings,false);
	fudgeSizeBox->setOrientation(GLMotif::RowColumn::HORIZONTAL);
	
	fudgeSizeValue=new GLMotif::TextField("FudgeSizeValue",fudgeSizeBox,7);
	fudgeSizeValue->setFieldWidth(7);
	fudgeSizeValue->setPrecision(3);
	fudgeSizeValue->setValue(double(fudgeSize));
	
	GLMotif::Slider* fudgeSizeSlider=new GLMotif::Slider("FudgeSizeSlider",fudgeSizeBox,GLMotif::Slider::HORIZONTAL,ss.fontHeight*10.0f);
	fudgeSizeSlider->setValueRange(0.0,double(fudgeSize)*2.0,double(fudgeSize)*0.1);
	fudgeSizeSlider->setValue(double(fudgeSize));
	fudgeSizeSlider->getValueChangedCallbacks().add(this,&GridEditor::EditLocator::fudgeSizeSliderCallback);
	
	fudgeSizeBox->manageChild();
	
	/* Create a radio box to select editing modes: */
	GLMotif::Label* editModeLabel=new GLMotif::Label("EditModeLabel",settings,"Editing Mode");
	
	GLMotif::RadioBox* editModeBox=new GLMotif::RadioBox("EditModeBox",settings,false);
	editModeBox->setOrientation(GLMotif::RowColumn::HORIZONTAL);
	editModeBox->setPacking(GLMotif::RowColumn::PACK_GRID);
	editModeBox->setSelectionMode(GLMotif::RadioBox::ALWAYS_ONE);
	
	editModeBox->addToggle("Add");
	editModeBox->addToggle("Subtract");
	editModeBox->addToggle("Smooth");
	editModeBox->addToggle("Drag");
	
	switch(editMode)
		{
		case ADD:
			editModeBox->setSelectedToggle(0);
			break;
		
		case SUBTRACT:
			editModeBox->setSelectedToggle(1);
			break;
		
		case SMOOTH:
			editModeBox->setSelectedToggle(2);
			break;
		
		case DRAG:
			editModeBox->setSelectedToggle(3);
			break;
		}
	editModeBox->getValueChangedCallbacks().add(this,&GridEditor::EditLocator::changeEditModeCallback);
	editModeBox->manageChild();
	
	settings->manageChild();
	
	/* Pop up the settings dialog: */
	Vrui::popupPrimaryWidget(settingsDialog,Vrui::getNavigationTransformation().transform(Vrui::getDisplayCenter()));
	}

GridEditor::EditLocator::~EditLocator(void)
	{
	/* Pop down the settings dialog: */
	Vrui::popdownPrimaryWidget(settingsDialog);
	
	/* Delete the settings dialog: */
	delete settingsDialog;
	}

void GridEditor::EditLocator::motionCallback(Vrui::LocatorTool::MotionCallbackData* cbData)
	{
	/* Update the locator's position and radius in model coordinates: */
	Vrui::NavTrackerState newOrientation=cbData->currentTransformation;
	modelCenter=Point(newOrientation.getOrigin());
	modelRadius=float(influenceRadius*newOrientation.getScaling());
	
	if(active)
		{
		/* Determine the subdomain of the grid affected by the brush: */
		EditableGrid::Index min,max;
		for(int i=0;i<3;++i)
			{
			min[i]=int(Math::floor((modelCenter[i]-modelRadius-fudgeSize)/grid->getCellSize(i)));
			if(min[i]<0)
				min[i]=0;
			max[i]=int(Math::ceil((modelCenter[i]+modelRadius+fudgeSize)/grid->getCellSize(i)));
			if(max[i]>grid->getNumVertices(i))
				max[i]=grid->getNumVertices(i);
			}
		
		/* Update the grid: */
		float minr2=modelRadius>fudgeSize?Math::sqr(modelRadius-fudgeSize):0.0f;
		float maxr2=Math::sqr(modelRadius+fudgeSize);
		switch(editMode)
			{
			case ADD:
				{
				for(EditableGrid::Index v=min;v[0]<max[0];v.preInc(min,max))
					{
					Point p;
					float dist=0.0f;
					for(int i=0;i<3;++i)
						{
						p[i]=float(v[i])*grid->getCellSize(i);
						dist+=Math::sqr(modelCenter[i]-p[i]);
						}
					if(dist<maxr2)
						{
						float val;
						if(dist<minr2)
							val=1.0f;
						else
							val=(modelRadius+fudgeSize-Math::sqrt(dist))/(2.0f*fudgeSize);
						float oldVal=grid->getValue(v);
						if(val>oldVal)
							grid->setValue(v,val);
						}
					}
				
				grid->invalidateVertices(min,max);
				break;
				}
			
			case SUBTRACT:
				{
				for(EditableGrid::Index v=min;v[0]<max[0];v.preInc(min,max))
					{
					Point p;
					float dist=0.0f;
					for(int i=0;i<3;++i)
						{
						p[i]=float(v[i])*grid->getCellSize(i);
						dist+=Math::sqr(modelCenter[i]-p[i]);
						}
					if(dist<maxr2)
						{
						float val;
						if(dist<minr2)
							val=0.0f;
						else
							val=1.0f-(modelRadius+fudgeSize-Math::sqrt(dist))/(2.0f*fudgeSize);
						float oldVal=grid->getValue(v);
						if(val<oldVal)
							grid->setValue(v,val);
						}
					}
				
				grid->invalidateVertices(min,max);
				break;
				}
			
			case SMOOTH:
				{
				for(int i=0;i<3;++i)
					{
					if(min[i]==0)
						++min[i];
					if(max[i]==grid->getNumVertices(i))
						--max[i];
					}
				for(EditableGrid::Index v=min;v[0]<max[0];v.preInc(min,max))
					{
					Point p;
					float dist=0.0f;
					for(int i=0;i<3;++i)
						{
						p[i]=float(v[i])*grid->getCellSize(i);
						dist+=Math::sqr(modelCenter[i]-p[i]);
						}
					if(dist<maxr2)
						{
						float avgVal=0.0f;
						EditableGrid::Index i;
						for(i[0]=v[0]-1;i[0]<=v[0]+1;++i[0])
							for(i[1]=v[1]-1;i[1]<=v[1]+1;++i[1])
								for(i[2]=v[2]-1;i[2]<=v[2]+1;++i[2])
									avgVal+=grid->getValue(i);
						avgVal/=27.0f;
						if(dist<minr2)
							newValues(v)=avgVal;
						else
							{
							float w=(modelRadius+fudgeSize-Math::sqrt(dist))/(2.0f*fudgeSize);
							newValues(v)=avgVal*w+grid->getValue(v)*(1.0f-w);
							}
						}
					else
						newValues(v)=grid->getValue(v);
					}
				
				for(EditableGrid::Index v=min;v[0]<max[0];v.preInc(min,max))
					grid->setValue(v,newValues(v));
				
				grid->invalidateVertices(min,max);
				break;
				}
			
			case DRAG:
				{
				Vrui::NavTrackerState t=lastOrientation;
				t*=Geometry::invert(newOrientation);
				Geometry::OrthogonalTransformation<float,3> pt(t);
				
				float r2=Math::sqr(modelRadius);
				for(EditableGrid::Index v=min;v[0]<max[0];v.preInc(min,max))
					{
					Point p;
					float dist=0.0f;
					for(int i=0;i<3;++i)
						{
						p[i]=float(v[i])*grid->getCellSize(i);
						dist+=Math::sqr(modelCenter[i]-p[i]);
						}
					if(dist<r2)
						{
						/* Compute the dragged position: */
						Point dp=pt.transform(p);
						float w=Math::sqrt(dist)/modelRadius;
						dp=Geometry::affineCombination(dp,p,w);
						
						/* Look up the grid value at the dragged position: */
						float dragVal=grid->getValue(dp);
						newValues(v)=dragVal;
						}
					else
						newValues(v)=grid->getValue(v);
					}
				
				for(EditableGrid::Index v=min;v[0]<max[0];v.preInc(min,max))
					grid->setValue(v,newValues(v));
				
				grid->invalidateVertices(min,max);
				break;
				}
			}
		
		Vrui::requestUpdate();
		}
	
	lastOrientation=newOrientation;
	}

void GridEditor::EditLocator::buttonPressCallback(Vrui::LocatorTool::ButtonPressCallbackData* cbData)
	{
	active=true;
	}

void GridEditor::EditLocator::buttonReleaseCallback(Vrui::LocatorTool::ButtonReleaseCallbackData* cbData)
	{
	active=false;
	}

void GridEditor::EditLocator::glRenderAction(GLContextData& contextData) const
	{
	glPushAttrib(GL_COLOR_BUFFER_BIT|GL_ENABLE_BIT|GL_LINE_BIT|GL_POLYGON_BIT);
	
	/* Retrieve context entry: */
	DataItem* dataItem=contextData.retrieveDataItem<DataItem>(application);
	
	/* Render the influence sphere: */
	glDisable(GL_LIGHTING);
	
	glPushMatrix();
	glTranslate(modelCenter-Point::origin);
	glScale(modelRadius);
	glCallList(dataItem->influenceSphereDisplayListId);
	glPopMatrix();
	
	glPopAttrib();
	}

void GridEditor::EditLocator::brushSizeSliderCallback(GLMotif::Slider::ValueChangedCallbackData* cbData)
	{
	/* Get the new brush size: */
	influenceRadius=Vrui::Scalar(cbData->value);
	
	/* Update the brush size value label: */
	brushSizeValue->setValue(double(cbData->value));
	}

void GridEditor::EditLocator::fudgeSizeSliderCallback(GLMotif::Slider::ValueChangedCallbackData* cbData)
	{
	/* Get the new fudge size: */
	fudgeSize=float(cbData->value);
	
	/* Update the fudge size value label: */
	fudgeSizeValue->setValue(double(cbData->value));
	}

void GridEditor::EditLocator::changeEditModeCallback(GLMotif::RadioBox::ValueChangedCallbackData* cbData)
	{
	switch(cbData->radioBox->getToggleIndex(cbData->newSelectedToggle))
		{
		case 0:
			editMode=ADD;
			break;
		
		case 1:
			editMode=SUBTRACT;
			break;
		
		case 2:
			editMode=SMOOTH;
			break;
		
		case 3:
			editMode=DRAG;
			break;
		}
	}

/***************************
Methods of class GridEditor:
***************************/

GLMotif::PopupMenu* GridEditor::createMainMenu(void)
	{
	/* Create a top-level shell for the main menu: */
	GLMotif::PopupMenu* mainMenuPopup=new GLMotif::PopupMenu("MainMenuPopup",Vrui::getWidgetManager());
	mainMenuPopup->setTitle("3D Grid Editor");
	
	/* Create the actual menu inside the top-level shell: */
	GLMotif::Menu* mainMenu=new GLMotif::Menu("MainMenu",mainMenuPopup,false);
	
	/* Create a button to reset the navigation coordinates to the default (showing the entire grid): */
	GLMotif::Button* centerDisplayButton=new GLMotif::Button("CenterDisplayButton",mainMenu,"Center Display");
	centerDisplayButton->getSelectCallbacks().add(this,&GridEditor::centerDisplayCallback);
	
	/* Create a button to save the grid to a file: */
	GLMotif::Button* saveGridButton=new GLMotif::Button("SaveGridButton",mainMenu,"Save Grid");
	saveGridButton->getSelectCallbacks().add(this,&GridEditor::saveGridCallback);
	
	/* Calculate the main menu's proper layout: */
	mainMenu->manageChild();
	
	/* Return the created top-level shell: */
	return mainMenuPopup;
	}

GridEditor::GridEditor(int& argc,char**& argv,char**& appDefaults)
	:Vrui::Application(argc,argv,appDefaults),
	 grid(0),
	 mainMenu(0)
	{
	if(argc>=2)
		{
		/* Load the grid from a float-valued vol file: */
		Misc::File volFile(argv[1],"rb",Misc::File::BigEndian);
		
		/* Read the file header: */
		EditableGrid::Index numVertices;
		volFile.read<int>(numVertices.getComponents(),3);
		int borderSize=volFile.read<int>();
		for(int i=0;i<3;++i)
			numVertices[i]+=borderSize*2;
		float domainSize[3];
		volFile.read<float>(domainSize,3);
		EditableGrid::Size cellSize;
		for(int i=0;i<3;++i)
			cellSize[i]=domainSize[i]/float(numVertices[i]-borderSize*2-1);
		
		/* Create the grid: */
		grid=new EditableGrid(numVertices,cellSize);
		
		/* Read all grid values: */
		for(EditableGrid::Index i(0);i[0]<grid->getNumVertices(0);i.preInc(grid->getNumVertices()))
			grid->setValue(i,volFile.read<float>());
		grid->invalidateVertices(EditableGrid::Index(0,0,0),grid->getNumVertices());
		}
	else
		{
		/* Create a new grid: */
		grid=new EditableGrid(EditableGrid::Index(256,256,256),EditableGrid::Size(1.0f,1.0f,1.0f));
		}
	
	/* Create the program GUI: */
	mainMenu=createMainMenu();
	Vrui::setMainMenu(mainMenu);
	
	/* Initialize the navigation transformation: */
	centerDisplayCallback(0);
	}

GridEditor::~GridEditor(void)
	{
	delete mainMenu;
	}

void GridEditor::initContext(GLContextData& contextData) const
	{
	/* Create a context data item and store it in the context: */
	DataItem* dataItem=new DataItem;
	contextData.addDataItem(this,dataItem);
	
	/* Create the influence sphere display list: */
	glNewList(dataItem->influenceSphereDisplayListId,GL_COMPILE);
	glDisable(GL_CULL_FACE);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask(GL_FALSE);
	glLineWidth(1.0f);
	glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
	glColor4f(1.0f,1.0f,0.0f,0.67f);
	glDrawSphereIcosahedron(1.0,5);
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glColor4f(0.5f,0.5f,0.1f,0.67f);
	glDrawSphereIcosahedron(1.0,5);
	glDepthMask(GL_TRUE);
	glDisable(GL_BLEND);
	glEndList();
	
	/* Create the domain box display list: */
	Point min=Point::origin;
	Point max;
	for(int i=0;i<3;++i)
		max[i]=float(grid->getNumVertices(i)-1)*grid->getCellSize(i);
	Vrui::Color fgColor=Vrui::getBackgroundColor();
	for(int i=0;i<3;++i)
		fgColor[i]=1.0f-fgColor[i];
	glNewList(dataItem->domainBoxDisplayListId,GL_COMPILE);
	glColor(fgColor);
	glBegin(GL_LINE_STRIP);
	glVertex(min[0],min[1],min[2]);
	glVertex(max[0],min[1],min[2]);
	glVertex(max[0],max[1],min[2]);
	glVertex(min[0],max[1],min[2]);
	glVertex(min[0],min[1],min[2]);
	glVertex(min[0],min[1],max[2]);
	glVertex(max[0],min[1],max[2]);
	glVertex(max[0],max[1],max[2]);
	glVertex(min[0],max[1],max[2]);
	glVertex(min[0],min[1],max[2]);
	glEnd();
	glBegin(GL_LINES);
	glVertex(max[0],min[1],min[2]);
	glVertex(max[0],min[1],max[2]);
	glVertex(max[0],max[1],min[2]);
	glVertex(max[0],max[1],max[2]);
	glVertex(min[0],max[1],min[2]);
	glVertex(min[0],max[1],max[2]);
	glEnd();
	glEndList();
	}

void GridEditor::toolCreationCallback(Vrui::ToolManager::ToolCreationCallbackData* cbData)
	{
	/* Check if the new tool is a locator tool: */
	Vrui::LocatorTool* locatorTool=dynamic_cast<Vrui::LocatorTool*>(cbData->tool);
	if(locatorTool!=0)
		{
		BaseLocator* newLocator;
		
		/* Create a new edit locator: */
		newLocator=new EditLocator(locatorTool,this);
		
		/* Add new locator to list: */
		baseLocators.push_back(newLocator);
		}
	}

void GridEditor::toolDestructionCallback(Vrui::ToolManager::ToolDestructionCallbackData* cbData)
	{
	/* Check if the to-be-destroyed tool is a locator tool: */
	Vrui::LocatorTool* locatorTool=dynamic_cast<Vrui::LocatorTool*>(cbData->tool);
	if(locatorTool!=0)
		{
		/* Find the base locator associated with the tool in the list: */
		for(BaseLocatorList::iterator blIt=baseLocators.begin();blIt!=baseLocators.end();++blIt)
			if((*blIt)->getTool()==locatorTool)
				{
				/* Remove the data locator: */
				delete *blIt;
				baseLocators.erase(blIt);
				break;
				}
		}
	}

void GridEditor::display(GLContextData& contextData) const
	{
	/* Get a pointer to the data item: */
	DataItem* dataItem=contextData.retrieveDataItem<DataItem>(this);
	
	/* Render the grid's domain box: */
	GLboolean lightingEnabled=glIsEnabled(GL_LIGHTING);
	if(lightingEnabled)
		glDisable(GL_LIGHTING);
	GLfloat lineWidth;
	glGetFloatv(GL_LINE_WIDTH,&lineWidth);
	glLineWidth(1.0f);
	glCallList(dataItem->domainBoxDisplayListId);
	if(lightingEnabled)
		glEnable(GL_LIGHTING);
	glLineWidth(lineWidth);
	
	/* Render the grid's current state: */
	glMaterial(GLMaterialEnums::FRONT,GLMaterial(GLMaterial::Color(0.5f,0.5f,0.5f),GLMaterial::Color(0.5f,0.5f,0.5f),25.0f));
	grid->glRenderAction(contextData);
	
	/* Render all locators: */
	for(BaseLocatorList::const_iterator blIt=baseLocators.begin();blIt!=baseLocators.end();++blIt)
		(*blIt)->glRenderAction(contextData);
	}

void GridEditor::centerDisplayCallback(Misc::CallbackData* cbData)
	{
	typedef Geometry::Box<float,3> Box;
	
	Box bb(Box::Point(0.0f,0.0f,0.0f),Box::Point(255.0f,255.0f,255.0f));
	
	/* Calculate the center and radius of the box: */
	Vrui::Point center=Geometry::mid(bb.min,bb.max);
	Vrui::Scalar radius=Vrui::Scalar(Geometry::dist(bb.min,bb.max))*Vrui::Scalar(0.5);
	Vrui::setNavigationTransformation(center,radius);
	}

void GridEditor::saveGridCallback(Misc::CallbackData* cbData)
	{
	/* Write the current contents of the grid to a floating-point vol file: */
	Misc::File volFile("Grid.fvol","wb",Misc::File::BigEndian);
	
	/* Write the vol file header: */
	volFile.write<int>(grid->getNumVertices().getComponents(),3);
	volFile.write<int>(0);
	float domainSize[3];
	for(int i=0;i<3;++i)
		domainSize[i]=float(grid->getNumVertices(i)-1)*grid->getCellSize(i);
	volFile.write<float>(domainSize,3);
	
	/* Write the grid data values: */
	for(EditableGrid::Index i(0);i[0]<grid->getNumVertices(0);i.preInc(grid->getNumVertices()))
		volFile.write<float>(grid->getValue(i));
	}

int main(int argc,char* argv[])
	{
	try
		{
		char** appDefaults=0;
		GridEditor ge(argc,argv,appDefaults);
		ge.run();
		}
	catch(std::runtime_error err)
		{
		std::cerr<<"Caught exception "<<err.what()<<std::endl;
		return 1;
		}
	
	return 0;
	}
