package edu.ucdavis.rj;

import edu.ucdavis.rj.BaseArmCode.Control;
import edu.ucdavis.rj.CoArm.CoKind;

import java.util.*;

/**
 * Like JR's co statement.
 */
public class Co {

    // regular arms:
    ArrayList<CoArm> arms;

    /**
     * Create a new instance of an Co.
     * @param firstArm  The first arm.
     * @param restArms  Any other arms.
     */
    public Co(CoArm firstArm, CoArm ... restArms) {
	createCo(firstArm, restArms);
    }

    /**
     * Common code called from constructors to create a new instance of an Co.
     * @param firstArm  The first arm.
     * @param restArms  Any other arms.
     */
    private void createCo(CoArm firstArm, CoArm ... restArms) {
	// flatten abstArms into arms
	this.arms = new ArrayList<CoArm>();
	int armCount = 0;
	oneCoArm(firstArm, armCount);
	for (CoArm a: restArms) {
	    armCount++;
	    oneCoArm(a, armCount);
	}
	//////////System.err.println("Co this.arms.size()="+this.arms.size());
    }

    /**
     * Handle one abstract arm for this Co.
     * @param a           The arm.
     * @param armCount    The arm's number (0-based).
     */
    private void oneCoArm(CoArm a, int armCount) {
        if (a == null) {
	    throw new rjRuntimeError("Co given null arm (arm "+armCount+
				     ", counting from 0)");
	}
	if (a.op != null) {
	    this.arms.add(a);
	}
	else if (a.oa != null) {
	    ArrayList<OpProxy> opsList = a.oa.getOpsList();
	    ArrayList<Invocation> invsList = a.oa.getInvocationsList();
	    ArmCode code = a.getCode();
	    ArrayList<Quantifiers> quantsList = a.oa.getQuantifiersList();
	    CoKind coKind = a.getCoKind();
	    // check number of ops and quants are equal.
	    if (quantsList != null && opsList.size() != quantsList.size()) {
		throw new rjRuntimeError("Co OpArray and quants sizes differ");
	    }
	    if (quantsList != null && invsList.size() != quantsList.size()) {
		throw new rjRuntimeError("Co OpArray and invocation list sizes differ");
	    }

	    int qk = 0;
	    for (OpProxy ia: opsList) {
		Quantifiers quants;
		quants = (quantsList == null)? null: quantsList.get(qk);
		Invocation inv = invsList.get(qk);

		////////////////////////////// give all of these a single returnop (not a quantified one); can do since Invocation contains quantifier.


		this.arms.add(new CoArm(quants, ia, inv, code, coKind));
		qk++;
	    }
	}
	else {
	    throw new rjRuntimeError("RJ internal error: bad CoArm");
	}
    }

    /**
     * Execute this co.
     * Perform all invocations specified in this co's arms.
     * A {@code COCALL} invocation completes when get reply from invoked operation.
     * A {@code COSEND} invocation completes immediately after invocation is sent.
     * As each invocation completes, execute the associated arm's
     * code ("post-processing code", aka PPC);
     * continue doing so until all invocations have 
     * or a PPC executes the equivalent of a {@code break} or {@code return}.
     * @return control flow value executed by last PPC.
     */
    public ArmCode.Control go() {
	// iterate through Invocations and send them out.
	// also build up the Inni for the Co.
	///////////System.out.println("Co.go "+arms.size());
	// build up Inni's arms as first and rest,
	// since that coincides with Inni's constructor.

	// Control value to return.
	ArmCode.Control retControl = ArmCode.Control.NORMAL;

	// for no arms, do nothing.
	if (arms.size() == 0) {
	    return retControl;
	}

	InniArm firstInniArm = null; // hush javac
	InniArm restInniArms [] = new InniArm [arms.size()-1];
	int armNumber = 0;
	int numInvs = 0;
	for (CoArm arm: arms) {
	    numInvs++;
	    OpProxy op = arm.op;
	    try {
		// For a cocall,
		// arrange to have the invoked op send its
		// reply to the CoArm's op.
		// A cosend is treated like a cocall,
		// but we send its reply here since
		// its servicer won't.
		// (Semantically, a servicer for it might not even exist,
		// but a cosend invocation is considered to have completed
		// once it has been sent, just as for a send invocation.)
		//
		// The exception is for a cocall to noop.
		// we need that to just return, but since the code below
		// uses send, we handle that case specially, so that
		// it is like a cosend, so that we get the reply invoked.
		// (We could handle that in the send methods with a special
		// case or add a new cosend method, but it seems better to 
		// keep it here.)
		if (arm.coKind == CoKind.COSEND || op.isNoop()) {
		    arm.replyOp.send();
		    // and just to be safe:
		    arm.inv.setReplyOp(null);
		}
		else if (arm.coKind == CoKind.COCALL) {
		    // this is why inv inv should be cloned before this.
		    arm.inv.setReplyOp(new OpProxy(arm.replyOp));
		}
		else {
		    throw new rjRuntimeError("RJ internal error: unknown arm.coKind");
		}
		// can't use internalSend here
		// since it's for OpProxy, not just OpImpl.
		op.send(arm.inv);
		////////System.out.println("sent "+op);
		///////System.out.println("sent "+arm.inv);
	    } catch (java.rmi.RemoteException e) {
		e.printStackTrace();
		System.exit(1);
	    }
	    // make the InniArm
	    InniArm theArm = new InniArm(arm.quants, new OpProxy(arm.replyOp),
					 null, null, arm.code);
	    if (armNumber == 0) {
		firstInniArm = theArm;
	    }
	    else {
		restInniArms[armNumber-1] = theArm;
	    }

	    armNumber++;
	}

	Inni coInni = new Inni(firstInniArm, restInniArms);

	// now handle each invocation finishing and execute the associated PPC
	// i.e., do the Inni, once for each invocation
	// or until a PPC changes control.

	for (int k = 1; k <= numInvs; k++) {
	    ArmCode.Control control = coInni.service();
	    // BREAK and CONTINUE affect only this scope.
	    // propagate a RETURN to enclosing scope;
	    if (control == Control.BREAK) break;
	    else if (control == Control.CONTINUE) continue;
	    else if (control == Control.RETURN) {
		retControl = Control.RETURN;
		break;
	    }
	}

	return retControl;

    }

}
