package edu.ucdavis.rj;

import java.util.*;

/**
 * An Inni services on invocation from a specified group of InniOps.
 * It consists of one or more possibly quantified arms;
 * each arm specifies an operation or quantified group of operations
 * and corresponding code to service the invocation.
 * An Inni also allows an elseArm,
 * which is executed if no invocation is selected for servicing.
 */
public class Inni {

    // regular arms:
    ArrayList<InniArm> arms;

    // (can't have both an elseArm and an elseafterArm)
    // the one else arm, if it's present:
    InniArmElse elseArm = null;
    // the one elseafter arm, if it's present:
    InniArmElseafter elseafterArm = null;

    /**
     * Create a new instance of an Inni.
     * This is the general case (with an elseArm).
     * @param elseArm   The elseArm.
     * @param firstArm  The first arm.
     * @param restArms  Any other arms.
     */
    public Inni(InniArmElse elseArm, InniArm firstArm, InniArm ... restArms) {
        if (elseArm == null) {
	    throw new rjRuntimeError("Inni given null elseArm");
	}
	this.elseArm = elseArm;
	createInni(elseArm, null, firstArm, restArms);
    }

    /**
     * Create a new instance of an Inni.
     * This is the general case (with an elseafterArm).
     * @param elseafterArm   The elseafterArm.
     * @param firstArm  The first arm.
     * @param restArms  Any other arms.
     */
    public Inni(InniArmElseafter elseafterArm, InniArm firstArm, InniArm ... restArms) {
        if (elseafterArm == null) {
	    throw new rjRuntimeError("Inni given null elseafterArm");
	}
	this.elseafterArm = elseafterArm;
	createInni(null, elseafterArm, firstArm, restArms);
    }

    /**
     * Create a new instance of an Inni.
     * This is the common case (without an elseArm or elseafterArm).
     * @param firstArm  The first arm.
     * @param restArms  Any other arms.
     */
    public Inni(InniArm firstArm, InniArm ... restArms) {
	createInni(null, null, firstArm, restArms);
    }

    /**
     * Common code called from constructors to create a new instance of an Inni.
     * @param elseArm   The elseArm.
     * @param elseafterArm   The elseafterArm.
     * @param firstArm  The first arm.
     * @param restArms  Any other arms.
     */
    private void createInni(InniArmElse elseArm, InniArmElseafter elseafterArm,
			    InniArm firstArm, InniArm ... restArms) {
	this.elseArm = elseArm;
	this.elseafterArm = elseafterArm;

	// flatten abstArms into arms
	this.arms = new ArrayList<InniArm>();
	int armCount = 0;
	oneInniArm(this.arms, firstArm, armCount);
	for (InniArm a: restArms) {
	    armCount++;
	    oneInniArm(arms, a, armCount);
	}
	//////////System.err.println("Inni this.arms.size()="+this.arms.size());
    }

    /**
     * Handle one abstract arm for this Inni.
     * @param arms        The list of arms to which to add this arm.
     * @param a           The arm.
     * @param armCount    The arm's number (0-based).
     */
    private void oneInniArm(ArrayList<InniArm> arms, InniArm a, int armCount) {
        if (a == null) {
	    throw new rjRuntimeError("Inni given null arm (arm "+armCount+
				     ", counting from 0)");
	}
	if (a.op != null) {
	    arms.add(a);
	}
	else if (a.oa != null) {
	    ArrayList<OpProxy> opsList = a.oa.getOpsList();
	    ArmCode code = a.getCode();
	    ArrayList<Quantifiers> quantsList = a.oa.getQuantifiersList();
	    // check number of ops and quants are equal.
	    if (quantsList != null && opsList.size() != quantsList.size()) {
		throw new rjRuntimeError("Inni OpArray and quants sizes differ");
	    }

	    int qk = 0;
	    for (OpProxy ia: opsList) {
		Quantifiers quants;
		quants = (quantsList == null)? null: quantsList.get(qk);
		arms.add(new InniArm(quants, ia, a.st, a.by, code));
		qk++;
	    }
	}
	else {
	    throw new rjRuntimeError("RJ internal error: unknown InniArm");
	}
    }

    /**
     * Service one invocation as specified by this Inni's arms
     * by executing the codeBlock in the associated arm.
     * If no invocation selected,
     * delay the executing process until such an invocation arrives,
     * unless this Inni has an elseArm,
     * in which case execute the codeBlock in the elseArm.
     * @return Value set within codeBlock indicating control flow to take.
     */
    public ArmCode.Control service() {

	// service these arms.
	// (changed only in elseafterArm case, since that changes arms)
	// so that Inni can be shared or even reused by same process
	ArrayList<InniArm> sArms = arms;

	// if there's an elseafterArm, start up a timer.
	// The timer will send an invocation to this Inni
	// when the timer goes off.
	// More specifically, create a replyOp here for the timer to use
	// and add a corresponding arm to the Inni for that replyOp.
	if (elseafterArm != null) {
	    // we're going to modify arms.
	    // we'll make a copy of it and use that copy henceforth.
	    sArms = (ArrayList<InniArm> )(arms.clone());
	    // add arm for replyOp to sArms:
	    OpInni replyOp = OpInni.newOpInni();
	    // this is a little tricky: InniArm needs ArmCode
	    // whereas we have an InniArmElseCode,
	    // so build up one of the former from the latter.
	    ArmCode elseafterArmCode = new ArmCode() {
		    public void codeBlock(Invocation inv) {
			// just invoke the real elseafterArm.code
			ArmCode.Control retControl = 
			    execElseArmCode(elseafterArm.code);
                        setControl(retControl);
		    }
		};
	    oneInniArm(sArms, new InniArm(replyOp, elseafterArmCode),
		       sArms.size());
	    // fire up the timer
	    elseafterTimer.send(
				new Invocation(
					       elseafterArm.millis,
					       elseafterArm.nanos,
					       replyOp));
	}

	////2012-10-10 Debug.println("===>Inni pre-InniLocker");
	InniLocker inniLocker = new InniLocker(sArms);
	////2012-10-10Debug.println("===>Inni post-InniLocker");

	inniLocker.lock();
	////2012-10-10 Debug.println("===>Inni post-InniLocker.lock()");
	    // returns after servicing
	    while (true) {
		// make FCFS servicing.
		// service in order of times.
		// note that an op that has no invocations will have
		// a time of -1, so it will be tried before others;
		// however, its selectInvocation will return null,
		// so all will be fine.
		// alternatives:
		//  discard -1 entries from orderedTimeNodes before for loop.
		//  test for -1 entry before calling selectInvocation.
		ArrayList<TimeNode> orderedTimeNodes = gatherAndSortTimes(sArms);
		for (TimeNode tn: orderedTimeNodes) {
		    InniArm arm = sArms.get(tn.armNum);
		    Invocation inv = arm.selectInvocation();
		    ////2012-10-10 Debug.println("===>Inni post-arm.selectInvocation() inv="+inv);
		    if (inv != null) {
			inniLocker.unlock();
			// service it
			Quantifiable.Helper.setQuantifiers(arm.code, arm.quants);
			// this code similar to execElseArmCode,
			// but here need to pass inv to the code.
			ArmCode.Control retControl = ArmCode.Control.NORMAL;
			if (arm.code != null) {
			    arm.code.setControl(ArmCode.Control.NORMAL);
			    arm.code.codeBlock(inv);
			    retControl = arm.code.getControl();
			}
			inv.replyToInvoker();
			return retControl;
		    }
		}
		if (elseArm == null) {
		    ////2012-10-10 Debug.println("===>Inni pre inniLocker.waitOnLock()");
			inniLocker.waitOnLock();
			////2012-10-10 Debug.println("===>Inni post inniLocker.waitOnLock()");
		}
		else { // do the else arm
		    inniLocker.unlock();
		    return execElseArmCode(elseArm.code);
		}
	    }
    }

    /**
     * Used for {@code ElseArm} and {@code ElseafterArm}.
     * Execute the arm code, if it's not {@code null}.
     * Returns the control info from the execution.
     * N.B., these are executed w/o an {@code Invocation}!
     * @param code The arm code.
     * @return The control info.
     */
    private ArmCode.Control execElseArmCode(InniArmElseCode code) {
	ArmCode.Control retControl = ArmCode.Control.NORMAL;
	if (code != null) {
	    code.setControl(ArmCode.Control.NORMAL);
	    code.codeBlock();
	    retControl = code.getControl();
	}
	return retControl;
    }

    /**
     * /////////////////// will be sorted by time...
     */
    static class TimeNode implements Comparable {
	public int armNum;
	public long time;

	public TimeNode(int armNum, long time) {
	    this.armNum = armNum;
	    this.time = time;
	}

	public int compareTo(Object o) {
	    TimeNode other = (TimeNode) o;
	    return this.time < other.time ? -1: this.time > other.time ? 1: 0;
	}
    }

    private ArrayList<TimeNode> gatherAndSortTimes(ArrayList<InniArm> arms ) {
	ArrayList<TimeNode> ret = new ArrayList<TimeNode>(arms.size());
	for (int a = 0; a < arms.size(); a++) {
	    long firstTime = -1; // hush javac
	    try {
		firstTime = arms.get(a).getOpProxy().getFirstTime();
	    } catch (java.rmi.RemoteException e) {
		throw new rjRuntimeError("RJ internal error: Inni gatherAndSortTimes");
	    }
	    ret.add(new TimeNode(a, firstTime));
	}
	Collections.sort(ret);
	return ret;
    }

    /**
     * This OpMethod is used by Inni's implementation
     * to provide a timer for the elseafter arm.
     * Note the parameters to its codeBlock.
     * ////////////////////////////////////
     */
    private static OpMethod elseafterTimer;
    static {
	try {
	    elseafterTimer = new OpMethod() {
		    public void codeBlock(Invocation inv) {
                        long millis = (Long)(inv.getParam(0));
                        int nanos = (Integer)(inv.getParam(1));
                        OpImpl replyOp = (OpImpl)(inv.getParam(2));
			RJ.nap(millis, nanos);
			// timer has gone off; notify invoker.
			replyOp.send();
		    }
		};
        } catch (Exception e) {
	    throw new rjRuntimeError("RJ internal error: Inni's elseafterTimer OpMethod");
        }
    }
}
