package edu.ucdavis.rj;

import java.util.*;


import java.rmi.*;
import java.rmi.server.*;


/**
 * Basic operation abstraction.
 *
 * In general, an {@link OpImpl} shouldn't be passed as a parameter.
 * Instead, use an {@link OpProxy}.
 * E.g., if {@code f} is an {@link OpImpl}, then pass {@code new OpProxy(f))}
 */
abstract public class OpImpl
    extends UnicastRemoteObject
    implements OpRemote {

    private static long seqNum = 1;
    private String id;

    /**
     * An OpImpl's Id is formed from VM name and seqNum
     * so it is unique across the entire program.
     * @return The Id.
     */
    private String makeId() {
	return rjvm.thisVM.getName() + "." + seqNum++;
    }

    public String getId() {
        return this.id;
    }
    
    /**
     * Since have one noop per VM, need to know whether an OpImpl is a noop.
     * (If c is an OpProxy, comparing
     *   noop == c
     *   c.equals(noop)
     * won't work w/o something extra.
     *
     * default method.
     * overriden for actual noop.
     */
    public boolean isNoop() {
        return false;
    }

    /**
     * Create an instance of an {@code OpImpl}.
     * @throws java.rmi.RemoteException if RMI fails.
     */
    public OpImpl() throws java.rmi.RemoteException {
	this.id = makeId();
    }

    /**
     * Invoke this operation asynchronously.
     * @param originalInv The invocation to service.
     */
    public void send(Invocation originalInv) {
	Invocation inv = originalInv.safeClone();
	this.internalSend(inv);
    }

    /**
     * Invoke this operation asynchronously.
     * Abbreviation for 
     *  {@code
        this.send (new Invocation(...));
	}
     * See {@link Op#send}.
     */
    public void send() {
	this.send(new Invocation());
    }

    // documentation inherited
    public void V() {
	this.send();
    }

    /**
     * Internal asynchronous invocation.
     * Note that both call and send come through here.
     * @param inv The invocation to service.
     * It is assumed to have already been cloned
     * and its {@code replyOp} set properly
     * ({@code null} for send; non-{@code null} for call).
     */
    abstract void internalSend(Invocation inv);

    /**
     * Invoke this operation synchronously.
     * @param  originalInv The invocation to service.
     * @return The invocation,
     * from which the return value of the operation can be extracted.
     * <P>
     * N.B., (the reference for) the returned invocation might not be the
     * same as {@code inv}.
     * For example, do <b> NOT </b> use, e.g.,
     *  {@code
        Invocation inv = new Invocation(...);
	thefun.call(inv);
        System.out.println(inv.getReturnValue());
	}
     * because {@code inv}'s value after the {@code call} will be the same
     * as its value before the call, and {@code inv.getReturnValue()} will
     * likely be {@code null}.
     * Instead, in this example, use
     *   {@code inv = thefun.call(inv);}.
     *
     */
    public Invocation call(Invocation originalInv) {
	rjvm.ariseAndReceive();  // from caller
	/////2012-10-09	Debug.println("Op call ..........  "+originalInv);
	Invocation inv = originalInv.safeClone();
	OpProxy back;
	if (rjvm.threadLocalRO) {
	    back = ThreadLocalReplyOp.get();
	    ///System.out.println("call reply op using ThreadLocal");
	}
	else {
	    back = new OpProxy(OpInni.newOpInni());
	    ///System.out.println("call reply op using new");
	}
	inv.setReplyOp(back);
	this.internalSend(inv);
	rjvm.sendAndDie(); // to recv
	try {
	    inv = back.receive();
	} catch (Exception e) {
	    throw new rjRuntimeError("RJ internal error: Op's call");
	}
	rjvm.ariseAndReceive(); // from recv
	// following line in JR generated code was in a finally clause:
	rjvm.sendAndDie();    // to caller
	return inv;
    }

    /**
     * Invoke this operation synchronously.
     * Abbreviation for 
     *  {@code
        this.call (new Invocation(...));
	}
     * See {@link Op#call}.
     */
    public Invocation call() {
	return this.call(new Invocation());
    }

    /**
     * Returns the number of Invocations pending for this operation.
     * @return The number of pending Invocations.
     */
    abstract public int length();

    /**
     * Compares this {@code OpImpl} object with the specified object,
     * which can be an {@code OpRemote} or an {@code OpProxy}.
     * @param  other The other {@code Object} to compare.
     * @return {@code true} if and only if the two objects are equal.
     */
    public boolean equals(Object other) {
	///System.out.println("  ** OpImpl");
	if (other instanceof OpRemote) {
	    /// System.out.println("  ** OpImpl vs OpRemote");
	    OpRemote otherOpRemote = (OpRemote)other;
	    return OpRemote.Helper.twoOpRemotesEqual(this, otherOpRemote);
	}
	if (other instanceof OpProxy) {
	    ///System.out.println("  ** OpImpl vs OpProxy");
	    OpProxy otherOpProxy = (OpProxy)other;
	    return OpRemote.Helper.twoOpRemotesEqual(this, otherOpProxy.getOpRemote());
	}
	if (other == null) {
	    return false;
	}
	throw new rjRuntimeError("RJ internal error: unknown OpImpl");
    }

    /**
     * Returns a hash code for this {@code OpImpl}.
     * The hashCode for {@code noop} is same across all VMs.
     * The hashCodes for non-{@code noop}s are different on and across VMs.
     * @return a hash code value for this object.
     */
    public int hashCode() {
	if (isNoop()) {
	    return "noop".hashCode();
	}
	return getId().hashCode();
    }

}
