package edu.ucdavis.rj;

import java.rmi.*;
import java.rmi.server.*;
import java.util.List;
import java.util.LinkedList;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * This class represents a lock for multiple processes trying to access
 * an InOp's message queue.
 *
 * EClock feature gives a thread (of which there can be only one active
 * because of a guarantee through jrx) trying to form an equivalence class
 * priority.
 *
 */
public class InLock_impl
    extends UnicastRemoteObject
    implements InLock
{
    static final long serialVersionUID = 0;
    private boolean locked;
    private PriorityList waitQ;		// assumed to be sorted by timestamp
    private PriorityList entryQ;	// assumed to be sorted by timestamp
    private PLIterator waitIt;
    private long onQc, enQc;
    private InLock theLock;
    private boolean switching;
    private long timestamp;
    private PairBoolean offQueue;
    private Thread thrd;
    private LockId myLockId;
    private Hashtable<OpRemote, OpRemote> waitingOps;
    private Semaphore mutex, forward;

    protected static final long ZONE_OFFSET =
	Calendar.getInstance().get(Calendar.ZONE_OFFSET);
    protected static final long DST_OFFSET =
	Calendar.getInstance().get(Calendar.DST_OFFSET);
    protected static final boolean ISEC = true,
				   ISMES = true;
    protected static final int MESSAGES = 0x0001,
			       ENTRY = 0x0010,
			       ECSELECT = 0x0010;

    public InLock_impl(String name, boolean locked)
	throws RemoteException
    {
	this.locked = locked;
	this.waitQ = new PriorityList();
	this.entryQ = new PriorityList();
	this.onQc = 0;
	this.enQc = 0;
	this.switching = false;
	this.theLock = this;
	this.mutex = new Semaphore(1);
	//this.myLockId = new LockId(jrvm.thisVM.getName(), nextLockNumber());
	this.myLockId = new LockId(name, nextLockNumber());
	this.waitingOps = new Hashtable<OpRemote, OpRemote>();
	rjvm.registerRemote(this);
    }

    // START: lock section
    public void lock()
	throws RemoteException
    {
	rjvm.ariseAndReceive();  // from caller
	try
	{
	    lock(!ISEC);
	}
	finally
	{
	    rjvm.sendAndDie(); // to caller
	}
    }

    public void lockEC()
	throws RemoteException
    {
	rjvm.ariseAndReceive();  // from caller
	try
	{
	    lock(ISEC);
	}
	finally
	{
	    rjvm.sendAndDie(); // to caller
	}
    }

    protected void lock(boolean isEClock)
	throws RemoteException
    {
	// get timestamp
	long mytimestamp = System.currentTimeMillis() +
				ZONE_OFFSET + DST_OFFSET;

	// the following looks like a spin lock but it really isn't
	// this loop just facilitates inner lock switches
	//while (!theLock.acquireInnerLock(mytimestamp, isEClock));
	boolean stopLoop;
	do
	{
	    rjvm.sendAndDie();  // to acquire
	    try
	    {
		stopLoop = theLock.acquireInnerLock(mytimestamp, isEClock);
	    }
	    finally
	    {
		rjvm.ariseAndReceive();  // from acquire
	    }
	} while (!stopLoop);
    }

    public boolean acquireInnerLock(long mytimestamp,
	boolean isEClock)
	throws RemoteException
    {
	boolean hadWaited = false, wrongLock = false;

	rjvm.ariseAndReceive();  // from caller
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "**ACQUIRE**: " + mytimestamp + ":" + Thread.currentThread());

	//synchronized (this)
	mutex.acquire();
	{
	    if (this.theLock != this)
	    {
		// lock has changed
		mutex.release();
		rjvm.sendAndDie();  // to acquire
		return this.theLock.acquireInnerLock(mytimestamp, isEClock);
		// receive from acquire and send to caller cancel each other
	    }
	    else
	    {
		PairBoolean myOffQueue = new PairBoolean(false, false);
		if (locked)
		{
		    // wait on entryQ
		    entryQ.insert(mytimestamp, Thread.currentThread(),
				isEClock, myOffQueue);
		    enQc++;
		    hadWaited = true;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ACQUIRE 1: enQc: " + enQc + " onQc: " + onQc + ":" + mytimestamp + ":" + Thread.currentThread());

		    // wait on entry queue
		    mutex.release();
		    synchronized (myOffQueue)
		    {
			// if somebody wakes me before I actually sleep,
			// I need to pretend to have died so that the
			// counts work out, otherwise I need to die
			rjvm.threadDeath();
			while (!myOffQueue.getFirstValue()) // &&
			    //((timestamp != mytimestamp) ||
			    //(thrd != Thread.currentThread())))
			{
			    try
			    {
				myOffQueue.wait();
			    }
			    catch (InterruptedException e)
			    {
				/* should not have been interrupted */
			    }
			}
		    }
		    mutex.acquire();

		    if (switching)
		    {
			enQc--;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ACQUIRE 2: enQc: " + enQc + ":" + mytimestamp + ":" + Thread.currentThread());
			// if (enQc == 0) this.notifyAll();
			if ((onQc == 0) && (enQc == 0))
			{
			    // last guy out, turn off the lights
			    forward.release();
			}
			mutex.release();
			rjvm.sendAndDie(); // to caller
			return false;
		    }
		}

		//if (!waitQ.empty() && !isEClock)
		// if the thread had waited then might just be moving
		// from one queue to another, otherwise it should be
		// allowed to acquire the lock and check the message
		// queue
		if (hadWaited && !waitQ.empty() && !isEClock)
		{
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ACQUIRE 3: onQc: " + onQc + ":" + mytimestamp + ":" + Thread.currentThread());

		    // wait on waitQ
		    mutex.release();
		    synchronized (myOffQueue)
		    {
			// if somebody wakes me before I actually sleep,
			// I need to pretend to have died so that the
			// counts work out, otherwise I need to die
			rjvm.threadDeath();
			while (!myOffQueue.getSecondValue())
			   //(timestamp != mytimestamp) ||
			   //(thrd != Thread.currentThread()))
			{
			    try
			    {
				myOffQueue.wait();
			    }
			    catch (InterruptedException e)
			    {
				/* should not have been interrupted */
			    }
			}
		    }
		    mutex.acquire();

		    if (switching)
		    {
			onQc--;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ACQUIRE 4: onQc: " + onQc + ":" + mytimestamp + ":" + Thread.currentThread());
			//if (onQc == 0) this.notifyAll();
			if ((onQc == 0) && (enQc == 0))
			{
			    // last guy out, turn off the lights
			    forward.release();
			}
			mutex.release();
			rjvm.sendAndDie();  // to caller
			return false;
		    }
		}
		else if (!isEClock)
		{
		    // place on the queue
		    PNode node = waitQ.insert(mytimestamp,
			Thread.currentThread(), isEClock, myOffQueue);
		    onQc++;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ACQUIRE 5: onQc: " + onQc + ":" + mytimestamp + ":" + Thread.currentThread());
		    this.timestamp = mytimestamp;
		    this.offQueue = myOffQueue;
		    this.thrd = Thread.currentThread();
		    //waitIt = waitQ.iterator();
		    waitIt = new PLIterator(node, waitQ);
		}

		locked = true;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> RETURNING " + Thread.currentThread());

		mutex.release();
		rjvm.sendAndDie();  // to caller
		return true;
	    }
	}

	// can't get here
    }

    public boolean lock_or_register(OpRemote theop)
	throws RemoteException
    {
	return this.theLock.innerLorR(theop);
    }
    public boolean innerLorR(OpRemote theop)
	throws RemoteException
    {
	boolean wrongLock = false;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "**LOCKORREGISTER**");
	//synchronized (this)
	mutex.acquire();
	{
	    if (this.theLock != this)
	    {
		mutex.release();
		return this.theLock.lock_or_register(theop);
	    }
	    else
	    {
		if (locked)
		{
		    // register
		    waitingOps.put(theop, theop);
		    mutex.release();
		    return false;
		}
		else
		{
		    locked = true;
		    mutex.release();
		    return true;
		}
	    }
	}

	// can't get here
    }
    // END: lock section

    // START: unlock section
    public void unlock()
	throws RemoteException
    {
	rjvm.ariseAndReceive(); // from caller
	try
	{
	    // rjvm.sendAndDie(); // to release
	    theLock.releaseInnerLock(!ISEC, !ISMES);
	}
	finally
	{
	    // rjvm.ariseAndReceive(); // from release
	    rjvm.sendAndDie(); // to caller
	}
    }

    public void unlockEC()
	throws RemoteException
    {
	rjvm.ariseAndReceive(); // from caller
	try
	{
	    // rjvm.sendAndDie(); // to release
	    theLock.releaseInnerLock(ISEC, !ISMES);
	}
	finally
	{
	    // rjvm.ariseAndReceive(); // from release
	    rjvm.sendAndDie(); // to caller
	}
    }

    public void unlockMes()
	throws RemoteException
    {
	// currently sending and receiving from releaseInnerLock,
	// but probably don't need to -- a thread should never
	// block while releasing a lock
	try
	{
	    // rjvm.sendAndDie(); // to release
	    theLock.releaseInnerLock(!ISEC, ISMES);
	}
	finally
	{
	    // rjvm.ariseAndReceive(); // from release
	}
    }

    // the release will pass the lock to the next selected thread
    // unfortunately, if this thread dies then deadlock occurs
    // ideally the dynamic deadlock detection will detect such a state and
    // halt the program
    public void releaseInnerLock(boolean isECunlock, boolean isMesUnlock)
	throws RemoteException
    {
	// rjvm.ariseAndReceive(); // from caller

	boolean wrongLock = false;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "**RELEASE**:" + timestamp + ":" + thrd);
	//synchronized (this)
	mutex.acquire();
	{
	    if (this.theLock != this) 
	    {
		mutex.release();
		// rjvm.sendAndDie(); // to inner
		try
		{
		    this.theLock.releaseInnerLock(isECunlock, isMesUnlock);
		}
		finally
		{
		    // rjvm.ariseAndReceive(); // from inner
		    // rjvm.sendAndDie(); // to caller
		}
		return;
	    }
	    else
	    {
		timestamp = -1;
		thrd = null;

		// remove the thread that just unlocked the lock
		if (!isECunlock && !isMesUnlock)
		{
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "RELEASE: onQc: " + onQc);
		    if (!waitIt.isNull())
		    {
			waitIt.remove();
			onQc--;
		    }
		    else
		    {
System.err.println("waitIt is null for some reason -- this probably isn't good");
		    }
		}

		// move entry waiting threads
		// allow registered ops to deliver messages
		int output;  // (EC thread, messages delivered)

		output = ((InLock_impl)this.theLock).
				allowDeliveries(MESSAGES | ENTRY);

		// if an EC thread was selected to run then don't select
		// a new thread
		if ((output & ECSELECT) != 0)
		{
		    mutex.release();
		    // rjvm.sendAndDie();  // to caller
		    return;
		}

		// if there is no change, then don't select a new thread
/*
		This has a bug that allows threads to gather on the waitQ
		and never be released even if there are messages that these
		threads have never looked at and can accept.
		if (isECunlock && ((output & MESSAGES) == 0))
		{
		    locked = false;
		    mutex.release();
		    // rjvm.sendAndDie();  /// to caller
		    return;
		}
*/

		// select a new thread
		// remove dead threads
		while (!waitQ.empty() && !waitQ.thread().isAlive())
		{
		    waitQ.removeHead();
		    onQc--;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "RELEASE 2: onQc: " + onQc);
		}

		// wake up the next timestamp
		if (onQc > 0)
		{
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "PICKING THREAD: " + waitQ.thread());
		    this.timestamp = waitQ.headPriority();
		    this.offQueue = waitQ.getLock();
		    this.thrd = waitQ.thread();
		    this.waitIt = waitQ.iterator();

		    if (this.offQueue == null)
		    {
			throw new rjRuntimeError("Null lock on queue : 2");
		    }

		    synchronized (this.offQueue)
		    {
			rjvm.threadBirth();
			this.offQueue.setSecondValue(true);
			this.offQueue.notify();
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "WAKING A THREAD : " + timestamp + ":" + thrd);
		    }
		}
		else
		{
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "UNLOCKING");
		    locked = false;
		}
	    }
	}
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "UNLOCKING bye");
	mutex.release();
	// rjvm.sendAndDie(); // to caller
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "UNLOCKING done");
    }
    // END: unlock section

    // START: wait section
    public void waitOnLock()
	throws RemoteException
    {
	////2012-10-10 Debug.println("InLoc_impl +waitOnLock()");
	rjvm.ariseAndReceive();  // from caller
	long mytimestamp = 0;
	rjvm.sendAndDie();  // to inner
	try
	{
	    mytimestamp = theLock.waitInnerLock();
	}
	finally
	{
	    rjvm.ariseAndReceive();  // from inner
	}
	////2012-10-10 Debug.println("InLoc_impl pre-mytimestamp >= 0 mytimestamp="+mytimestamp);
	//if ((mytimestamp = theLock.waitInnerLock()) >= 0)
	if (mytimestamp >= 0)
	{
	    //while (!theLock.acquireInnerLock(mytimestamp, false));
	    boolean stopLoop;
	    do
	    {
		rjvm.sendAndDie();  // to acquire
		try
		{
		    stopLoop = theLock.acquireInnerLock(mytimestamp, false);
		}
		finally
		{
		    rjvm.ariseAndReceive();  // from acquire
		}
	    } while (!stopLoop);
	}
	rjvm.sendAndDie();  // to caller
    }

    // allow the next thread to attempt to execute
    public long waitInnerLock()
	throws RemoteException
    {
	rjvm.ariseAndReceive();  // from caller

	//synchronized (this)
	mutex.acquire();
	{
	    long mytimestamp = timestamp;
	    PairBoolean myOffQueue = offQueue;
	    PLIterator waitClone;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "**WAIT**: " + mytimestamp + ":" + Thread.currentThread());
	    if (this.theLock != this) 
	    {
		mutex.release();
		rjvm.sendAndDie();
		return this.theLock.waitInnerLock();
		// receive and send cancel each other
	    }
	    else
	    {
		////2012-10-10 Debug.println("InLoc_impl +waitInnerLock() else");
		timestamp = -1;
		offQueue = null;
		thrd = null;

		// update the thread value
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "WAIT UPDATING THREAD: " + mytimestamp + ":" + Thread.currentThread() + " OLD => " + waitIt.priority() + ":" + waitIt.thread());
		waitIt.setThread(Thread.currentThread());

		waitClone = waitIt.cloneIterator();
		waitIt.next();
		// select a new thread
		// remove dead threads
		while ((!waitIt.isNull())  && !waitIt.thread().isAlive())
		{
		    waitIt.remove();  // steps to next element
		    onQc--;
//System.out.println("KILLING: " + myLockId.site + "::" + myLockId.num + " -> " + "WAIT: onQc: " + onQc + ":" + mytimestamp + ":" + Thread.currentThread());
		}

		// I might need to sleep
		myOffQueue.setSecondValue(false);

		// wake up the next timestamp
		if (!waitIt.isNull())
		{
		    waitClone = null;
		    this.timestamp = waitIt.priority();
		    this.offQueue = waitIt.getLock();
		    this.thrd = waitIt.thread();
		    if (this.offQueue == null)
		    {
			throw new rjRuntimeError("Null lock on queue : 3");
		    }
		    synchronized (this.offQueue)
		    {
			rjvm.threadBirth();
			this.offQueue.setSecondValue(true);
			this.offQueue.notify();
		    }
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "TELL NEXT" + ":" + timestamp + ":" + thrd);
		}
		else
		{
		    long oldQc = onQc;
		    int output = ((InLock_impl)this.theLock).allowDeliveries(
			MESSAGES | ENTRY);
		    // if an EC thread was selected to run then don't select
		    // a new thread
		    if ((output & ECSELECT) != 0)
		    {
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ECSELECT" + ":" + mytimestamp + ":" + Thread.currentThread());
			// intentionally blank
			waitClone = null;
		    }
		    else if (((output & MESSAGES) != 0) || (oldQc < onQc))
		    {
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "MESSAGES" + ":" + mytimestamp + ":" + Thread.currentThread());
			waitIt = waitQ.iterator();
			waitClone = null;
			this.timestamp = waitIt.priority();
			this.offQueue = waitIt.getLock();
			this.thrd = waitIt.thread();
			if (this.offQueue == null)
			{
			    throw new rjRuntimeError("Null lock on queue : 1");
			}
			synchronized (this.offQueue)
			{
			    rjvm.threadBirth();
			    this.offQueue.setSecondValue(true);
			    this.offQueue.notify();
			}
		    }
		    else
		    {
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "UNLOCKED : " + onQc + " : " + enQc + ":" + mytimestamp + ":" + Thread.currentThread());
			waitClone = null;
			locked = false;
		    }
		}

		mutex.release();
		////2012-10-10 Debug.println("InLoc_impl waitInnerLock() post-mutex.release()");
		synchronized (myOffQueue)
		{
		    // if somebody wakes me before I actually sleep,
		    // I need to pretend to have died so that the
		    // counts work out, otherwise I need to die
		    rjvm.threadDeath();
		    while (!myOffQueue.getSecondValue())
			//(timestamp != mytimestamp) ||
			//(thrd != Thread.currentThread()))
		    {
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "SLEEPING on " + myOffQueue + ": " + mytimestamp + ":" + Thread.currentThread());
			try
			{
			    ////2012-10-10 Debug.println("InLoc_impl waitInnerLock() pre-myOffQueue.wait()");
			    myOffQueue.wait();
			    ////2012-10-10 Debug.println("InLoc_impl waitInnerLock() post-myOffQueue.wait()");
			}
			catch (InterruptedException e)
			{
			    // should not have been interrupted
			}
		    }
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "I AM AWAKE: " + mytimestamp + ":" + Thread.currentThread());
		}
		mutex.acquire();
		////2012-10-10 Debug.println("InLoc_impl waitInnerLock() post-mutex.acquire() 2nd");
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "AWAKE WITH LOCK: " + mytimestamp + ":" + Thread.currentThread());
		if (switching)
		{
		    onQc--;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "WAIT 2: onQc: " + onQc);
		    //if (onQc == 0) this.notifyAll();
		    if ((onQc == 0) && (enQc == 0))
		    {
			// last guy out, turn off the lights
			forward.release();
		    }
		    mutex.release();
		    rjvm.sendAndDie();  // to caller
		    return mytimestamp;
		}
	    }
	}
	mutex.release();
	rjvm.sendAndDie();  // to caller
	return -1;
    }
    // END: wait section

    public InLock getInnerLock()
	throws RemoteException
    {
	mutex.acquire();
	try
	{
	    if (this.theLock != this)
	    {
		return theLock.getInnerLock();
	    }
	    else
	    {
		return this;
	    }
	}
	finally
	{
	    mutex.release();
	}
    }

    public LockId getLockId()
	throws RemoteException
    {
	mutex.acquire();
	try
	{
	    if (this.theLock != this)
	    {
		return theLock.getLockId();
	    }
	    else
	    {
		return myLockId;
	    }
	}
	finally
	{
	    mutex.release();
	}
    }

    // must hold lock
    // does not currently check to determine that the lock holder is doing
    // the forwarding
    public void forwardTo(InLock lock)
	throws RemoteException
    {
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "**FORWARDTO**" + Thread.currentThread());
	rjvm.ariseAndReceive();  // from caller
	boolean wrongLock = false;
	//synchronized (this)
	mutex.acquire();
	{
	    if (this.theLock != this)
	    {
		mutex.release();
		rjvm.sendAndDie();  // to forward
		theLock.forwardTo(lock);
		// receive and send cancel each other
		return;
	    }
	    else
	    {
		if (!locked)
		{
		    mutex.release();
		    rjvm.sendAndDie();  // to caller
		    // throw new LockNotHeldError(); 
		    new rjRuntimeError("RJ internal error: "+
				       "lock not held");
		}
if (this.theLock == lock)
{
   System.err.println("\n\t **** Forwarding to self ****");
   System.err.println("\t  Please report this error. \n");
}
		((InLock_impl)this.theLock).allowDeliveries(MESSAGES);
		if (theLock != lock)
		{
		    theLock = lock.getInnerLock();
                    myLockId = theLock.getLockId();
		}
		// make the threads attempt to lock again so that they will
		// be transfered to the new locks synchronized queue
		switching = true;

		// need to awaken all threads on the queue
		PLIterator iter = entryQ.iterator();
		while(!iter.isNull())
		{
		    //System.out.println(myLockId.site + "::" + myLockId.num + " -> " + Thread.currentThread() + " Awaking on entryQ " + iter.getLock() + ":" + iter.thread());
		    rjvm.threadBirth();
		    iter.releaseFirst();
		    //iter.getLock().setFirstValue(true);
		    //iter.getLock().notify();
		    iter.next();
		}
		iter = waitQ.iterator();
		while(!iter.isNull())
		{
		    PairBoolean pair = iter.getLock();
		    //System.out.println(myLockId.site + "::" + myLockId.num + " -> " + Thread.currentThread() + " Awaking on waitQ " + pair + ":" + iter.thread());
		    rjvm.threadBirth();
		    iter.releaseSecond();
		    iter.next();
		}

		if ((onQc + enQc) > 0)
		{
		    forward = new Semaphore(0);
		    mutex.release();
		    forward.acquire();  // wait until they're all done
		    mutex.acquire();
		}

		waitQ.clear();
		entryQ.clear();
		waitQ = null;
		entryQ = null;
		waitingOps = null;
	    }
	}
	rjvm.sendAndDie();  // to caller
	mutex.release();
    }

    private static int lockNumber = 0;
    public static synchronized int nextLockNumber()
    {
	return lockNumber++;
    }

    private int allowDeliveries(int which)
    {
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "**ALLOW**" + Thread.currentThread());
	int retval = 0;
	// notify each registered operation to deliver pending messages
	if ((which & MESSAGES) != 0)
	{
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "\tmessages begin");
	    Enumeration iter = waitingOps.elements();
	    if (iter.hasMoreElements())
		retval |= MESSAGES;
	    for ( ;iter.hasMoreElements(); )
	    {
		try{
		((OpRemote)iter.nextElement()).deliverPendingMessages();
		} catch (Exception e)
		{
		// what should I do here? ugh
		}
	    }
	    waitingOps.clear();
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "\tmessages end");
	}

	// move waiting (non-EC) threads from the entry queue
	if ((which & ENTRY) != 0)
	{
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "\tentry begin");
	    PLIterator iter = entryQ.iterator();
	    while(!iter.isNull())
	    {
		if (iter.isEC())
		{
		    retval |= ECSELECT;
		    iter.next();
		}
		else
		{
		    waitQ.insert(iter.priority(), iter.thread(),
			false, iter.getLock());
		    onQc++;
		    rjvm.threadBirth();
		    iter.releaseFirst();
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ALLOW: onQc: " + onQc);
		    iter.remove();
		    enQc--;
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "ALLOW 2: enQc: " + enQc);
		}
	    }

	    if ((retval & ECSELECT) != 0)
	    {
		iter = entryQ.iterator();
		timestamp = iter.priority();
		thrd = iter.thread();
		rjvm.threadBirth();
		iter.releaseFirst();
		iter.remove();
		enQc--;
	    }
//System.out.println(myLockId.site + "::" + myLockId.num + " -> " + "\tentry end");
	}
	return retval;
    }
}
