package edu.ucdavis.rj;

import java.util.*;

import java.rmi.*;

/**
 * An operation that's to be serviced via inni.
 */
abstract public class OpInni extends OpImpl {

    static final long serialVersionUID = 0;

    protected InLock theLock;

    protected List<Invocation> msgList  = new LinkedList<Invocation>();
    protected List<Invocation> waitList = new LinkedList<Invocation>();

    /**
     * Create a new instance of an operation that's to be serviced
     * via inni.
     * @throws java.rmi.RemoteException if RMI fails.
     */
     public OpInni() throws RemoteException {
	//////2012-10-09	Debug.println("===>OpInni constructor");
	theLock = new InLock_impl(rjvm.vmName, false);
     }

    /**
     * Instantiate an {@code OpInni}.
     * This method just calls the appropriate constructor,
     * but hides the {@code java.rmi.RemoteException}.
     * @return The new instance of the {@code OpInni}.
     */
    public static OpInni newOpInni() {
	OpInni ret = null;
	try {
	    // System.out.println("ABC "+ ThreadLocalReplyOp.get());
	    ret = new OpInniNonNoop();
	} catch (java.rmi.RemoteException e) {
	    throw new rjRuntimeError("RJ internal error: OpInni.newOpInni");
	}
	return ret;
    }

    /**
     * Instantiate an {@code OpInni} that looks somewhat like a semaphore.
     * @param initialSemValue The initial value of the semaphore.
     * @return The new instance of the {@code OpInni}.
     */
    public static OpInni newSem(int initialSemValue) {
	OpInni ret = null;
	try {
	    ret = new OpInniNonNoop();
	} catch (java.rmi.RemoteException e) {
	    throw new rjRuntimeError("RJ internal error: OpInni.newSem");
	}
	if (initialSemValue < 0) {
	    throw new rjRuntimeError("OpInni.newSem negative initialSemValue: "+
				     initialSemValue);
	}
	// set the semaphore's initial value
	// by generating initialSemValue invocations.
	for (int k = 1; k <= initialSemValue; k++) {
	    ret.V();
	}
	return ret;
    }

    /**
     * Default semaphore value is {@value}.
     * (@code public} so javadoc can see it for {@code newSem()}.
     */
    public static final int defaultSemValue = 0;

    /**
     * Invokes {@link #newSem(int)} with {@value #defaultSemValue}.
     * @return the new semaphore.
     */
    public static OpInni newSem() {
	return newSem(defaultSemValue);
    }

    /**
     * Invoking the noop operation via send has no effect.
     * Invoking the noop operation via call has no effect;
     * the invocation's return value is set to null.
     * ("no effect" means other than possible side effects
     * in parameter evaluation.)
     * noop's invocation queue is always empty.
     * Thus,
     * servicing the noop operation with receive causes the
     * servicing process to block forever;
     * servicing the noop operation within an inni arm means
     * that arm will never be selected for servicing.
     */
    public static final OpInni noop;

    static {
	// myNoop and finally block are to get around
	// javac complaint that noop might be uninitialized.
	OpInni myNoop = null;
	try {
	    myNoop = new OpInniNoop();
	} catch (RemoteException e) {
	    throw new rjRuntimeError("RJ internal error: OpInni's noop instantiation");
	}
	finally {
	    noop = myNoop;
	}
    }

    /**
     * Returns the number of Invocations pending for this operation.
     * @return The number of pending Invocations.
     */
    public int length() {
	return msgList.size();
    }

    /**
     * Get the first (oldest) pending invocation of this operation.
     * Wait if no invocation is pending.
     * @return The first pending invocation.

     */
    public Invocation receive() {
	try {
	    Invocation ret;
	    boolean empty, exception = false;
            rjvm.sendAndDie(); // as receiver in RJ generated code
	    rjvm.ariseAndReceive(); // from caller
	    try {
		exception = true;
		rjvm.sendAndDie(); // to lock
		this.theLock.lock();
		rjvm.ariseAndReceive(); // from lock

		exception = false;
		while (true) {
		    empty = false;
		    synchronized (this.msgList) {
			if (this.msgList.size() == 0)
			    empty = true;
		    }
		    if (empty) {
			exception = true;
			rjvm.sendAndDie(); // to lock
			this.theLock.waitOnLock();
			rjvm.ariseAndReceive(); // from lock
			exception = false;
		    }
		    else
			break;
		}

		try {
		    ret = this.msgList.remove(0);
		}
		catch (IndexOutOfBoundsException doh) {
		    exception = true;
		    rjvm.sendAndDie(); // to lock
		    this.theLock.unlock();
		    rjvm.ariseAndReceive(); // from lock
		    exception = false;
		    throw new rjRuntimeError("Unexpected empty operation!");
		}
		catch (Exception doh) {
		    exception = true;
		    rjvm.sendAndDie(); // to lock
		    this.theLock.unlock();
		    rjvm.ariseAndReceive(); // from lock
		    exception = false;
		    throw new rjRuntimeError("Exception while receiving from an operation!");
		}

		// release the lock
		exception = true;
		rjvm.sendAndDie(); // to lock
		this.theLock.unlock();
		rjvm.ariseAndReceive(); // from lock
		exception = false;
		if (ret == null)
		    throw new rjRuntimeError("Null receive node!");
		//////////////// in RJ, param is extracted before reply to invoker
		////////////////////// shouldn't matter.
		///////// missing rjvm.sendAndDie rjvm.ariseAndReceive pair
		/////////// e.g., see JR vsuite/receive/1
		//////// ..... maybe should do around all this???
		rjvm.ariseAndReceive(); // as receiver in RJ generated code
		ret.replyToInvoker();
		return ret;
	    }
	    finally {
		if (!exception) {
		    rjvm.sendAndDie(); // to caller
		}
	    }
	} catch (java.rmi.RemoteException e) {
	    throw new rjRuntimeError("RJ internal error: OpInni.receive");
	}

        ///////////////// how easy (but wrong) this used to be ;-)
	/*******************
	synchronized(lock) {
	    while(list.isEmpty()) {
		try {
		    lock.wait();
		} catch (java.lang.InterruptedException e) {
		    System.err.println("receive caught java.lang.InterruptedException");
		}
	    }
	    ret = list.remove(0);
	}
	ret.replyToInvoker();
	return ret;
	****************/
    }

    // documentation inherited
    public Invocation P() {
	return this.receive();
    }

    /**
     * Return timestamp (non-negative) of first invocation in this queue,
     * if there are any invocations; otherwise, return -1.
     * (For noop, queue is empty.)
     * @return The timestamp of the first pending invocation.
     */
     synchronized public long getFirstTime() {
	 if (this.msgList.size() != 0)
	     return (this.msgList.get(0)).timestamp.getTimestamp();
	 return -1;
     }

    // (For noop, queue is empty.)
    public boolean canAppearInInni()
	throws java.rmi.RemoteException {
	return true;
    }

    // (noop is treated just like an other OpInni.)
    public synchronized InLock getLock()
	throws java.rmi.RemoteException {
	return this.theLock;
    }

    public boolean isRemote(String site) throws java.rmi.RemoteException {
	return (rjvm.thisVM.getName().compareTo(site) != 0);
    }

    public synchronized void deliverPendingMessages() throws java.rmi.RemoteException {
        // the equiv class should already be locked
	try     {
	    synchronized (this.waitList) {
		synchronized (this.msgList) {
		    while (this.waitList.size() != 0) {
			this.msgList.add(this.waitList.remove(0));
		    }
		}
	    }
	}
	catch (IndexOutOfBoundsException doh) {
	    throw new rjRuntimeError("Unexpected empty operation!");
	}
	catch (Exception doh) {
	    throw new rjRuntimeError("Exception while sending to an operation!");
	}
    }




    public synchronized OpInniIterator elements()
	throws java.rmi.RemoteException {
	////2012-10-09Debug.println("===>OpInniArm elements this.msgList.size="+this.msgList.size());
	return new OpInniIteratorImpl(new OpInniIteratorRemoteImpl(this.msgList));
    }

}
