package edu.ucdavis.rj;

/**
 * The {@code RJ} class contains several useful methods.
 */
public class RJ
{
    private static VM thisVM = null;

    /**
     * Terminates the currently running RJ program.
     * The argument serves as a status code;
     * by convention, a nonzero status code indicates abnormal termination.
     * This method never returns normally.
     * @param status The exit status.
     */
    public static void exit(int status)
    {
	try
	{
	    rjvm.rjx.exit(status);
	}
	catch (Exception e)
	{
	    System.err.println("error exiting ... exit forced");
	    System.exit(status);
	}
    }

    /**
     * Invokes {@link #nap(long,int)} on {@code nap(millis,0)}.
     * @param millis the length of time to sleep in milliseconds.
     */
    public static void nap(long millis)
    {
	nap(millis, 0);
    }

    /**
     * Causes the currently executing thread to sleep (cease execution)
     * for the specified number of milliseconds
     * plus the specified number of nanoseconds.
     *
     * {@code nap} is just like Java's {@code Thread.sleep}
     * except that {@code nap} catches any {@code InterruptedException}.
     *
     * Reminder: Java's {@code Thread.sleep}
     * throws {@code IllegalArgumentException} for negative {@code millis}
     * or {@code nanos} out of range 0-999999.
     *
     * @param millis the length of time to sleep in milliseconds.
     * @param nanos 0-999999 additional nanoseconds to sleep.
     */
    public static void nap(long millis, int nanos)
    {
	try
            {
		Thread.sleep (millis, nanos);
            }
	catch(InterruptedException e){ /* nothing needed */ }
    }

    /**
     * Causes the currently executing thread
     * to temporarily pause and allow other threads to execute.
     */
    public static void yield()
    {
	Thread.yield ();
    }

    /**
     * Registers the quiescence action.
     * The argument is the operation that is invoked
     * implicitly via {@code send} when the RJ program quiesces.
     * @param cap The quiescence operation.
     * @throws QuiescenceRegistrationException any problems.
     */
    public static void registerQuiescenceAction(OpProxy cap)
	throws edu.ucdavis.rj.QuiescenceRegistrationException
    {
	try
	{
	    rjvm.rjx.registerQuiescenceAction(cap);
	}
	catch (Exception e)
	{
	    throw new edu.ucdavis.rj.QuiescenceRegistrationException();
	}
    }

}
