package edu.ucdavis.rj;

import java.util.*;

/**
 * Represents a possibly quantified arm on an inni.
 *
 * It has many constructors to allow for optional parts
 * and quantified operations in different forms
 * ({@code OpInni} or {@code OpProxy}).
 */
public class InniArm {

    Quantifiers quants;
    OpProxy op;
    SuchThat st;
    By by;
    ArmCode code;

    // either oa or op will be null
    OpArray oa;

    // general, quantified case:
    // (all single OpProxy cases pass through here)
    //////////// explain more in comment........ about quant value or null.
    InniArm(Quantifiers quant, OpProxy op, SuchThat st, By by, ArmCode code) {
	if (op == null) {
	    throw new NullPointerException("OpProxy given to InniArm is null");
	}
	commonSingle(quant, op, st, by, code);
    }

    void commonSingle(Quantifiers quant, OpProxy op, SuchThat st, By by, ArmCode code) {
	checkOpProxyIsForOpInni(op, -1);
	this.quants = quant;
	this.op = op;
	this.st = st;
	this.by = by;
	this.code = code;
    }

    // provides constructors for
    //    OpInni
    //    OpProxy
    // the OpInni constructors are provided to make the user code
    // a bit cleaner -- so user doesn't need to convert it to OpProxy.
    // similar comments apply to OpArray.

    // single OpInni:  general, unquantified case:
    // (all single OpInni cases pass through here)
    public InniArm(OpInni op, SuchThat st, By by, ArmCode code) {
	if (op == null) {
	    throw new NullPointerException("OpInni given to InniArm is null");
	}
	commonSingle( null, new OpProxy(op), st, by, code);
    }

    // single OpInni cases w/o everything specified:
    public InniArm(OpInni op, SuchThat st, By by              ) {
	this(             op,          st,    by,         null);
    }
    public InniArm(OpInni op, SuchThat st,        ArmCode code) {
	this(             op,          st,  null,         code);
    }
    public InniArm(OpInni op, SuchThat st                     ) {
	this(             op,          st,  null,         null);
    }
    public InniArm(OpInni op,              By by, ArmCode code) {
	this(             op,        null,    by,         code);
    }
    public InniArm(OpInni op,              By by              ) {
	this(             op,        null,    by,         null);
    }
    public InniArm(OpInni op,                     ArmCode code) {
	this(             op,        null,  null,         code);
    }
    public InniArm(OpInni op                                  ) {
	this(             op,        null,  null,         null);
    }

    // single OpProxy:  general, unquantified case:
    // (all single OpProxy cases pass through here)
    public InniArm(OpProxy op, SuchThat st, By by, ArmCode code) {
	this( null, op, st, by, code);
    }

    // single OpProxy cases w/o everything specified:
    public InniArm(OpProxy op, SuchThat st, By by              ) {
	this(              op,          st,    by,         null);
    }
    public InniArm(OpProxy op, SuchThat st,        ArmCode code) {
	this(              op,          st,  null,         code);
    }
    public InniArm(OpProxy op, SuchThat st                     ) {
	this(              op,          st,  null,         null);
    }
    public InniArm(OpProxy op,              By by, ArmCode code) {
	this(              op,        null,    by,         code);
    }
    public InniArm(OpProxy op,              By by              ) {
	this(              op,        null,    by,         null);
    }
    public InniArm(OpProxy op,                     ArmCode code) {
	this(              op,        null,  null,         code);
    }
    public InniArm(OpProxy op                                  ) {
	this(              op,        null,  null,         null);
    }



    // array of ops:  general, quantified case:
    // (all OpArray cases pass through here)
    public InniArm(OpArray oa, SuchThat st, By by, ArmCode code) {
	if (oa == null) {
	    throw new NullPointerException("OpArray given to InniArm is null");
	}
	this.oa = oa;
	this.st = st;
	this.by = by;
	this.code = code;
	checkArray();
    }

    private void checkArray() {
	// OpArray constructer already checked that sizes equal.
	oa.checkQuantifiers();
	oa.checkOps();
	int cnt = 0;
	for (OpProxy op: oa.opsList) {
	    checkOpProxyIsForOpInni(op, cnt);
	    cnt++;
	}
    }

    // array of ops cases w/o everything specified:
    public InniArm(OpArray oa, SuchThat st, By by              ) {
	this(              oa,          st,    by,         null);
    }
    public InniArm(OpArray oa, SuchThat st,        ArmCode code) {
	this(              oa,          st,  null,         code);
    }
    public InniArm(OpArray oa, SuchThat st                     ) {
	this(              oa,          st,  null,         null);
    }
    public InniArm(OpArray oa,              By by, ArmCode code) {
	this(              oa,        null,    by,         code);
    }
    public InniArm(OpArray oa,              By by              ) {
	this(              oa,        null,    by,         null);
    }
    public InniArm(OpArray oa,                     ArmCode code) {
	this(              oa,        null,  null,         code);
    }
    public InniArm(OpArray oa                                  ) {
	this(              oa,        null,  null,         null);
    }

    void checkOpProxyIsForOpInni(OpProxy op, int index) {
	try {
	    if (!op.canAppearInInni()) {
		String msg = "can't use an OpMethod in an InniArm";
		if (index>=0) {
		    msg += " (item "+index+" in op list) ";
		}
		throw new rjRuntimeError(msg);
	    }
	} catch (java.rmi.RemoteException e) {
	    throw new rjRuntimeError("RJ internal error:  InniArm op.canAppearInInni()");
	}
    }

    /////////// public so "jrx/InniLocker" can use it...
    public OpProxy getOpProxy() {
	return op;
    }

    ArmCode getCode() {
	return code;
    }

    /**
     * An array of operations for this InniArm
     * and (typically) the associated quantifier values.
     * The arm will service one of the operations.
     *
     * Chose not to define object that contains Quantifiers and an Op
     * since typical programs declare an array of Ops anyway.
     * So constructing an array of such objects would be slightly annoying
     * (vs. constructing an array of Quantifiers).
     *
     * Used to have constructors take OpInni, but changed that to OpImpl
     * because:
     *     CoArm (which extends this class) uses OpImpl
     *           (and want to keep this simple)
     *     Need to check OpProxy to see whether they are OpInni anyway,
     *       so this is consistent with that.
     */
    public static class OpArray {

	ArrayList<Quantifiers> quantsList;
	ArrayList<OpProxy> opsList;

	// extending classes should reinitialize this
	String classNameForErrors = "InniArm";
	
	public OpArray(Quantifiers [] quantsArray,
		       OpImpl [] ops) {
	    this.quantsList = safeNewArrayList(quantsArray);
	    this.opsList = getArrayListFromArray(ops);
	    commonCheck();
	}
	public OpArray(ArrayList<Quantifiers> quantsList,
		       ArrayList<OpImpl> opsList) {
	    this.quantsList = safeNewArrayList(quantsList);
	    this.opsList = getArrayListFromArrayList(opsList);
	    commonCheck();
	}

	// no quantifier: "implied" quantifier of entire array
	public OpArray(OpImpl [] ops) {
	    this.opsList = getArrayListFromArray(ops);
	    commonCheck();
	}
	// no quantifier: "implied" quantifier of entire array
	public OpArray(ArrayList<OpImpl> opsList,
                       Dummy ... dummy) {
	    this.opsList = getArrayListFromArrayList(opsList);
	    commonCheck();
	}

	// convert OpInni [] to ArrayList<OpProxy>
	// can't use (Arrays.asList(ops)) since need OpProxy
	ArrayList<OpProxy> getArrayListFromArray(OpImpl [] ops) {
	    ArrayList<OpProxy>  ret = new ArrayList<OpProxy>(ops.length);
	    // this checking duplicates checkOps,
	    // but needed here to protect  new OpProxy(null).
	    // could restructure things, but this isn't so bad.
	    int cnt = 0;
	    for (OpImpl op: ops) {
		if (op == null) {
		    throw new NullPointerException("op "+cnt+" in OpArray given to "+classNameForErrors+" is null");
		}
		cnt++;
		ret.add(new OpProxy(op));
	    }
	    return ret;
	}

	// convert ArrayList<OpInni> to ArrayList<OpProxy>
	// can't use (opsList.clone()) since need OpProxy
	ArrayList<OpProxy> getArrayListFromArrayList(ArrayList<OpImpl> opsList) {
	    ArrayList<OpProxy> ret = new ArrayList<OpProxy>(opsList.size());
	    for (OpImpl op: opsList) {
		ret.add(new OpProxy(op));
	    }
	    return ret;
	}

	public OpArray(Quantifiers [] quantsArray,
		       OpProxy [] ops) {
	    this.quantsList = new ArrayList<Quantifiers>(Arrays.asList(quantsArray));
	    this.opsList = new ArrayList<OpProxy>(Arrays.asList(ops));
	    commonCheck();
	}
	public OpArray(ArrayList<Quantifiers> quantsList,
		       ArrayList<OpProxy> opsList,
		       Dummy ... dummy) {
	    this.quantsList = (ArrayList<Quantifiers>)(quantsList.clone());
	    this.opsList = (ArrayList<OpProxy> )(opsList.clone());
	    commonCheck();
	}

	/**
	 * used to workaround Java's "same type erasure"
	 *for constructors for OpImpl and similars one for OpProxy.
	 * protected so that user can't actually use it.
	 * (so constructor invocation has 0 Dummy, as we want.)
	 */
	protected class Dummy {}


	// no quantifier: "implied" quantifier of entire array
	public OpArray(OpProxy [] ops) {
	    this.opsList = new ArrayList<OpProxy>(Arrays.asList(ops));
	    commonCheck();
	}
	// no quantifier: "implied" quantifier of entire array
	public OpArray(ArrayList<OpProxy> opsList) {
	    this.opsList = (ArrayList<OpProxy> )(opsList.clone());
	    commonCheck();
	}

	/////
	// checks needed by all constructors:
	//    null quantList
	// or same number of elements
	//
	// further semantic checks (elements not null, elements OpInni)
	// left for later (InniArm or CoArm)
	private void commonCheck() {
	    if (quantsList != null) {
		if (quantsList.size() != opsList.size()) {
		    String msg = "in constructing OpArray, "
			+"size of quantifiers list ("
			+quantsList.size() +") differs from size of op list ("
			+opsList.size()+")";
		    throw new rjRuntimeError(msg);
		}
	    }
	}

	// "safe" protects from null
	private ArrayList<Quantifiers> safeNewArrayList(ArrayList<Quantifiers> quantsList) {
	    return (quantsList == null)
		? null 
		: (ArrayList<Quantifiers>)(quantsList.clone());

	}

	// "safe" protects from null
	private ArrayList<Quantifiers> safeNewArrayList(Quantifiers [] quantsArray) {
	    return (quantsArray == null)
		? null 
		: new ArrayList<Quantifiers>(Arrays.asList(quantsArray));
	}


	ArrayList<OpProxy> getOpsList() {
	    return opsList;
	}
	ArrayList<Quantifiers> getQuantifiersList() {
	    return quantsList;
	}

	protected void checkQuantifiers() {
	    if (quantsList == null) {
		return;
	    }
	    int cnt = 0;
	    for (Quantifiers quant: quantsList) {
		if (quant == null) {
		    throw new NullPointerException("quantifier "+cnt+" in OpArray given to "+classNameForErrors+" is null");
		}
		cnt++;
	    }
	}

	protected void checkOps() {
	    int cnt = 0;
	    for (OpProxy op: opsList) {
		if (op == null) {
		    throw new NullPointerException("op "+cnt+" in OpArray given to "+classNameForErrors+" is null");
		}
	    }
	}
    }

    /**
     * A `such that clause' specifies which invocations
     * are eligible to be serviced.
     * (Aka `synchronization expression'.)
     * Only invocations with a `such that clause' that evaluates to true
     * are serviced by this arm.
     */
    abstract public static class SuchThat implements Quantifiable {
	/**
	 * Expression applied to an invocation to determine
	 * whether it is eligible to be serviced.
	 * @param  inv  The invocation.
	 * @return  True if the invocation is eligible; false otherwise.
	 */
	abstract public boolean expr(Invocation inv);

	/**
	 * Value of quantifiers, if any, are accessible via this field.
	 */
	public Quantifiers q;

	public void setQuantifiers(Quantifiers q) {
	    this.q = q;
	}

	
    }

    /**
     * A `by clause' specifies the order in which invocations are serviced.
     * (Aka `scheduling expression'.)
     * Invocations will be serviced in order of the by clause,
     * smallest values first.
     */
    abstract public static class By implements Quantifiable {

	/**
	 * Expression applied to an invocation to determine
	 * the invocation's `rank'.
	 * @param  inv  The invocation.
	 * @return  The invocation's `rank'.
	 */
	abstract public Comparable expr(Invocation inv);

	/**
	 * Comparison method used to compare two invocations'
	 * ranks.
	 * The specific types of the two ranks are the same,
	 * i.e., the type returned by {@link InniArm.By#expr}.
	 * This method may be overridden, but generally it won't need to be.
	 * (But, see {@link Common.Largest}).
	 * @param  o1  First rank.
	 * @param  o2  Second rank.
	 * @return  The result of comparing the two ranks.
	 *          Its value is the usual value for {@code compareTo}
	 *          ({@link Comparable#compareTo}).
	 */
	public int compareTo(Comparable o1, Comparable o2) {
	    return o1.compareTo(o2);
	}

	/**
	 * Value of quantifiers, if any, are accessible via this field.
	 */
	public Quantifiers q;

	public void setQuantifiers(Quantifiers q) {
	    this.q = q;
	}
    }

    // returns invocation or null if none
    // assumes caller already has locked.

    ///////// if have quantifier, MUST use quantifier form of suchthat and by?
    ////// ????

    public Invocation selectInvocation() {

	boolean hasSt = (st != null);
	boolean hasBy = (by != null);
	// minimum invocation (according to by expression)
	int minIndex = -1; // any non-index (negative) value would work
	Invocation minInv = null; // hush javac
	Comparable minValue = null; // hush javac

	// set up iterator over op's invocations
	// (op might be remote, in which case iterator uses RMI)

	//// maybe do catch elsewhere?????????

	OpInniIterator iter = null;
	try {
	    iter = op.elements();
	} catch (java.rmi.RemoteException e) {
	    throw new rjRuntimeError("RJ internal error: InniArm op.elements()");
	}

	/////2012-10-10 Debug.println("===>InniArm post-op.elements()");
	while (iter.hasNext()) {
	    ///////////// maybe chnage later so don't need cast
	    Invocation inv = (Invocation) (iter.next());
	    /////2012-10-10 Debug.println("===>InniArm in loop inv="+inv);
	    Quantifiable.Helper.setQuantifiers(st, quants);
	    // no st => considered true.
	    if (st == null || st.expr(inv)) {
		if (!hasBy) {
		    // got invocation and no by, so
		    // no need to look at other invocations.
		    minIndex = iter.getCurInvocationIndex();
		    minInv = inv;
		    break;
		}
		else {
		    Quantifiable.Helper.setQuantifiers(by, quants);
		    Comparable value = by.expr(inv);
		    // have min if:
		    //    first invocation found.
		    //    not first, and comparison shows < current min.
		    if (   minIndex < 0
			|| by.compareTo(value, minValue) < 0) {
			minIndex = iter.getCurInvocationIndex();
			minInv = inv;
			minValue = value;
		    }
		}
	    }
	}
	if (minIndex >= 0) {
	    iter.remove(minIndex);
	}

	////////////////////// here?
	/////////////	JRInstmt0.releaseIter();
	/////////////// in Inni, not here........?
	///////////	JRInstmt0.unlock();

	iter.unregister(); // like JRInstmt0.releaseIter();

	// will be null if no selectable invocations.
	return minInv;
    }

}
