package edu.ucdavis.rj;

import java.util.*;
import java.io.Serializable;

/**
 * Basic invocation abstraction.
 */
public class Invocation
    implements Serializable,
	       Cloneable {

    /**
     * Required by Serializable.
     */
    static final long serialVersionUID = 0;

    /**
     * Holds the parameters of the invocation.
     */
    private Object [] params;

    /**
     * Holds the return value for the invocation.
     */
    private Object returnValue;

    /**
     * The timestamp for the invocation.
     * //////////////////////////// ADDDDDDDDDDDDDDD note
     * about "precise" fairness.
     */
    Timestamp timestamp;

    /**
     * Create an invocation with the given parameters.
     * @param  params The parameters of the invocation.
     */
    public Invocation(Object ... params) {
	this.params = params;
	this.timestamp = new Timestamp(rjvm.getTimestamp());
	////// Debug.println("Invocation timestamp="+this.timestamp+" "+rjvm.thisVM.getHost());
    }

    /**
     * Return the number of parameters in the invocation.
     * @return The number of parameters in the invocation.

     */
    public int paramsLength() {
	return params.length;
    }

    /**
     * Return a parameter of the invocation.
     * @param  index Index (zero-based) of the invocation parameter to return.
     * @return The index-th parameter.
     /////////// if out of range, throws IndexOutOfBoundsException
     */
    public Object getParam(int index) {
	return params[index];
    }

    /**
     * Set the return value for the invocation.
     * @param  value The return value.
     */
    public void setReturnValue(Object value) {
	returnValue = value;
    }

    /**
     * Get the return value for the invocation.
     * @return The return value.

     */
    public Object getReturnValue() {
	return returnValue;
    }

    /**
     * Holds the reply operation for the invocation.
     * NEEEEEEEEEEEEEEEEDDDDDDDDDDDD NOOOOOOOOOOOOOOOP.
     * for now, test if != null................
     */
    private OpProxy replyOp;

    /**
     * Set the reply operation for the invocation.
     * @param  replyOp The reply operation.
     //////////////////////////// 2010-08-31 public for testing
     //////////////// deferred reply stuff
     */
    public void setReplyOp(OpProxy replyOp) {
	this.replyOp = replyOp;
    }

    /**
     * Get the reply operation for the invocation.
     * @return The reply operation.
     //////////////////////////// 2010-08-31 public for testing
     //////////////// deferred reply stuff
     */
    public OpProxy getReplyOp() {
	return replyOp;
    }

    /**
     * Do the reply for this invocation.
     * Give the invoker the reply (i.e., the go-ahead and return values)
     * for a call invocation.
     * For send invocation (or invocation to which already replied,
     * do nothing).
     */
    void replyToInvoker() {
	internalReplyToInvoker(false);
    }

    /**
     * Does the work for
     * {@link Invocation#replyToInvoker and @Invocation#reply}
     * @param makeCopy Whether to send original or copy to invoker.
     * send original for call and send; send copy for reply.
     */
    private void internalReplyToInvoker(boolean makeCopy) {
	OpProxy back;
	// test and set on back need to be atomic
	// in case two threads are sharing an Invocation.
	synchronized (this) {
	    back = this.getReplyOp();
	    if (back != null) {
		// prevent a second reply
		// e.g., multiple inv.reply()
		// or this Invocation gets reused in another invocation.
		this.setReplyOp(null);
	    }
	}
	if (back != null) {
	    Invocation ret;
	    if (makeCopy) {
		// be sure to copy after setReplyOp(null).
		ret = this.safeClone();
	    }
	    else {
		ret = this;
	    }
	    try {
		// update the timestamp.
		// this update is to make co statement work.
		// the ret invocation might be being sent here
		// to a return op in a co statement.
		// need to make sure that co statement handles
		// those invocations in the correct order.
		ret.timestamp = new Timestamp(rjvm.getTimestamp());
		back.send(ret);
	    } catch (Exception e) {
		throw new rjRuntimeError("RJ internal error: Invocation.replyToInvoker");
	    }
	}
    }

    /**
     * Early reply for this invocation to the invoker.
     * Subsequent replies are ignored.
     * Make copy of invocation to send back to invoker
     * since current process will continue to use this invocation.
     */
    public void reply() {
	//// 2012-10-10 Debug.println("Invocation reply ..........  "+this);
	internalReplyToInvoker(true);
    }

    /**
     * ////////////////
     * Abbreviation for 
     *  {@code
        this.setReplyOp(value);
        this.reply();
	}
     *
     * @param value The return value.
     */
    public void reply(Object value) {
	this.setReturnValue(value);
	this.reply();
    }

    /**
     * Forward an invocation to the specified operation.
     * Most importantly, the ability to reply to the original invoker
     * is also forwarded.
     * The invocation forwarded is either a copy of this invocation,
     * if no invocation is explicitly specified as a parameter,
     * or it's the invocation specified.
     *
     * (Caution on forwarding the current invocation:
     * if code has extracted parameters from the invocation
     * into local variables
     * and modified those variables,
     * those changes don't appear in the invocation;
     * create and forward a new invocation from the local variables.
     *
     * In either case, after forwarding, the forwarder continues to
     * execute using the current invocation.
     * Subsequent forwards behave like a send.
     * A forward of an invocation that was sent also acts like a send.
     *
     * @param fwdOpProxy The operation to which to forward.
     * @param fwdInvocation The invocation to forward.
     */
    public void forward(OpProxy fwdOpProxy, Invocation fwdInvocation) {
	internalForward(fwdOpProxy, fwdInvocation);
    }

    /**
     * Invokes {@link #forward(OpProxy,Invocation)}
     * on {@code forward(new OpProxy(fwdOpImpl), fwdInvocation))}.
     * @param fwdOpImpl forward to that op.
     * @param fwdInvocation invocation to forward.
     */
    public void forward(OpImpl fwdOpImpl, Invocation fwdInvocation) {
	this.forward(new OpProxy(fwdOpImpl), fwdInvocation);
    }

    /**
     * Forwards this invocation.
     * Invokes {@link #forward(OpProxy,Invocation)}
     * on {@code forward(new OpProxy(fwdOpImpl), this))}.
     * @param fwdOpProxy forward to that op (proxy).
     */
    public void forward(OpProxy fwdOpProxy) {
	this.forward(fwdOpProxy, this);
    }

    /**
     * Forwards this invocation.
     * Invokes {@link #forward(OpProxy,Invocation)}
     * on {@code forward(new OpProxy(fwdOpImpl), this))}.
     * @param fwdOpImpl forward to that op.
     */
    public void forward(OpImpl fwdOpImpl) {
	this.forward(new OpProxy(fwdOpImpl));
    }

    /**
     * Does the work for
     * {@link Invocation#forward}
     * @param fwdOpProxy OpProxy to which to forward.
     * @param originalFwdInvocation Invocation to which to forward.
     */
    private void internalForward(OpProxy fwdOpProxy,
				 Invocation originalFwdInvocation) {
	////2012-10-10 Debug.println("Invocation forward ..........  "+this);
	OpProxy back = this.getReplyOp();
	//// 2012-10-10 Debug.println("             back  ..........  "+back);
	// prevent a second forward from being forwarded or reply-ed
	this.setReplyOp(null);
	// prevent user from doing something silly.
	// don't make copy until after setReplyOp
	// since originalFwdInvocation might == this.
	Invocation fwdInvocation = originalFwdInvocation.safeClone();
	// pass ability to reply to original invoker on to new invocation.
	fwdInvocation.setReplyOp(back);
	try {
	    fwdOpProxy.send(fwdInvocation);
	} catch (Exception e) {
	    throw new rjRuntimeError("RJ internal error: Invocation.internalForward");
	}
    }

    /**
     * Get the Timestamp for the invocation.
     * @return The Timestamp.

     */
    Timestamp getTimestamp() {
	return timestamp;
    }

    /**
     * Update the VM's timestamp with this invocation's timestamp,
     * which comes from a send or call
     * (which might have originated on a different VM).
     */
    void setRJVMTimestamp() {
	rjvm.setTimestamp(getTimestamp().getTimestamp());
    }

    /**
     * Clone this Invocation.
     */
    protected Invocation clone()
	throws CloneNotSupportedException {
	return (Invocation) super.clone();
    }

    /**
     * Clone this Invocation, without possibility of exceptions.
     * @return The cloned Invocation.
     */
    Invocation safeClone() {
	Invocation inv = null;
	try {
	    inv = clone();
	} catch (Exception e) {
	    throw new rjRuntimeError("RJ internal error: Invocation's safeClone");
	}
	return inv;
    }

}
