
#include "cow.h"
#include "cgi.h"
#include "logs.h"
#include "request.h"
#include "mime.h"
#include "http_message.h"

/*
 * The HTTP request handler
 */

static void fixup_server_root(void);
static void spawn_threads(void);

/* from hand.c */
void *handler(void *);
void *hand(void *);
void *depot(void *);
# include "pmsg.h"

/* for initialize static reply messages */
int response_init();

//# define COW_STACK_SIZE 64*1024        // 16 is too small
# define COW_STACK_SIZE 128*1024        // 16 is too small
//# define COW_STACK_SIZE 256*1024       // 16 is too small
//# define COW_STACK_SIZE 1024*1024      // 16 is too small
//# define COW_STACK_SIZE 2048*1024      // 16 is too small

pth_barrier_t init_bar;
pth_t tid_depot;
pth_t tid_main;
reqnode* new_req;

/*
 * And the server main procedure
 */

// COW_NOHAND = max # hands
#define COW_NOHAND_X COW_NOHAND*3

/* pool of pmsg's used to avoid malloc overhead */
reqnode* pmsgpool;
reqnode* pmsgpool_first = NULL;

int cow_nohands_running = 0;    // number of handler running
int cow_nohands = COW_NOHAND;   // max numbef or handler (see defines.h for val)

static int s_socket;
pth_attr_t attr;

/* these values are "number of hands running at the same time" */
unsigned long cow_max_conn = 0;  /* max simultaneous connection of session */
unsigned long cow_cur_conn = 0;  /* current simultaneous connection */

char* cfgfile = NULL;
char cwd[400];

/****
 * myexit()
 *
 * Signal handler used to catch SIGINT (i.e. Ctrl+C) or SIGTERM (i.e. kill)
 ****/
static void myexit(int sig) {
    if (sig != SIGINT && sig != SIGTERM)
        fprintf(ELOG, "myexit() called without SIGINT or SIGTERM!?!?\n");

    COW_CLOSE(s_socket);
    pth_attr_destroy(attr);
    pth_kill();
    fprintf(stderr, "\nCoW received kill signal -> BEEF\n");
    if (verbose) {
        printf("cow_max_conn = %lu\n", cow_max_conn);
        printf("cow_cur_conn = %lu\n", cow_cur_conn);
        printf("cow_noof = %d\n", cow_noof);
    }

    /*
     * Change back to our original directory if possible so that
     * profiling data gets written to the right place.  If this
     * fails, we don't really care.
     */
    chdir(cwd);

    exit(0);
}

int main(int argc, char *argv[]) {
    int stay_in_foreground = 0;
    struct sockaddr_in sar;
    struct protoent *pe;
    struct sockaddr_in peer_addr;
    socklen_t peer_len;
    int sr;
    int c;                      /* command line arg */
    FILE *mime_fp;
    pth_msgport_t mp_depot;     // port to the depot
    int i;
    int one = 1;

    /*
     * Takashi had hardcoded MAX_OPEN_FD into defines.h to 1024, but that
     * undoubtedly varies between systems.  I think this is a more portable
     * way of finding out what the OS' file descriptor limit is for a process.
     */
    unsigned long maxfd = sysconf(_SC_OPEN_MAX);
    struct rlimit mylimit;
    if (0 > getrlimit(RLIMIT_NOFILE, &mylimit)) {
        perror("getrlimit");
        /* not fatal */
    } else {
        if (mylimit.rlim_cur < maxfd)
            mylimit.rlim_cur = maxfd;
        mylimit.rlim_max = maxfd;
        if (0 > setrlimit(RLIMIT_NOFILE, &mylimit)) {
            perror("setrlimit");
            /* not fatal */
        }
    }

    /* Parse command line options */
    while (-1 != (c = getopt(argc, argv, "c:dr:p:v"))) {
        switch (c) {
        case 'c':
            cfgfile = optarg;
            break;
        case 'd':
            stay_in_foreground = 1;
            break;
        case 'r':
            server_root = strdup(optarg);
            break;
        case 'p':
            server_port = atoi(optarg);
            if (server_port <= 0 || server_port >= 65535) {
                fprintf(stderr, "Illegal port: %d\n", server_port);
                exit(1);
            }
            break;
        case 'v':
            verbose = 1;
            break;
        default:
            fprintf(stderr, "Usage: %s [-c <cfg file>] [-d] [-p <port>] "
                            "[-r <dir>]\n", argv[0]);
            exit(1);
        }
    }

    /* Background this process */
    if (!stay_in_foreground) {
        switch(fork()) {
            case -1:
                /* non-fatal failure */
                perror("Warning: could not daemon-ize CoW");
                break;
            case 0:
                /* Child; continue running CoW */
                break;
            default:
                /* Parent; exit */
                exit(0);
        }
    }

    /* Read cow.conf */
    read_config_files(cfgfile);

    /* Register mime types if we're not using a single, global mime type */
    if (!global_mime) {
        mime_fp = fopen("/etc/mime.types", "r");
        if (NULL == mime_fp) {
            fprintf(stderr, "FATAL: Unable to open /etc/mime.types\n");
            exit(-1);
        }
        mime_init(mime_fp);
        if (fclose(mime_fp)) {
            perror("fclose");
        }
    }

    /* setting up the server root */
    fixup_server_root();
    if (server_port == 0)
        server_port = DEFAULT_SERVER_PORT;

    printf("\n" PACKAGE_STRING "\n");
    printf("Send feedback and bugreports to " PACKAGE_BUGREPORT "\n");
    if (stay_in_foreground)
        printf("Hit CTRL-C to stop the server.\n\n");
    if (verbose) {
        printf("Using server port = %d\n", server_port);
        printf("Using server root = %s\n", server_root);
        printf("PthVersion 0x V RR T LL is %lX\n", pth_version());
    }

    /* initialize scheduler */
    if (!pth_init()) {
        fprintf(stderr, "pth_init failure -> exit\n");
        exit(-1);
    }
    signal(SIGPIPE, SIG_IGN);
    signal(SIGINT, myexit);
    signal(SIGTERM, myexit);

    /* If logging was compiled in, open the logs */
    open_logs();

    /* for now, this is a dummy function */
    create_common_env();

    /* create TCP socket */
    if ((pe = getprotobyname("tcp")) == NULL) {
        perror("getprotobyname");
        exit(1);
    }
    errno = 0;
    if ((s_socket = socket(AF_INET, SOCK_STREAM, pe->p_proto)) == -1) {
        perror("socket");
        exit(1);
    }

    /*
     * Reuse the server socket; this way we don't have to wait for a
     * couple minutes after shutdown to get rid of the "Address in Use"
     * messages.  Failure here is not fatal.
     */
    setsockopt(s_socket, SOL_SOCKET, SO_REUSEADDR,
               (void*)&one, sizeof(one));

    /*
     * Allocate a pool of pmsg structures; we'll keep re-using these in
     * order to avoid the overhead of malloc().  We technically only need
     * one message per handler thread, but we'll allocate a multiple of that
     * in order to let the depot queue up some requests.  POOL_SIZE_FACT
     * is set in defines.h.
     */
    pmsgpool = (reqnode*)malloc(POOL_SIZE_FACT*cow_nohands*sizeof(reqnode));
    if (pmsgpool == NULL) {
        perror("Failed to allocate request message pool");
        exit(1);
    }
    memset(pmsgpool, 0, POOL_SIZE_FACT * cow_nohands * sizeof(reqnode));
    for (i = 0; i < cow_nohands - 1; ++i)
        pmsgpool[i].next = &pmsgpool[i+1];
    pmsgpool[cow_nohands - 1].next = NULL;
    pmsgpool_first = pmsgpool;

    /* bind socket to port */
    sar.sin_family = AF_INET;
    sar.sin_addr.s_addr = INADDR_ANY;
    sar.sin_port = htons(server_port);
    if (bind(s_socket, (struct sockaddr *)&sar, sizeof(struct sockaddr_in))
        == -1) {
        perror("socket");
        exit(1);
    }

    /* start listening on the socket */
    if (listen(s_socket, SOMAXCONN) == -1) {
        perror("listen");
        exit(1);
    }

    /* initialize static reply message */
    if (response_init()) {
        perror("response_init");
        exit(1);
    }

#include "schedule.h"           // defines use POOL or not
#ifdef USE_POOL

    spawn_threads();

    /* Get a handle for the depot's message port */
    mp_depot = pth_msgport_find(DEPOT);
    if (mp_depot == NULL) {
        perror("Failed to get depot's message port");
        exit(1);
    }

    /* loop for requests */
    for (;;) {
        /* accept next connection */
        peer_len = sizeof(peer_addr);
        if (-1 == (sr = pth_accept(s_socket, (struct sockaddr *)&peer_addr, &peer_len))) {
            if (EMFILE == errno) {
                perror("EMFILE in accept");
                continue;
            } else {
                perror("accept");
                myexit(sr);
            }
        }
        ++cow_noof;
        ++cow_cur_conn;
        cow_max_conn = cow_max_conn < cow_cur_conn ?
            cow_cur_conn : cow_max_conn;

        /* Optional: Disable Nagle algorithm */
        if (disable_nagle) {
            int one = 1;
            setsockopt(sr, IPPROTO_TCP, TCP_NODELAY,
                       (void*)&one, sizeof(one));
        }

        /*
         * Stick the information about the incoming connection in a pmsg
         * structure and ship it to the depot so that it can be dispatched
         * to a a handler thread.  If the pmsg pool is empty, then it means
         * the depot has a backlog of requests.  We'll wait here (and not
         * accept any new connections) until some pmsg structures are freed
         * up.
         */
        while (pmsgpool_first == NULL)
            pth_yield(NULL);

        /* Just grab the first pmsg off the stack */
        new_req = pmsgpool_first;
        pmsgpool_first = pmsgpool_first->next;
        new_req->next = NULL;

        /* Fill in the pmsg fields and then send it */
        (new_req->m).fd  = sr;
        (new_req->m).act = POP;
        if (TRUE != pth_msgport_put(mp_depot, (pth_message_t*)new_req)) {
            fprintf(stderr, "Couldn't send request to depot (msgport_put)\n");
            exit(-2);
        }
    }

#else /* NON POOL VERSION */
#ifdef NON_POOL_SINGLE
    if (verbose)
        printf("using NON POOL && NO SPAWNING THREAD.\n");
    for (;;) {
        if ((sr =
             pth_accept(s_socket, (struct sockaddr *)&peer_addr,
                        &peer_len)) == -1) {
            perror("accept");
            continue;
        }

        {
            msg req;
            req.fdp = sr;
            ++cow_noof;
            read_request(&req);
            if (cow_noof > 100 && verbose)
                printf("--- cow_noof == %d ---\n", cow_noof);
        DEB_FD}
    }
#else
    if (verbose)
        printf("using NON POOL (i.e. spawn a new thread per request.\n");

    /* finally loop for requests */

    pth_attr_set(attr, PTH_ATTR_NAME, "handler");
    pth_attr_set(attr, PTH_ATTR_JOINABLE, FALSE);
    if (verbose)
        printf("listening on port %d (max %d simultaneous connections)\n",
                server_port, cow_nohands);

/* to increase|decrease the priority of the main thread
  {
    pth_attr_t this_attr = NULL;
    pth_t this_id = pth_self();
    int *prio;
    this_attr = pth_attr_of(this_id);
    // pth_attr_get(this_id, PTH_ATTR_PRIO, prio);
    // printf("prio = %d\n", *prio);
    pth_attr_set(this_id, PTH_ATTR_PRIO, 100);
  }
*/

    for (;;) {
        /* accept next connection */
        peer_len = sizeof(peer_addr);
        if ((sr =
             pth_accept(s_socket, (struct sockaddr *)&peer_addr,
                        &peer_len)) == -1) {
            if (EMFILE == errno) {
                // simply yield or sleep is no good.
                // so let the hands process some, then yield.
                while (pth_ctrl
                       (PTH_CTRL_GETTHREADS_READY
                        | PTH_CTRL_GETTHREADS_WAITING) > cow_nohands / 4)
                    pth_yield(NULL);
                // tune this number
                continue;
            }
            else {
                // really bad. (not simply out of fd).
                perror("accept");
                exit(errno);
            }
        }

/* debug :
    {
      int cthreads = pth_ctrl (PTH_CTRL_GETTHREADS);
      int cnew = pth_ctrl (PTH_CTRL_GETTHREADS_NEW);
      int cready = pth_ctrl (PTH_CTRL_GETTHREADS_READY);
      int cwaiting = pth_ctrl (PTH_CTRL_GETTHREADS_WAITING);
      int crunning = pth_ctrl (PTH_CTRL_GETTHREADS_RUNNING);    // always 1
      int csuspend = pth_ctrl (PTH_CTRL_GETTHREADS_SUSPENDED);  // should be 0
      int cdead = pth_ctrl (PTH_CTRL_GETTHREADS_DEAD);
      printf ("[ <new %d><ready %d><waiting %d><dead %d> ]\n",
              cnew, cready, cwaiting, cdead);
    }
 * end of debug */

        if (pth_ctrl
            (PTH_CTRL_GETTHREADS_READY
             | PTH_CTRL_GETTHREADS_NEW
             | PTH_CTRL_GETTHREADS_WAITING
             | PTH_CTRL_GETTHREADS_RUNNING
             | PTH_CTRL_GETTHREADS_SUSPENDED) >= cow_nohands) {
            fprintf(ERROR_LOG, "currently no more connections acceptable\n");
            COW_CLOSE(sr);
            pth_yield(NULL);
            continue;
        }
        fprintf(ERROR_LOG,
                "connection established (fd: %d, ip: %s, port: %d)\n",
                sr, inet_ntoa(peer_addr.sin_addr), ntohs(peer_addr.sin_port));

        /* spawn new handling thread for connection */
        {
            pth_t tid;
            tid = pth_spawn(attr, handler, (void *)((long)sr));
            if (0 > tid) {
                perror("pth_spawn (non-pool version in cow.c)");
            }
            pth_yield(NULL);
        }
    }
# endif /* NON_POOL_SINGLE */
# endif /* USE_POOL */

    pth_exit(0);
}

/*
 * Name: fixup_server_root
 *
 * Description: Makes sure the server root is valid.
 */
static void fixup_server_root() {
    char *dirbuf;
    int dirbuf_size;

    /* If no server root has been set yet, set it to the default. */
    if (server_root == NULL) {
        server_root = strdup(DEFAULT_SERVER_ROOT);
        printf("Using default server root '%s'\n", server_root);
    }

    /*
     * Store the cwd so that we can change back to it at the end.
     * The only reason that this is really needed is so that
     * gprof's gmon.out get's written to the current directory
     * rather than the server root directory.
     */
    getcwd(cwd, 400);
    if (-1 == chdir(server_root)) {
        fprintf(stderr, "Could not chdir to \"%s\": aborting \n", server_root);
        exit(1);
    }

    if ('/' == server_root[0])
        /* Absolute path; no problem. */
        return;

    /*
     * If server_root is a relative path, then we need to make it absolute
     * for CGI scripts.
     */

    // dirbuf_size = MAXNAMLEN;   // BSD
    dirbuf_size = NAME_MAX;     /* POSIX */

    if (NULL == (dirbuf = (char *)malloc(dirbuf_size))) {
        fprintf(stderr, "Could not allocate directory buffer.\n");
        exit(1);
    }

#ifndef HAVE_GETCWD
    fprintf(stderr, "Server root must be an absolute path.\n");
    exit (1);
#else
    if (NULL == getcwd(dirbuf, dirbuf_size)) {
        if (ERANGE == errno)
            perror("cow: getcwd() failed (unable to get dir) -> aborting.");
        else if (EACCES == errno)
            perror("cow: getcwd() failed (no read access) -> aborting.");
        else
            perror("cow: getcwd() failed (unknown error) -> aborting.");
        exit(1);
    }
#endif

    free(server_root);
    server_root = dirbuf;
}


/****
 * spawn_threads()
 *
 * Pre-spawns a pool of handler threads in addition to the depot thread.
 ****/
static void spawn_threads(void) {
    int i = 0;
    pth_attr_t attr;

    pth_barrier_init(&init_bar, (cow_nohands + 2));   // +2 for main and depot
    if (verbose)
        printf("using POOL of hands (thread) of size %d\n", cow_nohands);

    tid_main = pth_self();

    attr = pth_attr_new();
    pth_attr_set(attr, PTH_ATTR_NAME, "depot");
    pth_attr_set(attr, PTH_ATTR_JOINABLE, FALSE);
    pth_attr_set(attr, PTH_ATTR_STACK_SIZE, COW_STACK_SIZE);
    tid_depot = pth_spawn(attr, depot, (void *)cow_nohands);
    if (tid_depot == NULL) {
        perror("Failed to spawn depot");
        exit(1);
    }

    /* Spawn all the handler threads */
    pth_attr_set(attr, PTH_ATTR_NAME, "hand");
    pth_attr_set(attr, PTH_ATTR_PRIO, PTH_PRIO_MAX);
    for (i = 0; i < cow_nohands; i++) {
        if (0 > (int)pth_spawn(attr, hand, (void *)NULL)) {
            perror("Failed to spawn handler");
            exit(errno);
        }
        ++cow_nohands_running;
    }
    pth_attr_destroy(attr);

    /* Wait for all threads to start up */
    pth_barrier_reach(&init_bar);
}
