/*
 * buff.c
 *
 * The CoW I/O buffering library.  This code started started out as
 * a modified version of Apache's buff.c buffering library and has
 * evolved from there.  Chunks of this library are still Apache code.
 *
 *
 * ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

#include "cow.h"

#include "buff.h"

#include <errno.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#ifndef NO_WRITEV
#include <sys/types.h>
#include <sys/uio.h>
#endif

#ifdef HAVE_BSTRING_H
#include <bstring.h>            /* for IRIX, FD_SET calls bzero() */
#endif

#ifndef SENDBUFSIZE
#define SENDBUFSIZE (4096)
#endif
/* This must be enough to represent (DEFAULT_BUFSIZE - 3) in hex,
 * plus two extra characters.
 */
#ifndef CHUNK_HEADER_SIZE
#define CHUNK_HEADER_SIZE (5)
#endif

#define ascii_CRLF "\015\012" /* A CRLF which won't pass the conversion machinery */
#define CRLF "\r\n"
#define CR '\r'
#define LF '\n'

/* bwrite()s of greater than this size can result in a large_write() call,
 * which can result in a writev().  It's a little more work to set up the
 * writev() rather than copy bytes into the buffer, so we don't do it for small
 * writes.  This is especially important when chunking (which is a very likely
 * source of small writes if it's a module using ap_bputc/ap_bputs)...because we
 * have the expense of actually building two chunks for each writev().
 */
#ifndef LARGE_WRITE_THRESHOLD
#define LARGE_WRITE_THRESHOLD 31
#endif


/*
 * Buffered I/O routines.
 * These are a replacement for the stdio routines.
 * Advantages:
 *  Known semantics for handling of file-descriptors (on close etc.)
 *  No problems reading and writing simultanously to the same descriptor
 *  No limits on the number of open file handles.
 *  Only uses memory resources; no need to ensure the close routine
 *  is called.
 *  Extra code could be inserted between the buffered and un-buffered routines.
 *  Timeouts could be handled by using select or poll before read or write.
 *  Extra error handling could be introduced; e.g.
 *   keep an address to which we should longjump(), or
 *   keep a stack of routines to call on error.
 */

/* Notes:
 *  On reading EOF, EOF will set in the flags and no further Input will
 * be done.
 *
 * On an error except for EAGAIN, ERROR will be set in the flags and no
 * futher I/O will be done
 */

/* the lowest level reading primitive */
static int ap_read(BUFF *fb, void *buf, int nbyte, pth_event_t timeout)
{
    if (timeout != NULL) {
        int rv = pth_read_ev(fb->fd_in, buf, nbyte, timeout);
        if (pth_event_status(timeout) == PTH_STATUS_OCCURRED) {
            errno = EINTR;
            return -1;
        } else {
            return rv;
        }
    } else {
        return pth_read(fb->fd_in, buf, nbyte);
    }
}

/* the lowest level writing primitive */
static int ap_write(BUFF *fb, const void *buf, int nbyte)
{
    return pth_write(fb->fd, buf, nbyte);
}

/* For CoW, we don't need any of the special stuff in buff_write... */
#define buff_write ap_write


static void doerror(BUFF *fb, int direction)
{
    int errsave = errno;        /* Save errno to prevent overwriting it below */

    fb->flags |= (direction == B_RD ? B_RDERR : B_WRERR);
    if (fb->error != NULL)
        (*fb->error) (fb, direction, fb->error_data);

    errno = errsave;
}

/* Buffering routines */
/*
 * Create a new buffered stream
 */
BUFF* ap_bcreate(int flags) {
    BUFF *fb;

    fb = malloc(sizeof(BUFF));
    fb->bufsiz = SENDBUFSIZE;
    fb->flags = flags & B_RDWR;

    if (flags & B_RD)
        fb->inbase = malloc(fb->bufsiz);
    else
        fb->inbase = NULL;

    /* overallocate so that we can put a chunk trailer of CRLF into this
     * buffer */
    if (flags & B_WR)
        fb->outbase = malloc(fb->bufsiz + 2);
    else
        fb->outbase = NULL;

    fb->inptr = fb->inbase;

    fb->incnt = 0;
    fb->outcnt = 0;
    fb->outchunk = -1;
    fb->error = NULL;
    fb->bytes_sent = 0L;

    fb->fd = -1;
    fb->fd_in = -1;

    return fb;
}

/*
 * Push some I/O file descriptors onto the stream
 */
void ap_bpushfd(BUFF *fb, int fd_in, int fd_out)
{
    fb->fd = fd_out;
    fb->fd_in = fd_in;
}


int ap_bsetopt(BUFF *fb, int optname, const void *optval)
{
    if (optname == BO_BYTECT) {
        fb->bytes_sent = *(const long int *) optval - (long int) fb->outcnt;;
        return 0;
    }
    else {
        errno = EINVAL;
        return -1;
    }
}

int ap_bgetopt(BUFF *fb, int optname, void *optval)
{
    if (optname == BO_BYTECT) {
        long int bs = fb->bytes_sent + fb->outcnt;
        if (bs < 0L)
            bs = 0L;
        *(long int *) optval = bs;
        return 0;
    }
    else {
        errno = EINVAL;
        return -1;
    }
}

static int bflush_core(BUFF *fb);

/*
 * Start chunked encoding.
 *
 * Note that in order for ap_bputc() to be an efficient macro we have to guarantee
 * that start_chunk() has always been called on the buffer before we leave any
 * routine in this file.  Said another way, if a routine here uses end_chunk()
 * and writes something on the wire, then it has to call start_chunk() or set
 * an error condition before returning.
 */
static void start_chunk(BUFF *fb)
{
    if (fb->outchunk != -1) {
        /* already chunking */
        return;
    }
    if ((fb->flags & (B_WRERR | B_EOUT | B_WR)) != B_WR) {
        /* unbuffered writes */
        return;
    }

    /* we need at least the header_len + at least 1 data byte
     * remember that we've overallocated fb->outbase so that we can always
     * fit the two byte CRLF trailer
     */
    if (fb->bufsiz - fb->outcnt < CHUNK_HEADER_SIZE + 1) {
        bflush_core(fb);
    }
    fb->outchunk = fb->outcnt;
    fb->outcnt += CHUNK_HEADER_SIZE;
}


/*
 * end a chunk -- tweak the chunk_header from start_chunk, and add a trailer
 */
static void end_chunk(BUFF *fb)
{
    int i;
    unsigned char *strp;

    if (fb->outchunk == -1) {
        /* not chunking */
        return;
    }

    if (fb->outchunk + CHUNK_HEADER_SIZE == fb->outcnt) {
        /* nothing was written into this chunk, and we can't write a 0 size
         * chunk because that signifies EOF, so just erase it
         */
        fb->outcnt = fb->outchunk;
        fb->outchunk = -1;
        return;
    }

    /* we know this will fit because of how we wrote it in start_chunk() */
    i = snprintf((char *) &fb->outbase[fb->outchunk], CHUNK_HEADER_SIZE,
                "%x", fb->outcnt - fb->outchunk - CHUNK_HEADER_SIZE);

    /* we may have to tack some trailing spaces onto the number we just wrote
     * in case it was smaller than our estimated size.  We've also written
     * a \0 into the buffer with ap_snprintf so we might have to put a
     * \r back in.
     */
    strp = &fb->outbase[fb->outchunk + i];
    while (i < CHUNK_HEADER_SIZE - 2) {
        *strp++ = ' ';
        ++i;
    }
    *strp++ = CR;
    *strp = LF;

    /* tack on the trailing CRLF, we've reserved room for this */
    fb->outbase[fb->outcnt++] = CR;
    fb->outbase[fb->outcnt++] = LF;

    fb->outchunk = -1;
}


/*
 * Set a flag on (1) or off (0).
 */
int ap_bsetflag(BUFF *fb, int flag, int value)
{
    if (value) {
        fb->flags |= flag;
        if (flag & B_CHUNK) {
            start_chunk(fb);
        }
    }
    else {
        fb->flags &= ~flag;
        if (flag & B_CHUNK) {
            end_chunk(fb);
        }
    }
    return value;
}


int ap_bnonblock(BUFF *fb, int direction)
{
    int fd;

    fd = (direction == B_RD) ? fb->fd_in : fb->fd;
#if defined(O_NONBLOCK)
    return fcntl(fd, F_SETFL, O_NONBLOCK);
#elif defined(O_NDELAY)
    return fcntl(fd, F_SETFL, O_NDELAY);
#elif defined(FNDELAY)
    return fcntl(fd, F_SETFL, FNDELAY);
#else
    /* XXXX: this breaks things, but an alternative isn't obvious...*/
    return 0;
#endif
}

int ap_bfileno(BUFF *fb, int direction)
{
    return (direction == B_RD) ? fb->fd_in : fb->fd;
}

/*
 * This is called instead of read() everywhere in here.  It implements
 * the B_SAFEREAD functionality -- which is to force a flush() if a read()
 * would block.  It also deals with the EINTR errno result from read().
 * return code is like read() except EINTR is eliminated.
 */


#define saferead saferead_guts


/* Test the descriptor and flush the output buffer if it looks like
 * we will block on the next read.
 *
 * Note we assume the caller has ensured that fb->fd_in <= FD_SETSIZE
 */
void ap_bhalfduplex(BUFF *fb)
{
    int rv;
    fd_set fds;
    struct timeval tv;

    /* We don't need to do anything if the connection has been closed
     * or there is something readable in the incoming buffer
     * or there is nothing flushable in the output buffer.
     */
    if (fb == NULL || fb->fd_in < 0 || fb->incnt > 0 || fb->outcnt == 0) {
        return;
    }
    /* test for a block */
    do {
        FD_ZERO(&fds);
        FD_SET(fb->fd_in, &fds);
        tv.tv_sec = 0;
        tv.tv_usec = 0;
        rv = select(fb->fd_in + 1, &fds, NULL, NULL, &tv);
    } while (rv < 0 && errno == EINTR && !(fb->flags & B_EOUT));

    /* treat any error as if it would block as well */
    if (rv != 1) {
        ap_bflush(fb);
    }
}

static int saferead_guts(BUFF *fb, void *buf, int nbyte, pth_event_t timeout)
{
    if (fb->flags & B_SAFEREAD) {
        ap_bhalfduplex(fb);
    }
    return ap_read(fb, buf, nbyte, timeout);
}


/* A wrapper around saferead which does error checking and EOF checking
 * yeah, it's confusing, this calls saferead, which calls ap_read...
 */
static int read_with_errors(BUFF *fb, void *buf, int nbyte, pth_event_t timeout)
{
    int rv;

    rv = saferead(fb, buf, nbyte, timeout);
    if (rv == 0) {
        fb->flags |= B_EOF;
    }
    else if (rv == -1 && errno != EAGAIN) {
        doerror(fb, B_RD);
    }
    return rv;
}

/****
 * Reads a complete line from the socket into the provided buffer.  A
 * line is defined as a sequence of characters ending with a \n.  The \n
 * and, if present, the \r before it are stripped before being copied to
 * the buffer.
 *
 * @param in    The socket to read from
 * @param buf   The buffer to read into
 * @param nbyte Size of buffer; TODO: either remove or check this
 * @param timeout Give up after waiting this long
 *
 * @return The length of the line copied to the buffer.  If we timeout
 *      before reading an entire line then -1 * (chars read + 1) is returned
 *      and errno is set to EINTR.
 ****/
int cow_read_line(BUFF* in, char* buf, int nbyte, pth_event_t timeout) {
    int i, nrd;
    char* eol;
    
    /* If there's a socket error, just return */
    if (in->flags & B_RDERR)
        return -1;

    /* Is there a complete line in the buffer? */
    nrd = in->incnt;
    if (nrd > 0) {
        eol = strchr(in->inptr, '\n');
        if (eol != NULL) {
            int len = (eol - (char*)in->inptr);
            memcpy(buf, in->inptr, len);
            in->incnt = nrd - (len+1);
            in->inptr += (len+1);

            /* Null terminate the user buffer */
            if (len > 0 && buf[len - 1] == '\r') {
                buf[len - 1] = '\0';
                return len - 1;
            } else {
                buf[len] = '\0';
                return len;
            }
        }
    }

    /*
     * Need to read some more from the socket.  Start by copying what we've
     * already got.
     */
    if (nrd > 0) {
        memcpy(buf, in->inptr, nrd);
        buf = nrd + (char *) buf;
    }
    in->incnt = 0;
    in->inptr = in->inbase;
    in->inptr[0] = '\0';
    if (in->flags & B_EOF)
        return nrd;

    /*
     * Read as much as we can directly into the user buffer.  If we read more
     * than one line in, copy the excess back into the hold buffer.
     */
    i = read_with_errors(in, buf, nbyte, timeout);
    if (i <= 0 || errno == EINTR)
        return -1;
    eol = strchr(buf, '\n');
    if (eol != NULL) {
        int len;

        /* Copy excess into hold buffer */
        if (eol != buf + i - 1) {
            in->incnt = i - (eol - buf) - 1;
            memcpy(in->inbase, eol+1, in->incnt);
            in->inptr = in->inbase;
        } else {
            in->incnt = 0;
            in->inptr = in->inbase;
        }
        
        /* Add null-terminator to user buffer */
        len = eol - buf;
        if (len > 0 && buf[len-1] == '\r') {
            buf[len-1] = '\0';
            return len - 1;
        } else {
            buf[len] = '\0';
            return len;
        }
    }

    /*
     * We read some stuff, but we still haven't seen that newline...
     * Call ourselves recursively to get the rest.
     */
    return cow_read_line(in, buf+i, nbyte-i, timeout);
}

    

/*
 * Read up to nbyte bytes into buf.
 * If fewer than byte bytes are currently available, then return those.
 * Returns 0 for EOF, -1 for error.
 */
int ap_bread(BUFF *fb, void *buf, int nbyte, pth_event_t timeout)
{
    int i, nrd;

    if (fb->flags & B_RDERR)
        return -1;
    if (nbyte == 0)
        return 0;

    if (!(fb->flags & B_RD)) {
        /* Unbuffered reading.  First check if there was something in the
         * buffer from before we went unbuffered. */
        if (fb->incnt) {
            i = (fb->incnt > nbyte) ? nbyte : fb->incnt;
            memcpy(buf, fb->inptr, i);
            fb->incnt -= i;
            fb->inptr += i;
            return i;
        }
        i = read_with_errors(fb, buf, nbyte, timeout);
        return i;
    }

    nrd = fb->incnt;
/* can we fill the buffer */
    if (nrd >= nbyte) {
        memcpy(buf, fb->inptr, nbyte);
        fb->incnt = nrd - nbyte;
        fb->inptr += nbyte;
        return nbyte;
    }

    if (nrd > 0) {
        memcpy(buf, fb->inptr, nrd);
        nbyte -= nrd;
        buf = nrd + (char *) buf;
        fb->incnt = 0;
    }
    if (fb->flags & B_EOF)
        return nrd;

/* do a single read */
    if (nbyte >= fb->bufsiz) {
/* read directly into caller's buffer */
        i = read_with_errors(fb, buf, nbyte, timeout);
        if (i == -1) {
            return nrd ? nrd : -1;
        }
    }
    else {
/* read into hold buffer, then memcpy */
        fb->inptr = fb->inbase;
        i = read_with_errors(fb, fb->inptr, fb->bufsiz, timeout);
        if (i == -1) {
            return nrd ? nrd : -1;
        }
        fb->incnt = i;
        if (i > nbyte)
            i = nbyte;
        memcpy(buf, fb->inptr, i);
        fb->incnt -= i;
        fb->inptr += i;
    }
    return nrd + i;
}


/*
 * Reads from the stream into the array pointed to by buff, until
 * a (CR)LF sequence is read, or end-of-file condition is encountered
 * or until n-1 bytes have been stored in buff. If a CRLF sequence is
 * read, it is replaced by a newline character.  The string is then
 * terminated with a null character.
 *
 * Returns the number of bytes stored in buff, or zero on end of
 * transmission, or -1 on an error.
 *
 * Notes:
 *  If null characters are expected in the data stream, then
 * buff should not be treated as a null terminated C string; instead
 * the returned count should be used to determine the length of the
 * string.
 *  CR characters in the byte stream not immediately followed by a LF
 * will be preserved.
 */
int ap_bgets(char *buff, int n, BUFF *fb)
{
    int i, ch, ct;

/* Can't do bgets on an unbuffered stream */
    if (!(fb->flags & B_RD)) {
        errno = EINVAL;
        return -1;
    }
    if (fb->flags & B_RDERR)
        return -1;

    ct = 0;
    i = 0;
    for (;;) {
        if (i == fb->incnt) {
/* no characters left */
            fb->inptr = fb->inbase;
            fb->incnt = 0;
            if (fb->flags & B_EOF)
                break;
            i = read_with_errors(fb, fb->inptr, fb->bufsiz, NULL);
            if (i == -1) {
                buff[ct] = '\0';
                return ct ? ct : -1;
            }
            fb->incnt = i;
            if (i == 0)
                break;          /* EOF */
            i = 0;
            continue;           /* restart with the new data */
        }

        ch = fb->inptr[i++];
        if (ch == LF) {  /* got LF */
            if (ct == 0)
                buff[ct++] = '\n';
/* if just preceeded by CR, replace CR with LF */
            else if (buff[ct - 1] == CR)
                buff[ct - 1] = '\n';
            else if (ct < n - 1)
                buff[ct++] = '\n';
            else
                i--;            /* no room for LF */
            break;
        }
        if (ct == n - 1) {
            i--;                /* push back ch */
            break;
        }

        buff[ct++] = ch;
    }
    fb->incnt -= i;
    fb->inptr += i;

    buff[ct] = '\0';
    return ct;
}

/*
 * Looks at the stream fb and places the first character into buff
 * without removing it from the stream buffer.
 *
 * Returns 1 on success, zero on end of transmission, or -1 on an error.
 *
 */
int ap_blookc(char *buff, BUFF *fb)
{
    int i;

    *buff = '\0';

    if (!(fb->flags & B_RD)) {  /* Can't do blookc on an unbuffered stream */
        errno = EINVAL;
        return -1;
    }
    if (fb->flags & B_RDERR)
        return -1;

    if (fb->incnt == 0) {       /* no characters left in stream buffer */
        fb->inptr = fb->inbase;
        if (fb->flags & B_EOF)
            return 0;

        i = read_with_errors(fb, fb->inptr, fb->bufsiz, NULL);
        if (i <= 0) {
            return i;
        }
        fb->incnt = i;
    }

    *buff = fb->inptr[0];
    return 1;
}

/*
 * Skip data until a linefeed character is read
 * Returns 1 on success, 0 if no LF found, or -1 on error
 */
int ap_bskiplf(BUFF *fb)
{
    unsigned char *x;
    int i;

/* Can't do bskiplf on an unbuffered stream */
    if (!(fb->flags & B_RD)) {
        errno = EINVAL;
        return -1;
    }
    if (fb->flags & B_RDERR)
        return -1;

    for (;;) {
        x = (unsigned char *) memchr(fb->inptr, '\012', fb->incnt);
        if (x != NULL) {
            x++;
            fb->incnt -= x - fb->inptr;
            fb->inptr = x;
            return 1;
        }

        fb->inptr = fb->inbase;
        fb->incnt = 0;
        if (fb->flags & B_EOF)
            return 0;
        i = read_with_errors(fb, fb->inptr, fb->bufsiz, NULL);
        if (i <= 0)
            return i;
        fb->incnt = i;
    }
}

/*
 * When doing chunked encodings we really have to write everything in the
 * chunk before proceeding onto anything else.  This routine either writes
 * nbytes and returns 0 or returns -1 indicating a failure.
 *
 * This is *seriously broken* if used on a non-blocking fd.  It will poll.
 *
 * Deals with calling doerror and setting bytes_sent.
 */
static int write_it_all(BUFF *fb, const void *buf, int nbyte)
{
    int i;

    if (fb->flags & (B_WRERR | B_EOUT))
        return -1;

    while (nbyte > 0) {
        i = buff_write(fb, buf, nbyte);
        if (i < 0) {
            if (errno != EAGAIN && errno != EINTR) {
                doerror(fb, B_WR);
                return -1;
            }
        }
        else {
            nbyte -= i;
            buf = i + (const char *) buf;
            fb->bytes_sent += i;
        }
        if (fb->flags & B_EOUT)
            return -1;
    }
    return 0;
}


#ifndef NO_WRITEV
/****
 * writev_all()
 *
 * Although building the iovec structures for writev() is simple, figuring
 * out what to do when only part of the data was written is a real pain.
 * This function will continue writev'ing until everything has been written
 * or until an error occurs.  This code is almost directly out of Apache's
 * source code.
 *
 * Similar to previous, but uses writev.  Note that it modifies vec.
 * return 0 if successful, -1 otherwise.
 *
 * Deals with doerror() and bytes_sent.
 *
 * @param fd   File descriptor to write to
 * @param vec  iovec pointing to set of buffers to send
 * @param nvec Number of buffers in iovec set
 *
 * @return 0 on success, -1 if writev fails with a fatal error (not EINTR)
 ****/
static int writev_it_all(BUFF *fb, struct iovec *vec, int nvec)
{
    int i, rv;
    
    /* while it's nice an easy to build the vector and crud, it's painful
     * to deal with a partial writev()
     */
    i = 0;
    while (i < nvec) {
        do
            rv = writev(fb->fd, &vec[i], nvec - i);
        while (rv == -1 && (errno == EINTR || errno == EAGAIN)
               && !(fb->flags & B_EOUT));
        if (rv == -1) {
            if (errno != EINTR && errno != EAGAIN) {
                doerror(fb, B_WR);
            }
            return -1;
        }
        fb->bytes_sent += rv;
        /* recalculate vec to deal with partial writes */
        while (rv > 0) {
            if (rv < vec[i].iov_len) {
                vec[i].iov_base = (char *) vec[i].iov_base + rv;
                vec[i].iov_len -= rv;
                rv = 0;
            }
            else {
                rv -= vec[i].iov_len;
                ++i;
            }
        }
        if (fb->flags & B_EOUT)
            return -1;
    }
    /* if we got here, we wrote it all */
    return 0;
}
#endif

/* A wrapper for buff_write which deals with error conditions and
 * bytes_sent.  Also handles non-blocking writes.
 */
static int write_with_errors(BUFF *fb, const void *buf, int nbyte)
{
    int rv;

    do
        rv = buff_write(fb, buf, nbyte);
    while (rv == -1 && errno == EINTR && !(fb->flags & B_EOUT));
    if (rv == -1) {
        if (errno != EAGAIN) {
            doerror(fb, B_WR);
        }
        return -1;
    }
    else if (rv == 0) {
        errno = EAGAIN;
        return -1;
    }
    fb->bytes_sent += rv;
    return rv;
}


/*
 * A hook to write() that deals with chunking. This is really a protocol-
 * level issue, but we deal with it here because it's simpler; this is
 * an interim solution pending a complete rewrite of all this stuff in
 * 2.0, using something like sfio stacked disciplines or BSD's funopen().
 *
 * Can be used on non-blocking descriptors, but only if they're not chunked.
 * Deals with doerror() and bytes_sent.
 */
static int bcwrite(BUFF *fb, const void *buf, int nbyte)
{
    char chunksize[16];         /* Big enough for practically anything */
#ifndef NO_WRITEV
    struct iovec vec[3];
#endif

    if (fb->flags & (B_WRERR | B_EOUT))
        return -1;

    if (!(fb->flags & B_CHUNK)) {
        return write_with_errors(fb, buf, nbyte);
    }

#ifdef NO_WRITEV
    /* without writev() this has poor performance, too bad */

    snprintf(chunksize, sizeof(chunksize), "%x" CRLF, nbyte);
    if (write_it_all(fb, chunksize, strlen(chunksize)) == -1)
        return -1;
    if (write_it_all(fb, buf, nbyte) == -1)
        return -1;
    if (write_it_all(fb, ascii_CRLF, 2) == -1)
        return -1;
    return nbyte;
#else
    vec[0].iov_base = chunksize;
    vec[0].iov_len = snprintf(chunksize, sizeof(chunksize), "%x" CRLF,
                              nbyte);
    vec[1].iov_base = (void *) buf;     /* cast is to avoid const warning */
    vec[1].iov_len = nbyte;
    vec[2].iov_base = ascii_CRLF;
    vec[2].iov_len = 2;

    return writev_it_all(fb, vec, (sizeof(vec) / sizeof(vec[0]))) ? -1 : nbyte;
#endif
}


#ifndef NO_WRITEV
/*
 * Used to combine the contents of the fb buffer, and a large buffer
 * passed in.
 */
static int large_write(BUFF *fb, const void *buf, int nbyte)
{
    struct iovec vec[4];
    int nvec;
    char chunksize[16];

    /* it's easiest to end the current chunk */
    if (fb->flags & B_CHUNK) {
        end_chunk(fb);
    }
    nvec = 0;
    if (fb->outcnt > 0) {
        vec[nvec].iov_base = (void *) fb->outbase;
        vec[nvec].iov_len = fb->outcnt;
        ++nvec;
    }
    if (fb->flags & B_CHUNK) {
        vec[nvec].iov_base = chunksize;
        vec[nvec].iov_len = snprintf(chunksize, sizeof(chunksize),
                                     "%x" CRLF, nbyte);
        ++nvec;
        vec[nvec].iov_base = (void *) buf;
        vec[nvec].iov_len = nbyte;
        ++nvec;
        vec[nvec].iov_base = ascii_CRLF;
        vec[nvec].iov_len = 2;
        ++nvec;
    }
    else {
        vec[nvec].iov_base = (void *) buf;
        vec[nvec].iov_len = nbyte;
        ++nvec;
    }

    fb->outcnt = 0;
    if (writev_it_all(fb, vec, nvec)) {
        return -1;
    }
    else if (fb->flags & B_CHUNK) {
        start_chunk(fb);
    }
    return nbyte;
}
#endif


/*
 * Write nbyte bytes.
 * Only returns fewer than nbyte if an error ocurred.
 * Returns -1 if no bytes were written before the error ocurred.
 * It is worth noting that if an error occurs, the buffer is in an unknown
 * state.
 */
int cow_write(SOCKTYPE out, const void *buf, int nbyte)
{
#ifdef USE_TCPCORK
    /*
     * pth_write will always return nbyte bytes unless an error occurs.  
     * pth_write returns -1 and EINTR only if the timeout expires so we
     * shouldn't loop on EINTR like normal code would.
     */;
    int rv;
    pth_event_t ev = pth_timeout(COW_WTO_S, COW_WTO_U);
    rv = pth_write_ev(out, buf, nbyte, ev);

    if (rv < nbyte && error == EINTR) {
        printf("DEBUG: Writing timed out!\n");
    } else if (rv < nbyte) {
        printf("DEBUG: Error writing!\n");
    }
    
    pth_event_free(ev, PTH_FREE_THIS);
    return rv;
#else /* if not USE_TCPCORK */
    int i, nwr, useable_bufsiz;

    if (out->flags & (B_WRERR | B_EOUT))
        return -1;
    if (nbyte == 0)
        return 0;

    if (!(out->flags & B_WR)) {
/* unbuffered write -- have to use bcwrite since we aren't taking care
 * of chunking any other way */
        return bcwrite(out, buf, nbyte);
    }

#ifndef NO_WRITEV
/*
 * Detect case where we're asked to write a large buffer, and combine our
 * current buffer with it in a single writev().  Note we don't consider
 * the case nbyte == 1 because modules which use rputc() loops will cause
 * us to use writev() too frequently.  In those cases we really should just
 * start a new buffer.
 */
    if (out->outcnt > 0 && nbyte > LARGE_WRITE_THRESHOLD
        && nbyte + out->outcnt >= out->bufsiz) {
        return large_write(out, buf, nbyte);
    }
#endif

/*
 * Whilst there is data in the buffer, keep on adding to it and writing it
 * out
 */
    nwr = 0;
    while (out->outcnt > 0) {
/* can we accept some data? */
        i = out->bufsiz - out->outcnt;
        if (i > 0) {
            if (i > nbyte)
                i = nbyte;
            memcpy(out->outbase + out->outcnt, buf, i);
            out->outcnt += i;
            nbyte -= i;
            buf = i + (const char *) buf;
            nwr += i;
            if (nbyte == 0)
                return nwr;     /* return if none left */
        }

/* the buffer must be full */
        if (out->flags & B_CHUNK) {
            end_chunk(out);
            /* it is just too painful to try to re-cram the buffer while
             * chunking
             */
            if (write_it_all(out, out->outbase, out->outcnt) == -1) {
                /* we cannot continue after a chunked error */
                return -1;
            }
            out->outcnt = 0;
            break;
        }
        i = write_with_errors(out, out->outbase, out->outcnt);
        if (i <= 0) {
            return nwr ? nwr : -1;
        }

        /* deal with a partial write */
        if (i < out->outcnt) {
            int j, n = out->outcnt;
            unsigned char *x = out->outbase;
            for (j = i; j < n; j++)
                x[j - i] = x[j];
            out->outcnt -= i;
        }
        else
            out->outcnt = 0;

        if (out->flags & B_EOUT)
            return -1;
    }
/* we have emptied the file buffer. Now try to write the data from the
 * original buffer until there is less than bufsiz left.  Note that we
 * use bcwrite() to do this for us, it will do the chunking so that
 * we don't have to dink around building a chunk in our own buffer.
 *
 * Note also that bcwrite never does a partial write if we're chunking,
 * so we're guaranteed to either end in an error state, or make it
 * out of this loop and call start_chunk() below.
 *
 * Remember we may not be able to use the entire buffer if we're
 * chunking.
 */
    useable_bufsiz = out->bufsiz;
    if (out->flags & B_CHUNK) useable_bufsiz -= CHUNK_HEADER_SIZE;
    while (nbyte >= useable_bufsiz) {
        i = bcwrite(out, buf, nbyte);
        if (i <= 0) {
            return nwr ? nwr : -1;
        }

        buf = i + (const char *) buf;
        nwr += i;
        nbyte -= i;

        if (out->flags & B_EOUT)
            return -1;
    }
/* copy what's left to the file buffer */
    out->outcnt = 0;
    if (out->flags & B_CHUNK)
        start_chunk(out);
    if (nbyte > 0)
        memcpy(out->outbase + out->outcnt, buf, nbyte);
    out->outcnt += nbyte;
    nwr += nbyte;
    return nwr;
#endif /* if not USE_TCPCORK */
}


static int bflush_core(BUFF *fb)
{
    int i;

    while (fb->outcnt > 0) {
        i = write_with_errors(fb, fb->outbase, fb->outcnt);
        if (i <= 0)
            return -1;

        /*
         * We should have written all the data, but if the fd was in a
         * strange (non-blocking) mode, then we might not have done so.
         */
        if (i < fb->outcnt) {
            int j, n = fb->outcnt;
            unsigned char *x = fb->outbase;
            for (j = i; j < n; j++)
                x[j - i] = x[j];
        }
        fb->outcnt -= i;

        /* If a soft timeout occurs while flushing, the handler should
         * have set the buffer flag B_EOUT.
         */
        if (fb->flags & B_EOUT)
            return -1;
    }

    return 0;
}

/*
 * Flushes the buffered stream.
 * Returns 0 on success or -1 on error
 */
int ap_bflush(BUFF *fb)
{
    int ret;

    if ((fb->flags & (B_WRERR | B_EOUT | B_WR)) != B_WR)
        return -1;

    if (fb->flags & B_CHUNK)
        end_chunk(fb);

    ret = bflush_core(fb);

    if (ret == 0 && (fb->flags & B_CHUNK)) {
        start_chunk(fb);
    }

    return ret;
}

/*
 * Flushes and closes the file, even if an error occurred.
 * Discards an data that was not read, or not written by bflush()
 * Sets the EOF flag to indicate no futher data can be read,
 * and the EOUT flag to indicate no further data can be written.
 */
int ap_bclose(BUFF *fb)
{
    int rc1 = 0;
    int rc2 = 0;
    int rc3 = 0;

    /* Flush any unsent output */
    if (fb->flags & B_WR)
        rc1 = ap_bflush(fb);
    rc2 = close(fb->fd);
    if (fb->fd_in != fb->fd)
        rc3 = close(fb->fd_in);

    /* Clean up the output/input socket buffers */
    if (fb->inbase != NULL)  free(fb->inbase);
    if (fb->outbase != NULL) free(fb->outbase);
    free(fb);

    return rc1 || rc2 || rc3;
}

void ap_bonerror(BUFF *fb, void (*error) (BUFF *, int, void *),
                          void *data)
{
    fb->error = error;
    fb->error_data = data;
}

