%{
#include "cow.h"
#include "cow_config.h"
#include "defines.h"

#undef realloc

#define VARMATCH(varname) (0 == strncmp(yytext, varname, strlen(varname)))

    enum svar { none, sroot, port, hands, verb, maxcon, maxdur, alog,
                elog, allmime, defmime, disnagle, indexfile, unknown };

    int yyerror(char*);
    void set_var(enum svar, char*);

    /* Variables that can actually be set by config file */
    char *server_root = NULL;
    char *access_log = NULL;
    char *error_log = NULL;
    char *global_mime = NULL;
    char *default_mime = NULL;
    int server_port = 0;
    int ka_max_con;
    int ka_max_dur;
    int max_handler;
    int verbose = 0;
    int disable_nagle = 0;
    idxfilename* index_files = NULL;

    /* TODO: Why are these here?  They should be elsewhere... */
    int cow_rto_s;              /* cow read/recv timeout in sec */
    int cow_rto_u;              /* cow read/recv timeout in usec */
    int cow_noof = 0;           /* number of open files (from accept to close) */

    /* Variables used for parsing config file */
    static enum svar curvar = none;
    static char *filename;
    static char *p, value_buffer[256];
    static int value_len = 0;
    static int curline = 1;
    static idxfilename* last_idx = NULL;
%}

%x STRVALUE

%%

[ \t]+          ;
#.*             ;
\n              ++curline;

[A-Za-z]+       {
                    /* This should be an identifier.  Figure out which one */
                    if      (VARMATCH("ServerRoot"))        curvar = sroot;
                    else if (VARMATCH("ServerPort"))        curvar = port;
                    else if (VARMATCH("NumHands"))          curvar = hands;
                    else if (VARMATCH("Verbose"))           curvar = verb;
                    else if (VARMATCH("KeepAliveMaxConn"))  curvar = maxcon;
                    else if (VARMATCH("KeepAliveMaxDur"))   curvar = maxdur;
                    else if (VARMATCH("AccessLog"))         curvar = alog;
                    else if (VARMATCH("ErrorLog"))          curvar = elog;
                    else if (VARMATCH("GlobalMime"))        curvar = allmime;
                    else if (VARMATCH("DefaultMime"))       curvar = defmime;
                    else if (VARMATCH("DisableNagle"))      curvar = disnagle;
                    else if (VARMATCH("IndexFile"))         curvar = indexfile;
                    else { curvar = unknown; yyerror("Unrecognized variable"); }
                }

[0-9]+          {
                    /* A numeric variable value. */
                    set_var(curvar, yytext);
                }

\"              {
                    /*
                     * Start of string value; clear our buffer and prepare to
                     * copy the value into the buffer.
                     */
                    BEGIN(STRVALUE);
                    value_len = 0;
                    p = value_buffer;
                }

<STRVALUE>{
\"              {
                    /* Found end of string value */
                    BEGIN(INITIAL);
                }

\n              {
                    /* EOL in middle of string?  I don't think so. */
                    yyerror("Found linebreak midstring");
                }


[^\n\"]+        {
                    set_var(curvar, yytext);
                }
}


%%

/*
 * Name: read_config_files
 *
 * Description: Reads config files via yyparse, then makes sure that
 * all required variables were set properly.
 */
void read_config_files(char* cfgfilename) {
    struct stat confstat;

    /* Set default values before parsing config file */
    ka_max_con = KA_MAX_CON;
    ka_max_dur = KA_MAX_DUR;
    max_handler = 100;

    /* for now, use default val from defines.h */
    cow_rto_s = COW_RTO_S;
    cow_rto_u = COW_RTO_U;


    /*
     * If a config filename was specified on the command line, open
     * that file.  Otherwise default to "cow.conf" in the current
     * directory.
     */
    if (cfgfilename != NULL) {
        yyin = fopen(cfgfilename, "r");
        filename = cfgfilename;
    } else if (0 == stat("/etc/cow.conf", &confstat)) {
        yyin = fopen("/etc/cow.conf", "r");
        filename = "/etc/cow.conf";
    } else if (0 == stat("/usr/local/etc/cow.conf", &confstat)) {
        yyin = fopen("/usr/local/etc/cow.conf", "r");
        filename = "/usr/local/etc/cow.conf";
    } else {
        yyin = fopen("cow.conf", "r");
        filename = "./cow.conf";
    }

    if (yyin == NULL) {
        fprintf(stderr, "Warning: couldn't open config file '%s'; "
                        "using defaults\n", filename);
        return;
    }

    /* Start reading the config file */
    yylex();
}


void set_var(enum svar var, char* val) {
    idxfilename* idx;

    switch(var) {
        case none:
            yyerror("Bare value encountered");
            break;
        case unknown:
            /* Ignore since we already warned about strange variable */
            break;
        case sroot:
            /* Only set server root if it wasn't overridden on the cmd line */
            if (server_root == NULL)
                server_root = strdup(val);
            else
                fprintf(stderr, "Overriding config file's ServerRoot\n");
            break;
        case port:
            /* Only set server port if it wasn't overridden on cmd line */
            if (server_port == 0)
                server_port = atoi(val);
            else
                fprintf(stderr, "Overriding config file's ServerPort\n");
            break;
        case hands:
            cow_nohands = atoi(val);
            if (cow_nohands <= 0)
                yyerror("Invalid setting for NumHands; using default");
            break;
        case maxcon:
            ka_max_con = atoi(val);
            if (ka_max_con <= 0)
                yyerror("Invalid setting for MaxKeepAliveConn; using default");
            break;
        case maxdur:
            ka_max_dur = atoi(val);
            if (ka_max_dur <= 0)
                yyerror("Invalid setting for MaxKeepAliveDur; using default");
            break;
        case verb:
            verbose = atoi(val);
            break;
        case alog:
            access_log = strdup(val);
            break;
        case elog:
            error_log = strdup(val);
            break;
        case allmime:
            global_mime = strdup(val);
            break;
        case defmime:
            default_mime = strdup(val);
            break;
        case disnagle:
            disable_nagle = atoi(val);
            break;
        case indexfile:
            idx = malloc(sizeof(idxfilename));
            if (idx == NULL) {
                perror("Ran out of memory allocating index file records");
                exit(1);
            }
            idx->name = strdup(val);
            idx->next = NULL;
            if (last_idx == NULL) {
                last_idx = index_files = idx;
            } else {
                last_idx->next = idx;
                last_idx = idx;
            }
            break;
        default:
            yyerror("FIXME: Config file parser needs to be updated!");
    }
    curvar = none;
}


int yywrap(void) {
    return 1;
}


int yyerror(char* msg) {
    fprintf(stderr, "\n**WARNING** Line %d of %s: %s\n", curline, filename, msg);
    return 1;
}
