#include "cow.h"
#include "config.h"
#include "cow_config.h"
#include "utl.h"                // to_upper
#include "request.h"
#include "defines.h"
#include "response.h"

#define MIN(a,b)  (a <= b ? a : b)

/****
 * read_request()
 *
 * Reads an HTTP request from a request socket.  This function will return
 * when it has read at least to the end of the message headers, although
 * it may also read part of the request body (e.g. for POST) as well.
 *
 * @param req A request structure containing the file descriptor to read from
 *            and buffers to read into.
 * @return 0 or 1 on success, -1 on failure
 ****/
int read_request(msg* req) {
    int ret_val;

    /* Initialize connection settings */
    req->ka_status = KA_ACTIVE;         /* by default by http/1.1 */
    req->ka_count = ka_max_con - 1;     /* Limit keep-alive req's remaining */
    req->ka_timeout = ka_max_dur + time(NULL);

    /* Keep-alive loop */
    for (;;) {
        pth_event_t ev_read;
        
        /* Set per-request settings */
        req->simple = 1;                    /* Assume HTTP/0.9 by default */
        req->status = START_LINE;
        req->is_cgi = 0;
        req->p_buf_left = MAX_BUF_SIZE - 1; // off 1 to put '\0'
        req->p_buf_head = req->p_buf;
        req->p_eof = 0;
        req->p_buf_read = 0;
        req->p_entity_body = NULL;

        /* Wait the "keep-alive" timeout even for new connections */
        ev_read = pth_event(PTH_EVENT_TIME,
                            pth_timeout(MIN(cow_rto_s, ka_max_dur), 0));
        ret_val = cow_read_line(req->fdp, req->p_buf, MAX_BUF_SIZE, ev_read);
        pth_event_free(ev_read, PTH_FREE_THIS);

        /* If a socket error/timeout occurs, just close the connection */
        if (ret_val < 0) {
            cow_close_socket(req->fdp);
            --cow_cur_conn;
            return (ret_val);
        }
        //printf("DEBUG: read line '%s' with retval %d, %d remains (%s)\n", req->p_buf, ret_val, req->fdp->incnt, req->fdp->inptr);
        
        /* Parse the first line of the request */
        ret_val = parse_request_line(req);
        if (ret_val < 0) {
            cow_close_socket(req->fdp);
            return ret_val;
        }

        /* Parse all header lines */
        for (;;) {
            ev_read = pth_event(PTH_EVENT_TIME,
                                pth_timeout(cow_rto_s, cow_rto_u));
            int len = cow_read_line(req->fdp, req->p_buf, MAX_BUF_SIZE, ev_read);
            pth_event_free(ev_read, PTH_FREE_THIS);
            if (len < 0) {
                /* Error; close connection */
                cow_close_socket(req->fdp);
                return len;
            } else if (len == 0) {
                /* Blank header line --> end of headers */
                break;
            }
            //printf("DEBUG: read header '%s' with ret val %d, %d remains (%s)\n", req->p_buf, len, req->fdp->incnt, req->fdp->inptr);

            /* Parse the header */
            parse_header(req);
        }

        /* Done reading/parsing headers.  Actually do the request now */
        cow_tot_parsed++;
        switch (req->p_method) {
            case M_HEAD:
            case M_GET:
                ret_val = response_get(req);
                break;
            case M_POST:                // not yet
                STATIC_RESPONSE(req->fdp, 501);
                ret_val = -1;
                break;
#ifdef ENABLE_PUT
            case M_PUT:
                PRINT("method is put....\n");
                process_header_end(req);
                ret_val = response_put(req);
                PRINT("breaking put\n");
                break;
#endif
            default:
                /* Send back a 503 if we get something we don't recognize */
                PRINT("def...\n");
                STATIC_RESPONSE(req->fdp, 503);
                ret_val = -1;
        }
#ifdef USE_TCPCORK
        /* Flush pending output */
        setsockopt(req->fdp, SOL_TCP, TCP_CORK, &zero, sizeof(zero));
#else
        ap_bflush(req->fdp);
#endif

        /* Any type of error condition means keep-alive can't continue. */
        if (ret_val < 0) {
            /* Error */
            cow_close_socket(req->fdp);
            --cow_cur_conn;
            return -1;
        }

        /*
         * If keep-alive is not active or if we've done our maximum number of
         * requests on this connection, close the connection.
         */
        if ((req->ka_status != KA_ACTIVE) || (--req->ka_count == 0)) {
            cow_close_socket(req->fdp);
            --cow_cur_conn;
            return 0;
        }

        /* Loop back and wait for another request on this socket */
    }
}

inline int read_request_ka(msg * req) {
    /* timeout */
    pth_event_t ev_read;

    /* Clear our buffers and prepare for the next request (if any) */
    req->p_buf_left = MAX_BUF_SIZE - 1; // off 1 to put '\0'
    req->p_buf_head = req->p_buf;
    req->p_header_end = NULL;
    req->p_eof = 0;
    req->p_buf_read = 0;
    req->p_entity_body = NULL;

    /*
     * Set the timeout for reading to be the minimum of either the
     * keepalive timeout or the regular socket timeout.
     */
    ev_read = pth_event(PTH_EVENT_TIME,
                        pth_timeout(MIN(cow_rto_s, ka_max_dur), 0));
#ifdef USE_TCPCORK
    /* Try to read another request from the socket; -1 here means timeout */
    req->p_buf_read =
        pth_read_ev(req->fdp, req->p_buf_head, req->p_buf_left, ev_read);
#else
    req->p_buf_read =
        ap_bread(req->fdp, req->p_buf_head, req->p_buf_left, ev_read);
#endif
    pth_event_free(ev_read, PTH_FREE_THIS);

    if (0 == req->p_buf_read) {
        /* 0 is returned iff EOF was read, and it was the only char read */
        req->p_eof = 1;
        return -1;
    } else if (0 > req->p_buf_read) {
        /* read error, timeout or bad connection */
        return -1;
    }

    req->p_buf_left -= req->p_buf_read;
    req->p_buf_head += req->p_buf_read;
    //req->p_buf_head = (char *)(&req->p_buf_head[req->p_buf_read]);
    req->p_buf_head[0] = '\0';

    /*
     * If the end of the data we've read ends with \r\n\r\n or \n\n then we're
     * done reading the request.
     */
    if (strcmp((req->p_buf_head-4), "\r\n\r\n") == 0 ||
        strcmp((req->p_buf_head-4), "\n\n") == 0) {
        req->p_header_end = req->p_buf_head;
        return 1;
    } else {
        /* Still haven't seen the end of the request */
        req->p_header_end = NULL;
        return 0;
    }
}


/****
 * parse_request_line()
 *
 * Parses the first line of an HTTP request.  The request type, URI and
 * protocol version information is all stored in the provided request
 * structure.
 *
 * @param req The HTTP request structure which contains the request buffers
 *            and which the request information should be stored to.
 *
 * @return 0 on success, -1 on failure
 ****/
int parse_request_line(msg* req) {
    /* BNF:
     *
     *    Request-Line   = Method SP Request-URI SP HTTP-Version CRLF
     */

    char *uri, *stop;
    char *logline = req->p_buf;

    /* BNF: Method */
    if (!memcmp(logline, "GET ", 4)) {
        req->p_method = M_GET;
        uri = logline + 3;
    } else if (!memcmp(logline, "PUT ", 4)) {
        req->p_method = M_PUT;
        uri = logline + 3;
    } else if (!memcmp(logline, "HEAD ", 5)) {
        req->p_method = M_HEAD;
        uri = logline + 4;
    } else if (!memcmp(logline, "POST ", 5)) {
        req->p_method = M_POST;
        uri = logline + 4;
    } else {
        // bad request or not implemented
        //printf("DEBUG: Service = '%s'\n", logline);
        STATIC_RESPONSE(req->fdp, 501);
        return -1;
    }

    /* Scan to start of non-whitespace (i.e. the URI) */
    while (*(++uri) == ' ')
        ;

    /* scan to end of URI */
    stop = uri;
    for (;;) {
        /*
         * End of URI will be a space (if we're using HTTP/1.0 or above) or
         * the null-terminator (if we got an HTTP/0.9 simple request).
         */
        if (*stop == '\0' || *stop == ' ')
            break;

        /*
         * \r should mean end of URI and be followed by \n, but since
         * cow_read_line already stripped trailing \r\n's, this must be
         * a stray (and an error).
         */
        if (*stop == '\r') {
            STATIC_RESPONSE(req->fdp, 400);
            return -1;
        }

        /* Not there yet... */
        ++stop;
    }

    /* If the URI is too long, return an error page */
    if (stop - uri > MAX_REQ_URI_LEN) {
        STATIC_RESPONSE(req->fdp, 414);
        return -1;
    }

    /* Record the URI in the request structure */
    req->p_uri_len = stop - uri;
    memcpy(req->p_uri, uri, req->p_uri_len);
    req->p_uri[req->p_uri_len] = '\0';

    /* HTTP-Version */
    if (*stop == ' ') {
        /* if found, we should get an HTTP/x.x */
        int p1, p2;

        if (sscanf(++stop, "HTTP/%d.%d", &p1, &p2) == 2 && p1 >= 1) {
            req->hv_major = p1;
            req->hv_minor = p2;
            req->simple = 0;
            if (p1 == 0 || p1+p2 < 2) {
                req->ka_status = KA_INACTIVE;
            }
        } else {
#ifdef LOGGING
            fprintf(ELOG, "Malformed request '%s %s'\n",
                    req->p_uri, stop);
#endif
            STATIC_RESPONSE(req->fdp, 400);
            return -1;
        }
    } else {
        /* No HTTP/x.y present; use HTTP/0.9 */
        req->simple = 1;
        req->ka_status = KA_INACTIVE;
    }

    return 0;
}

/*
 * Name: process_header_end
 *
 * Description: takes a request and performs some final checking before
 * init_cgi or init_get
 * Returns -1 for error or NPH;  0 for success
 */

int process_header_end(msg * req) {
/* (section 4.3)
Roxen Community: RFC 2616 Hypertext Transfer Protocol -- HTTP/1. (p33 of 172)

          The  presence of a message-body in a request is signaled by the
          inclusion of a Content-Length or Transfer-Encoding header field
          in  the  request's  message-headers. A message-body MUST NOT be
          included  in  a  request  if  the  specification of the request
          method (section 5.1.1) does not allow sending an entity-body in
          requests. 
*/

    if (req->p_method == M_PUT) {
        // should check the presence of "Content-Length" or "Transfre-Encoding",
        // and if it's present, check the size.
        // -> not yet. but we continue...

        // req->p_entity_body points to the poper location (see parse_request)
        // if it's =='\0', no body has read so far.
        // if it's !='\0', body is present, and patially/completely read.

        // for now, just double check in this way
        if (req->p_entity_body == req->p_buf_head) {
            printf("\n--- no body yet! in process_header_end() ---\n");
        }
        else {
            printf
                ("\n--- something there in body! in process_header_end() ---\n");
        }
        return 0;
    }

    return 0;
}

int parse_header(msg * req) {
/*

[70]RFC 2068                        HTTP/1.1                    January 1997


   equivalent to the parameters on a programming language method
   invocation.

          request-header = Accept                   ; Section 14.1
                         | Accept-Charset           ; Section 14.2
                         | Accept-Encoding          ; Section 14.3
                         | Accept-Language          ; Section 14.4
                         | Authorization            ; Section 14.8
                         | From                     ; Section 14.22
                         | Host                     ; Section 14.23
                         | If-Modified-Since        ; Section 14.24
                         | If-Match                 ; Section 14.25
                         | If-None-Match            ; Section 14.26
                         | If-Range                 ; Section 14.27
                         | If-Unmodified-Since      ; Section 14.28
                         | Max-Forwards             ; Section 14.31
                         | Proxy-Authorization      ; Section 14.34
                         | Range                    ; Section 14.36
                         | Referer                  ; Section 14.37
                         | User-Agent               ; Section 14.42

   Request-header field names can be extended reliably only in
   combination with a change in the protocol version. However, new or
   experimental header fields MAY be given the semantics of request-
   header fields if all parties in the communication recognize them to
   be request-header fields.  Unrecognized header fields are treated as
   entity-header fields.

*/

    char c;
    char *value;
    char *line;
    int check_CRLF;

    for (;;) {
        line = req->p_buf;

        /*
         * Look for the ":" between the header name and value.  If we don't
         * find it, we'll just ignore this header and continue with the next
         * one.
         */
        value = strchr(line, ':');
        if (NULL == value)
            return 0;
        
        /*
         * Figure out which header this is; it looks like this code may have
         * been taken from Boa.
         */
        *value++ = '\0';
        to_upper(line);         // header type is case insensitive

        /* Find the start of the value */
        for (;;) {
            int retval;
            c = *value;

            if ('\0' == c) {
                /* Empty header value */
                return 0;
            } else if (c == ' ' || c == '\t') {
                /* Ignore whitespace */
                value++;
            } else {
                break;
            }
        }

        if (!memcmp(line, "HOST", 5)) {
            // TODO

        } else if (!memcmp(line, "IF_MODIFIED_SINCE", 18)
                 && !req->if_modified_since) {
            // TODO
            
        } else if (!memcmp(line, "CONTENT_TYPE", 13) && !req->content_type) {
            // TODO

        } else if (!memcmp(line, "CONTENT_LENGTH", 15) && !req->content_length) {
            // TODO

        } else if (!memcmp(line, "CONNECTION", 11) && 
                    ka_max_con && req->ka_status != KA_STOPPED) {
            if (0 == memcmp(value, "close", 5)) {
                req->ka_status = KA_INACTIVE;
            }
            // TODO: deal with other "connection" headers

        } else if (!memcmp(line, "REFERER", 8)) {
#ifdef LOGGING
            // TODO
#endif

        } else if (!memcmp (line, "ACCEPT", 7)) {
            // TODO: Honor Accept: headers

        } else if (!memcmp(line, "USER_AGENT", 11)) {
#ifdef LOGGING
            req->header_user_agent = value;
#endif
        }
    }
}
